package com.agilex.healthcare.mobilehealthplatform.restservice;

import java.io.IOException;
import java.io.InputStream;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Request;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.ResponseBuilder;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.document.DocumentFilter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.document.DocumentFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.datalayer.document.DocumentService;
import com.agilex.healthcare.mobilehealthplatform.domain.DataIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.Document;
import com.agilex.healthcare.mobilehealthplatform.domain.DocumentBody;
import com.agilex.healthcare.mobilehealthplatform.domain.Documents;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.DocumentLinkBuilder;
import com.agilex.healthcare.utility.StreamHelper;

@Path("patient/{assigning-authority}/{patient-id}/documents")
@Component
@Scope("request")
public class PatientDocumentsResource {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientDocumentsResource.class);

	@GET
	@Produces({ "application/xml", "application/json" })
	public Documents getPatientDocuments(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DocumentFilter filter = DocumentFilterFactory.createFilterFromUri(u.getRequestUri());
		final ScopeFilter scopeFilter = ScopeFilter.getInstanceForLongitudinalScope();

		return fetchPatientDocuments(patientIdentifier, filter, scopeFilter, u);
	}

	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("scope/{scope}")
	public Documents getPatientDocumentsByScope(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("scope") String scope, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DocumentFilter filter = DocumentFilterFactory.createFilterFromUri(u.getRequestUri());
		final ScopeFilter scopeFilter = ScopeFilter.getInstanceForScope(scope);

		return fetchPatientDocuments(patientIdentifier, filter, scopeFilter, u);
	}

	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("system/{system-id}")
	public Documents getPatientDocumentsBySystem(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemIdentifier, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DocumentFilter filter = DocumentFilterFactory.createFilterFromUri(u.getRequestUri());
		final ScopeFilter scopeFilter = ScopeFilter.getInstanceForScope(systemIdentifier);

		return fetchPatientDocuments(patientIdentifier, filter, scopeFilter, u);
	}

	private Documents fetchPatientDocuments(PatientIdentifier patientIdentifier, DocumentFilter filter, ScopeFilter scopeFilter, UriInfo u) {
		DocumentService service = new DocumentService();
		Documents documents = service.fetchDocumentsForPatient(patientIdentifier, filter, scopeFilter);

		DocumentLinkBuilder linkbuilder = new DocumentLinkBuilder(u.getBaseUri());
		linkbuilder.fillLinks(documents, u.getRequestUri());

		return documents;
	}

	@GET
	@Path("system/{system-id}/id/{id}")
	@Produces({ "application/xml", "application/json" })
	public Document getDocument(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemIdentifier, @PathParam("id") String id, @Context UriInfo u,
			@Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DataIdentifier identifier = new DataIdentifier(systemIdentifier, id);

		DocumentService service = new DocumentService();
		Document document = service.fetchSingleDocument(patientIdentifier, identifier);

		DocumentLinkBuilder linkbuilder = new DocumentLinkBuilder(u.getBaseUri());
		linkbuilder.fillLinks(document, u.getRequestUri());

		if (document == null) {
			throw ExceptionFactory.generateNoDataFoundException();
		}
		return document;
	}

	@PUT
	@Path("system/{system-id}/id/{id}")
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	public Document updateDocument(Document document, @PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemIdentifier, @PathParam("id") String id,
			@Context UriInfo u, @Context HttpHeaders headers) {

		document.setPatientIdentifier(assigningAuthority, patientId);
		document.setDataIdentifier(systemIdentifier, id);

		DocumentService service = new DocumentService();
		Document savedDocument = service.updateDocument(document);

		return savedDocument;
	}

	@GET
	@Path("system/{system-id}/id/{id}/content")
	public Response getDocumentBody(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemIdentifier, @PathParam("id") String id, @Context UriInfo u,
			@Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DataIdentifier identifier = new DataIdentifier(systemIdentifier, id);
		logger.debug("retrieve document");

		DocumentService service = new DocumentService();
		Document document = service.fetchSingleDocument(patientIdentifier, identifier);

		if (document == null) {
			throw ExceptionFactory.generateNoDataFoundException();
		}

		DocumentBody body = document.getBody();
		logger.debug("retrieved document");

		byte[] content = body.getContents();
		ResponseBuilder builder = Response.ok(content);
		String mimetype = document.getMimeType();
		builder.type(mimetype);
		builder.header("resource3-header", mimetype);
		return builder.build();
	}

	// todo: is it proper to use /body for PUT (rather than /body/body). Also,
	// should this have any return value or just accept the body?
	@PUT
	@Path("system/{system-id}/id/{id}/content")
	public void saveDocumentContent(InputStream documentContent, @PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemIdentifier,
			@PathParam("id") String documentId, @Context UriInfo u, @Context HttpHeaders headers, @Context Request request) {

		logger.debug("invoked updateDocumentBody");
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DataIdentifier documentIdentifier = new DataIdentifier(systemIdentifier, documentId);

		String contentType = determineContentType(headers);
		logger.debug("save document context");

		byte[] documentContentAsBytes = StreamHelper.streamToBytes(documentContent, true);

		if (documentContent != null) {
			try {
				documentContent.close();
			} catch (IOException e) {
				logger.error("Error closing uploaded document content before attempting to save", e);
			}
		}

		DocumentService service = new DocumentService();
		service.saveDocumentContent(patientIdentifier, documentIdentifier, documentContentAsBytes, contentType);
		logger.debug("saved document");
	}

	private String determineContentType(HttpHeaders headers) {
		MediaType mediaType = headers.getMediaType();
		String mediaTypeAsString = null;
		if (mediaType != null)
			mediaTypeAsString = mediaType.toString();
		return mediaTypeAsString;
	}

	@POST
	@Consumes({ "application/xml", "application/json" })
	public Document createDocument(Document documentMetaData, @PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo u, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

		DocumentService service = new DocumentService();
		ScopeFilter scopeFilter = ScopeFilter.getInstanceForLongitudinalScope();
		Document savedDocument = service.createDocument(documentMetaData, scopeFilter);

		// TODO: put this in the service
		DataIdentifier documentIdentifier = savedDocument.getDataIdentifier();
		byte[] temporaryDocumentContentAsBytes = getTemporaryImageToPreventUserErrors();
		service.saveDocumentContent(patientIdentifier, documentIdentifier, temporaryDocumentContentAsBytes, "image/jpeg");

		DocumentLinkBuilder linkbuilder = new DocumentLinkBuilder(u.getBaseUri());
		linkbuilder.fillLinks(savedDocument, u.getRequestUri());

		logger.debug("created document");

		return savedDocument;
	}

	@GET
	@Path("{domain}/{data-id}")
	@Produces({ "application/xml", "application/json" })
	public Documents getDocumentsAssociatedWithSpecificDataItem(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("domain") String relatedDataDomain,
			@PathParam("data-id") String relatedDataId, @Context UriInfo u, @Context HttpHeaders headers) {
		logger.debug("get document by domain/identifier");

		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

		DocumentService service = new DocumentService();

		// TOOD: this needs to be part of the path, not hardcoded
		String relatedDataIdentifierSystem = "repo";

		DataIdentifier relatedDataIdentifier = new DataIdentifier(relatedDataIdentifierSystem, relatedDataId);
		ScopeFilter scopeFilter = ScopeFilter.getInstanceForLongitudinalScope();
		Documents documents = service.fetchDocumentsForPatientAssociatedWithPatientData(patientIdentifier, relatedDataDomain, relatedDataIdentifier, scopeFilter);

		DocumentLinkBuilder linkbuilder = new DocumentLinkBuilder(u.getBaseUri());
		linkbuilder.fillLinks(documents, u.getRequestUri());

		return documents;
	}

	@POST
	@Path("{domain}/{data-id}")
	@Consumes({ "application/xml", "application/json" })
	public Document createDocumentAssociatedWithSpecificDataItem(Document document, @PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("domain") String relatedDataDomain,
			@PathParam("data-id") String relatedDataId, @Context UriInfo u, @Context HttpHeaders headers) {
		logger.debug("post document by domain/identifier");

		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

		String relatedDataIdentifierSystem = "repo";
		final DataIdentifier relatedDataIdentifier = new DataIdentifier(relatedDataIdentifierSystem, relatedDataId);

		final ScopeFilter scopeFilter = ScopeFilter.getInstanceForLongitudinalScope();

		DocumentService service = new DocumentService();
		document.setPatientIdentifier(patientIdentifier);
		Document savedDocument = service.createDocumentAssociatedWithPatientData(document, relatedDataDomain, relatedDataIdentifier, scopeFilter);

		DocumentLinkBuilder linkbuilder = new DocumentLinkBuilder(u.getBaseUri());
		linkbuilder.fillLinks(savedDocument, u.getRequestUri());

		return savedDocument;
	}

	@DELETE
	@Path("system/{system-id}/id/{id}")
	public void deleteDocument(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemIdentifier, @PathParam("id") String id, @Context UriInfo u,
			@Context HttpHeaders headers) {
		Document document = new Document();
		document.setPatientIdentifier(assigningAuthority, patientId);
		document.setDataIdentifier(systemIdentifier, id);

		DocumentService service = new DocumentService();
		service.deleteDocument(document);
	}

	private byte[] getTemporaryImageToPreventUserErrors() {
		InputStream documentContent = null;
		try {
			documentContent = this.getClass().getClassLoader().getResourceAsStream("no-image.jpeg");
			return StreamHelper.streamToBytes(documentContent, true);
		} finally {
			if (documentContent != null) {
				try {
					documentContent.close();
				} catch (IOException e) {
					logger.error("Error closing document content", e);
				}
			}
		}
	}

}
