package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.ws.rs.*;
import javax.ws.rs.core.*;

import com.agilex.healthcare.mobilehealthplatform.datalayer.document.DocumentService;
import com.agilex.healthcare.mobilehealthplatform.dataservice.PdfDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.pdf.PdfGenerationContext;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dailyevent.DailyEventDataService;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.DailyEventLinkBuilder;

import java.io.IOException;
import java.io.OutputStream;

@Path("/patient/{assigning-authority}/{patient-id}/dailyevents")
@Component
@Scope("request")
public class PatientDailyEventsResource extends AbstractJournalResource<DailyEvents, DailyEvent> {
    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientDailyEventsResource.class);

    public PatientDailyEventsResource() {
		super(new DailyEventDataService(), DailyEvent.class);
	}

	@Override
	protected void fillLinks(DailyEvents datalist, UriInfo u) {
		DailyEventLinkBuilder linkbuilder = new DailyEventLinkBuilder(u.getBaseUri());
		linkbuilder.fillLinks(datalist, u.getRequestUri());
	}

	@Override
	protected void fillLinks(DailyEvent data, UriInfo u) {
		DailyEventLinkBuilder linkbuilder = new DailyEventLinkBuilder(u.getBaseUri());
		linkbuilder.fillLinks(data, u.getRequestUri());
	}

    @GET
    @Path("{data-id}/pdf")
    @Produces("application/pdf")
    public Response getDataItemAndAssociatedDocumentsAsPdf(@PathParam("assigning-authority") String assigningAuthority,
               @PathParam("patient-id") String patientId, @PathParam("data-id") String relatedDataId,
               @Context UriInfo uriInfo, @Context HttpHeaders headers) {

        String relatedDataIdentifierSystem = "repo";
        String systemId = "sed";
        String dataDomain = "dailyevents";

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

        DailyEvent dailyEvent = this.fetchData(assigningAuthority, patientId, systemId, relatedDataId, uriInfo, headers);

        DocumentService service = new DocumentService();
        DataIdentifier relatedDataIdentifier = new DataIdentifier(relatedDataIdentifierSystem, relatedDataId);
        ScopeFilter scopeFilter = ScopeFilter.getInstanceForLongitudinalScope();
        Documents documents = service.fetchDocumentsForPatientAssociatedWithPatientData(patientIdentifier,
            dataDomain, relatedDataIdentifier, scopeFilter);

        PdfGenerationContext context = buildContext(uriInfo, dailyEvent, documents);

        StreamingOutput stream = generatePdfToStreamingOutput(context);
        return Response.ok(stream).build();
    }

    private PdfGenerationContext buildContext(UriInfo uriInfo, DailyEvent dailyEvent, Documents documents) {
        PdfGenerationContext context = new PdfGenerationContext();
        context.setSection("event");
        context.setPatient(MhpUserFactory.createFromSecurityContext().getPatient());
        context.setDailyEvent(dailyEvent);
        context.setAttachments(documents);

        return context;
    }

    private StreamingOutput generatePdfToStreamingOutput(final PdfGenerationContext context) {
        StreamingOutput stream = new StreamingOutput() {
            public void write(OutputStream output) throws IOException, WebApplicationException {
                try {
                    PdfDataService pdfDataService = new PdfDataService();
                    pdfDataService.generateReport(output, context);
                } catch (Exception e) {
                    logger.error("Error generating PDF Report", e);
                    throw new WebApplicationException(Response.Status.INTERNAL_SERVER_ERROR);
                }
            }
        };
        return stream;
    }
}
