package com.agilex.healthcare.mobilehealthplatform.restservice;

import java.io.FileNotFoundException;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.appointment.AppointmentDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.calendar.CalendarDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.calendar.CalendarEventDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.calendar.CalendarKeyDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointment;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointments;
import com.agilex.healthcare.mobilehealthplatform.domain.CalendarDirectory;
import com.agilex.healthcare.mobilehealthplatform.domain.CalendarEvent;
import com.agilex.healthcare.mobilehealthplatform.domain.CalendarEvents;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.utils.QueryParameters;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.CalendarLinkBuilder;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.PatientAppointmentLinkBuilder;
import com.agilex.healthcare.utility.NullChecker;

@Path("/patient/{assigning-authority}/{patient-id}/calendar")
@Component
@Scope("request")
public class PatientCalendarResource {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientCalendarResource.class);

	@GET
	@Path("key")
	@Produces({ "application/xml", "application/json" })
	public CalendarDirectory getKey(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo uriInfo, @Context HttpHeaders headers) throws FileNotFoundException {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

		CalendarKeyDataService calendarKeyDataService = new CalendarKeyDataService();
		String calendarKey = calendarKeyDataService.fetchKeyByPatient(patientIdentifier);

		CalendarDirectory patientCalendarResourceDirectory = null;
		if (NullChecker.isNotNullish(calendarKey)) {
			CalendarLinkBuilder linkBuilder = new CalendarLinkBuilder(uriInfo.getBaseUri());
			patientCalendarResourceDirectory = linkBuilder.createPatientCalendarResourceDirectory(patientIdentifier, calendarKey);
		} else {
			throw new FileNotFoundException();
		}

		return patientCalendarResourceDirectory;
	}

	/**
	 * Provides a mechanism to save calendar entries associated with an event.
	 * 
	 * @param calendarEvent
	 *            A {@link CalendarEvent} body representing an event to save to
	 *            a patient's iCal feed.
	 * @param assigningAuthority
	 *            The organization that the patientId is associated with
	 * @param patientId
	 *            The patientId associated with the Assigning Authority
	 * @return The {@link CalendarEvent} that is now associated with the
	 *         {@link Patient}
	 */
	@POST
	@Path("calendarEvent")
	@Consumes({ "application/xml", "application/json" })
	public CalendarEvent saveCalendarEvent(CalendarEvent calendarEvent, @PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo uriInfo) {

		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		calendarEvent.setPatientIdentifier(patientIdentifier);

		CalendarEventDataService dataService = new CalendarEventDataService();
		CalendarEvent savedCalendarEvent = dataService.saveCalendarEvent(calendarEvent, ScopeFilter.getInstanceForLongitudinalScope());

		return savedCalendarEvent;
	}

	@GET
	@Path("calendarEvent")
	@Produces({ "application/xml", "application/json" })
	public CalendarEvents fetchCalendarEvents(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo uriInfo) {
		CalendarEventDataService dataService = new CalendarEventDataService();
		return dataService.getCalendarEvents(new PatientIdentifier(assigningAuthority, patientId), ScopeFilter.getInstanceForLongitudinalScope());
	}

	@POST
	@Path("key")
	@Produces({ "application/xml", "application/json" })
	public CalendarDirectory createKey(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

		CalendarKeyDataService calendarKeyDataService = new CalendarKeyDataService();
		String calendarKey = calendarKeyDataService.createKeyForPatient(patientIdentifier);
//		logger.debug(String.format("created key for patient [pat=%s][key=%s]", patientIdentifier, calendarKey));

		CalendarLinkBuilder linkBuilder = new CalendarLinkBuilder(uriInfo.getBaseUri());
		CalendarDirectory patientCalendarResourceDirectory = linkBuilder.createPatientCalendarResourceDirectory(patientIdentifier, calendarKey);

		return patientCalendarResourceDirectory;
	}

	@GET
	@Path("events")
	@Produces({ "application/xml", "application/json" })
	public Appointments getCalendarEvents(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

		AppointmentDataService dataservice = new AppointmentDataService();
		DateFilter dateFilter = DateFilterFactory.createCalendarApplicationDateFilter(uriInfo.getRequestUri());
		Appointments appointments = dataservice.getPatientAppointments(patientIdentifier, dateFilter);

		appointments = sort(appointments, uriInfo);

		PatientAppointmentLinkBuilder linkbuilder = new PatientAppointmentLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(appointments, uriInfo.getRequestUri());

		return appointments;
	}

	/**
	 * Returns an iCal formatted calendar feed of a patient's
	 * {@link Appointment} and {@link CalendarEvent} data.
	 * 
	 * @param assigningAuthority
	 *            The organization that the patientId is associated with
	 * @param patientId
	 *            The patientId associated with the Assigning Authority
	 * @return The {@link CalendarEvent} that is now associated with the
	 *         {@link Patient}
	 */
	@GET
	@Path("feed")
	@Produces({ "text/calendar" })
	public String getCalendarFeed(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo uriInfo) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

		CalendarDataService dataservice = new CalendarDataService();
		return dataservice.getAggregatedPatientCalendar(patientIdentifier, DateFilterFactory.createCalendarApplicationDateFilter(uriInfo.getRequestUri()));
	}

	private Appointments sort(Appointments appointments, UriInfo u) {
		AppointmentDataService dataservice = new AppointmentDataService();

		QueryParameters queryParams = new QueryParameters(u.getRequestUri().getQuery());
		String sortRequest = queryParams.getValue("sort");
		if (NullChecker.isNullish(sortRequest)) {
			sortRequest = "date.desc";
		}

		if (sortRequest.contentEquals("date") || sortRequest.contentEquals("date.asc"))
			appointments = dataservice.sortByDate(appointments, false);
		else if (sortRequest.contentEquals("date.desc"))
			appointments = dataservice.sortByDate(appointments, true);

		return appointments;
	}
}
