package com.agilex.healthcare.mobilehealthplatform.restservice;

import java.util.Date;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.assessment.AssessmentDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.AssessmentResultGraphGenerator;
import com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.AssessmentResultsDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.reports.PainConsolidatedReportGeneratorParameter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.paindiary.PainSummaryGraphGenerator;
import com.agilex.healthcare.mobilehealthplatform.datalayer.paingoal.PainGoalDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.Assessment;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentEligibility;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestion;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResult;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResultReport;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResults;
import com.agilex.healthcare.mobilehealthplatform.domain.Assessments;
import com.agilex.healthcare.mobilehealthplatform.domain.GraphData;
import com.agilex.healthcare.mobilehealthplatform.domain.PainDiaryEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.PainGoals;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.ToolTrackingResults;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.tooltrackingresult.ToolTrackingResultDataService;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.AssessmentDraftsLinkBuilder;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.AssessmentEligibilityLinkBuilder;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.AssessmentLinkBuilder;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.AssessmentResultsLinkBuilder;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.PatientAssessmentLinkBuilder;
import com.agilex.healthcare.utility.DateHelper;

@Component
@Scope("request")
@Path("patient/{assigning-authority}/{patient-id}/assessments")
public class PatientAssessmentsResource {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientAssessmentsResource.class);
	private static final ScopeFilter scope = ScopeFilter.getInstanceForLongitudinalScope();
	private final AssessmentResultsDataService dataservice = new AssessmentResultsDataService();

	@GET
	@Produces({ "application/xml", "application/json" })
	public Assessments fetchAssessments(@PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
			@QueryParam("id") String assessmentId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		logger.debug("PatientAssessmentsResource.fetchAssessments() called");
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		AssessmentDataService dataService = new AssessmentDataService();

		Assessments assessments = new Assessments();
		if (assessmentId != null) {
			assessments.add(dataService.fetchAssessment(assessmentId, patientIdentifier));
		} else {
			assessments = dataService.fetchActiveAssessments();
		}

		PatientAssessmentLinkBuilder linkBuilder = new PatientAssessmentLinkBuilder(uriInfo.getBaseUri());
		return linkBuilder.fillLinks(assessments, patientIdentifier, uriInfo.getRequestUri());
	}

	/**
	 * A resource that returns the indicated {@link Assessment}
	 * 
	 * @return Returns a given {@link Assessment}
	 */
	@GET
	@Produces({ "application/json" })
	@Path("id/{assessmentId}")
	public Assessment fetchAssessment(@PathParam("assessmentId") String assessmentId, @PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		logger.debug("PatientAssessmentsResource.fetchAssessment() called");
		AssessmentDataService dataService = new AssessmentDataService();
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		
		Assessment assessment = dataService.fetchAssessment(assessmentId, patientIdentifier);
		
		AssessmentLinkBuilder linkBuilder = new AssessmentLinkBuilder(uriInfo.getBaseUri());
		return linkBuilder.fillLinks(assessment, uriInfo.getRequestUri());
	}

	/**
	 * A resource that returns the indicated {@link AssessmentQuestion}
	 * 
	 * @return Returns a question for a given {@link Assessment}
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("id/{assessment-id}/questions/id/{questionId}")
	public AssessmentQuestion fetchQuestion(@PathParam("assessment-id") String assessmentId, @PathParam("questionId") String questionId,
			@Context UriInfo uriInfo, @Context HttpHeaders headers) {
		logger.debug("PatientAssessmentsResource.fetchQuestion() called");
		AssessmentDataService dataService = new AssessmentDataService();
		AssessmentQuestion assessmentQuestion = dataService.fetchQuestion(assessmentId, questionId);
		AssessmentLinkBuilder linkBuilder = new AssessmentLinkBuilder(uriInfo.getBaseUri());
		return linkBuilder.fillLinks(assessmentQuestion, uriInfo.getRequestUri());
	}

	@GET
	@Path("id/{assessment-id}/results")
	@Produces({ "application/xml", "application/json" })
	public AssessmentResults fetchAssessmentResults(@PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
			@PathParam("assessment-id") String assessmentId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		logger.debug("PatientAssessmentsResource.fetchAssessmentResults() called");

		DateFilter sixMonthFilter = DateFilterFactory.createFilterFromDate(DateHelper.minusMonths(DateHelper.getToday(), 6), DateHelper.getToday());
		AssessmentResults assessmentResults = fetchAssessmentResults(patientId, assigningAuthority, assessmentId, sixMonthFilter, uriInfo, headers);

		AssessmentResultsLinkBuilder linkbuilder = new AssessmentResultsLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(assessmentResults, uriInfo.getBaseUri());
		return assessmentResults;
	}

	//Not currently exposed as REST service
	public AssessmentResults fetchAssessmentResults(@PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
			@PathParam("assessment-id") String assessmentId, Date startDate, Date endDate, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		logger.debug("PatientAssessmentsResource.fetchAssessmentResults() called");

		DateFilter dateFilter = DateFilterFactory.createFilterFromDate(startDate, endDate);
		AssessmentResults assessmentResults = fetchAssessmentResults(patientId, assigningAuthority, assessmentId, dateFilter, uriInfo, headers);

		return assessmentResults;
	}

	public AssessmentResults fetchAssessmentResults(@PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
			@PathParam("assessment-id") String assessmentId, DateFilter dateFilter, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

		AssessmentResults assessmentResults = dataservice.fetchResultsByAssessment(patientIdentifier, assessmentId, dateFilter, scope);
		assessmentResults.sortDescending("dateTaken");

		return assessmentResults;
	}

	@GET
	@Path("id/{assessment-id}/results/graphdata")
	@Produces({ "application/xml", "application/json" })
	public GraphData fetchAssessmentResultsAsGraphData(@PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
			@PathParam("assessment-id") String assessmentId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		
		logger.debug("PatientAssessmentsResource.fetchAssessmentResultsAsGraphData() called");
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		
		DateFilter filter = DateFilterFactory.createFilterFromUri(uriInfo);
		AssessmentResults assessmentResults = dataservice.fetchResultsByAssessment(patientIdentifier, assessmentId, filter, scope);
		assessmentResults.sortDescending("dateTaken");

		AssessmentResultGraphGenerator graphGenerator = new AssessmentResultGraphGenerator();
		GraphData graphData = graphGenerator.createGraphData(assessmentResults);

		AssessmentResultsLinkBuilder linkbuilder = new AssessmentResultsLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(graphData, uriInfo.getBaseUri());

		return graphData;
	}

	/**
	 * A resource for getting the {@link AssessmentEligibility} for this patient and assessmentId
	 */
	@GET
	@Path("id/{assessment-id}/eligibility")
	@Produces({ "application/xml", "application/json" })
	public AssessmentEligibility fetchAssessmentEligibility(@PathParam("patient-id") String patientId,
			@PathParam("assigning-authority") String assigningAuthority, @PathParam("assessment-id") String assessmentId, @Context UriInfo uriInfo,
			@Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		AssessmentEligibility availability = dataservice.fetchAssessmentEligibility(assessmentId, patientIdentifier, scope);
		AssessmentEligibilityLinkBuilder linkbuilder = new AssessmentEligibilityLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(availability, uriInfo.getBaseUri());
		return availability;
	}

	/**
	 * A resource for getting a specific {@link AssessmentResult} for this patient
	 */
	@GET
	@Path("id/{assessment-id}/results/id/{result-id}")
	@Produces({ "application/xml", "application/json" })
	public AssessmentResult fetchAssessmentResultByAssessmentAndResultId(@PathParam("patient-id") String patientId,
			@PathParam("assigning-authority") String assigningAuthority, @PathParam("result-id") String resultId,
			@PathParam("assessment-id") String assessmentId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		AssessmentResult assessmentResult = dataservice.fetchAssessmentResult(patientIdentifier, resultId);
		if (assessmentResult == null) {
			ExceptionFactory.generateNoDataFoundException(uriInfo.getRequestUri(), AssessmentResult.class);
		}

		AssessmentResultsLinkBuilder linkBuilder = new AssessmentResultsLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(assessmentResult, uriInfo.getRequestUri());
		return assessmentResult;
	}

	// duplicative of the results resource above, but not currently discoverable
	@GET
	@Path("results/id/{result-id}")
	@Produces({ "application/xml", "application/json" })
	public AssessmentResult fetchAssessmentResult(@PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
			@PathParam("result-id") String resultId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		AssessmentResult assessmentResult = dataservice.fetchAssessmentResult(patientIdentifier, resultId);
		AssessmentResultsLinkBuilder linkBuilder = new AssessmentResultsLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(assessmentResult, uriInfo.getRequestUri());
		return assessmentResult;
	}

	/**
	 * This resource allows the client to save a new assessment result for this user/patient
	 */
	@POST
	@Path("id/{assessment-id}/results")
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	public AssessmentResult saveAssessmentResultByAssessment(@PathParam("patient-id") String patientId,
			@PathParam("assigning-authority") String assigningAuthority, @PathParam("assessment-id") String assessmentId, AssessmentResult assessmentResult,
			@Context UriInfo uriInfo, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		assessmentResult.setPatientIdentifier(patientIdentifier);

		boolean assessmentIdDoesNotMatch = assessmentResult != null && !assessmentResult.getAssessmentId().contentEquals(assessmentId);
		if (assessmentIdDoesNotMatch) {
			throw ExceptionFactory.generatedBadRequestException();
		}

		assessmentResult.setAssessmentId(assessmentId);
		AssessmentResult assessmentResultWithScore = dataservice.saveResult(assessmentResult, scope);

		AssessmentResultsLinkBuilder linkbuilder = new AssessmentResultsLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(assessmentResultWithScore, uriInfo.getBaseUri());

		logger.debug("saved assessment result for an assessment");
		return assessmentResultWithScore;
	}

	/**
	 * This resource allows the client to save a new assessment result for this user/patient
	 */
	@POST
	@Path("results")
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	public AssessmentResult saveAssessmentResult(@PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
			AssessmentResult assessmentResult, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		assessmentResult.setPatientIdentifier(patientIdentifier);

		AssessmentResult assessmentResultWithScore = dataservice.saveResult(assessmentResult, scope);

		AssessmentResultsLinkBuilder linkbuilder = new AssessmentResultsLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(assessmentResultWithScore, uriInfo.getBaseUri());

		logger.debug("saved assessment result for an assessment");
		return assessmentResultWithScore;
	}

	/**
	 * A resource for getting a specific "in progress" {@link AssessmentResult} for this patient and assessmentId.
	 */
	@GET
	@Path("id/{assessment-id}/draft")
	@Produces({ "application/xml", "application/json" })
	public AssessmentResult fetchAssessmentDraft(@PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
			@PathParam("assessment-id") String assessmentId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		AssessmentResults assessmentResults = dataservice.fetchAssessmentDraftsByPatient(patientIdentifier, scope);
		AssessmentResult assessmentResult = assessmentResults.getResultByAssessmentId(assessmentId);
		
		if (assessmentResult == null) {
			assessmentResult = new AssessmentResult();
			assessmentResult.setUniqueId("placeHolderAssessmentResult");
		} else {
			AssessmentDraftsLinkBuilder linkBuilder = new AssessmentDraftsLinkBuilder(uriInfo.getBaseUri());
			linkBuilder.fillLinks(assessmentResult, uriInfo.getRequestUri());
		}
		
		return assessmentResult;
	}

	
	/**
	 * A resource for getting the {@link AssessmentEligibility} for this patient and assessmentId
	 */
	@GET
	@Path("eligibility/id/{assessment-id}")
	@Produces({ "application/xml", "application/json" })
	public AssessmentEligibility checkAssessmentEligibility(@PathParam(value = "patient-id") String patientId,
			@PathParam(value = "assigning-authority") String assigningAuthority, @PathParam(value = "assessment-id") String assessmentId,
			@Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		AssessmentEligibility eligibility = dataservice.fetchAssessmentEligibility(assessmentId, patientIdentifier, scope);
		AssessmentEligibilityLinkBuilder linkbuilder = new AssessmentEligibilityLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(eligibility, uriInfo.getBaseUri());
		return eligibility;
	}

	@GET
	@Path("/painmanagementgraphdata")
	@Produces({ "application/xml", "application/json" })
	public GraphData getPainManagementSummaryAsGraph(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @QueryParam("startdate") String startDate, @QueryParam("enddate") String endDate, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		Date startDateValue = DateHelper.parseDate(startDate);
		Date endDateValue = DateHelper.parseDate(endDate);
		DateFilter dateFilter = DateFilterFactory.createFilterFromDate(startDateValue, endDateValue);

		PainDiaryEntries painDiaryEntries = fetchPainDiaryEntries(assigningAuthority, patientId, dateFilter, uriInfo, headers);
		
		String assessmentId = "monthlyPain";
		AssessmentResults painAssessmentResults = fetchAssessmentResults(patientId, assigningAuthority, assessmentId, startDateValue, endDateValue, uriInfo, headers); 
		PainSummaryGraphGenerator graphGenerator = new PainSummaryGraphGenerator();
		GraphData graphData = graphGenerator.createGraphData(painDiaryEntries, painAssessmentResults);

		AssessmentResultsLinkBuilder linkbuilder = new AssessmentResultsLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(graphData, uriInfo.getRequestUri());
		
		return graphData;
	}	

	@GET
	@Path("/painmanagementsummaryreport")
	@Produces({ "application/xml", "application/json" })
	public AssessmentResultReport getPainManagementSummaryReport(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @QueryParam("startdate") String startDate, @QueryParam("enddate") String endDate, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
        final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

        Patient patient = MhpUserFactory.createFromSecurityContext().getPatient();

        Date startDateValue = DateHelper.parseDate(startDate);
        Date endDateValue = DateHelper.endOfDay(DateHelper.parseDate(endDate));
        
        DateFilter dateFilter = DateFilterFactory.createFilterFromDate(startDateValue, endDateValue);

        PainDiaryEntries painDiaryEntries = fetchPainDiaryEntries(assigningAuthority, patientId, dateFilter, uriInfo, headers);

        String assessmentId = "monthlyPain";
        AssessmentResults painAssessmentResults = fetchAssessmentResults(patientId, assigningAuthority, assessmentId, dateFilter, uriInfo, headers);

        PainGoals painGoals = fetchPainGoals(patientIdentifier, dateFilter);

        ToolTrackingResults toolTrackingResults = fetchToolTrackingResults(patientIdentifier, dateFilter);

        AssessmentResultsDataService service = new AssessmentResultsDataService();
        AssessmentResultReport report = service.getPainConsolidatedReport(new PainConsolidatedReportGeneratorParameter(patient, startDateValue, endDateValue, painAssessmentResults, painDiaryEntries, painGoals, toolTrackingResults));
        return  report;
	}

    private PainGoals fetchPainGoals(PatientIdentifier patientIdentifier, DateFilter dateFilter) {
        PainGoalDataService dataService = new PainGoalDataService();
        return dataService.fetchAllPainGoals(patientIdentifier, dateFilter);
    }

    private ToolTrackingResults fetchToolTrackingResults(PatientIdentifier patientIdentifier, DateFilter dateFilter){
        ToolTrackingResultDataService service = new ToolTrackingResultDataService();
        return service.fetchResults(patientIdentifier, dateFilter);
    }

	private PainDiaryEntries fetchPainDiaryEntries(String assigningAuthority, String patientId, DateFilter dateFilter, UriInfo u, HttpHeaders headers) {

		PainDiaryResource resource = new PainDiaryResource();
		PainDiaryEntries painEntries = resource.fetchPainDiaryEntries(assigningAuthority, patientId, dateFilter, u, headers);

		return painEntries;
	}
}
