package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.appointment.AppointmentDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointment;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointments;
import com.agilex.healthcare.mobilehealthplatform.domain.DataIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.ProviderNote;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.utils.QueryParameters;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.PatientAppointmentLinkBuilder;
import com.agilex.healthcare.utility.NullChecker;

@Path("/patient/{assigning-authority}/{patient-id}/appointments")
@Component
@Scope("request")
public class PatientAppointmentsResource {

	/**
	 * Deprecated. Please use {@link #fetchAppointmentsByScope(String, String, String, UriInfo, HttpHeaders)}
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Deprecated
	public Appointments fetchAppointments(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(u.getRequestUri());
		Appointments results = fetchAppointments(u, patientIdentifier, dateFilter, ScopeFilter.getInstanceForLongitudinalScope());
                return results;
        }

	/**
	 * Retrieves {@link Appointments} for the configured scope requested.
	 * 
	 * This takes common date parameters {startDate, endDate}
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * @param scope
	 * @return {@link Appointments}
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("scope/{scope}")
	public Appointments fetchAppointmentsByScope(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("scope") String scope, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(u.getRequestUri());
		Appointments results = fetchAppointments(u, patientIdentifier, dateFilter, ScopeFilter.getInstanceForScope(scope));
		return results;

        }

	/**
	 * Retrieves {@link Appointments} for the system provided.
	 * 
	 * This takes common date parameters {startDate, endDate}
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * @param system
	 * @return {@link Appointments}
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("system/{system-id}")
	public Appointments fetchAppointmentsBySystem(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String system, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(u.getRequestUri());
                Appointments results = fetchAppointments(u, patientIdentifier, dateFilter, ScopeFilter.getInstanceForSystem(system));
        
                return results;
        }
	
	/**
	 * 
	 * @return
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("system/{system-id}/id/{appointment-id}/visit/{visit-id}/note")
	public ProviderNote fetchClinicNoteForAppointment(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String system, @PathParam("visit-id") String visitId,@Context UriInfo u, @Context HttpHeaders headers) {
		
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DataIdentifier dataIdentifier = new DataIdentifier(system, visitId);
		ScopeFilter scopeFilter = ScopeFilter.getInstanceForSystem(system);
		
		AppointmentDataService dataService = new AppointmentDataService();
		ProviderNote clinicNote = dataService.fetchClinicNotes(patientIdentifier, dataIdentifier, scopeFilter);
		
		return clinicNote;
	}
	
	/**
	 * Retrieves {@link Appointments} where the stop code is equivalent to a Mental Health Appointment based upon the scope provided. 
	 * 
	 * This takes common date parameters {startDate, endDate}
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * @param scope
	 * @return {@link Appointments}
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("mental-health/scope/{scope}")
	public Appointments getMentalHealthAppointments(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("scope") String scope, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(uriInfo);
		ScopeFilter scopeFilter = ScopeFilter.getInstanceForScope(scope);
		
		AppointmentDataService dataService = new AppointmentDataService();
		Appointments mentalHealthAppointments = dataService.fetchMentalHealthAppointments(patientIdentifier, dateFilter, scopeFilter);
		
		return mentalHealthAppointments;
	}

	/**
	 * Deprecated. Please use {@link #fetchAppointmentsByScope(String, String, String, UriInfo, HttpHeaders)} with date filter query parameters
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("past")
	@Deprecated
	public Appointments fetchPastAppointments(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createPastFilterFromUri(u.getRequestUri());
		Appointments pastAppointments = fetchAppointments(u, patientIdentifier, dateFilter, ScopeFilter.getInstanceForLongitudinalScope());

		Appointments keptPastAppointments = new Appointments();
		for (Appointment pastAppointment : pastAppointments) {
			if ("kept".equalsIgnoreCase(pastAppointment.getStatus())) {
				keptPastAppointments.add(pastAppointment);
			}
		}
                return keptPastAppointments;
	}

	/**
	 * Deprecated. Please use {@link #fetchAppointmentsByScope(String, String, String, UriInfo, HttpHeaders)} with date filter query parameters
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("future")
	@Deprecated
	public Appointments fetchFutureAppointments(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo u, @Context HttpHeaders headers) {
        
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createFutureFilterNext120DaysFromUri(u.getRequestUri());
		Appointments results = fetchAppointments(u, patientIdentifier, dateFilter, ScopeFilter.getInstanceForOperationalScope());
        
                return results;
        }
	
	/**
	 * Retrieves a specific {@link Appointment}. This URI represents the self-link of an Appointment.
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * @param systemId
	 * @param id
	 * @return {@link Appointment}
	 */
	@GET
	@Path("system/{system-id}/id/{id}")
	@Produces({ "application/xml", "application/json" })
	public Appointment fetchAppointment(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("id") String id, @Context UriInfo u,
			@Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DataIdentifier appointmentIdentifier = new DataIdentifier(systemId, id);

		AppointmentDataService dataservice = new AppointmentDataService();
		Appointment appointment = dataservice.getPatientAppointment(patientIdentifier, appointmentIdentifier);

		PatientAppointmentLinkBuilder linkbuilder = new PatientAppointmentLinkBuilder(u.getBaseUri());
		linkbuilder.fillLinks(appointment, u.getRequestUri());

                return appointment;
	}

	private Appointments fetchAppointments(UriInfo u, final PatientIdentifier patientIdentifier, DateFilter dateFilter, ScopeFilter scopeFilter) {
		AppointmentDataService dataservice = new AppointmentDataService();

		if (dateFilter == null)
			dateFilter = DateFilterFactory.createFilterFromUri(u.getRequestUri());

		Appointments appointments = dataservice.getPatientAppointments(patientIdentifier, dateFilter, scopeFilter);
		appointments = sort(appointments, u);

		PatientAppointmentLinkBuilder linkbuilder = new PatientAppointmentLinkBuilder(u.getBaseUri());
		linkbuilder.fillLinks(appointments, u.getRequestUri());

		return appointments;
	}

	// TODO: I would like to move this to the retriever if possible
	private Appointments sort(Appointments appointments, UriInfo u) {
		AppointmentDataService dataservice = new AppointmentDataService();

		QueryParameters queryParams = new QueryParameters(u.getRequestUri().getQuery());
		String sortRequest = queryParams.getValue("sort");
		if (NullChecker.isNullish(sortRequest)) {
			sortRequest = "date.desc";
		}

		if (sortRequest.contentEquals("date") || sortRequest.contentEquals("date.asc"))
			appointments = dataservice.sortByDate(appointments, false);
		else if (sortRequest.contentEquals("date.desc"))
			appointments = dataservice.sortByDate(appointments, true);

		return appointments;
	}

}
