
package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.paingoal.PainGoalDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.paingoal.PainGoalReportGenerator;
import com.agilex.healthcare.mobilehealthplatform.domain.DataIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PainGoal;
import com.agilex.healthcare.mobilehealthplatform.domain.PainGoalEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.PainGoalEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.PainGoalReport;
import com.agilex.healthcare.mobilehealthplatform.domain.PainGoals;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.PainGoalEntryLinkBuilder;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.PainGoalLinkBuilder;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.PainGoalEntryStatus;

@Path("/patient/{assigning-authority}/{patient-id}/pain-goals")
@Component
@Scope("request")
public class PainGoalResource {
	
	@GET
	@Produces({ "application/xml", "application/json" })
	public PainGoals fetchPainGoals(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo u, @Context HttpHeaders headers) {
		
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

		PainGoalDataService dataService = new PainGoalDataService();
		PainGoals painGoals = dataService.fetchPainGoals(patientIdentifier);

		populateNextActionStep(painGoals);

		PainGoalLinkBuilder linkbuilder = new PainGoalLinkBuilder(u.getBaseUri());
		linkbuilder.fillLinks(painGoals, u.getRequestUri());
		
		return painGoals;
	}

	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("system/{system-id}/id/{id}")
	public PainGoal fetchPainGoal(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("id") String id, @Context UriInfo u, @Context HttpHeaders headers) {
		
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DataIdentifier dataIdentifier = new DataIdentifier(systemId, id);
		
		PainGoalDataService dataService = new PainGoalDataService();
		PainGoal painGoal = dataService.fetchPainGoal(patientIdentifier, dataIdentifier);

		populateNextActionStep(painGoal);

		PainGoalLinkBuilder linkbuilder = new PainGoalLinkBuilder(u.getBaseUri());
		linkbuilder.fillLinks(painGoal, u.getRequestUri());
		
		return painGoal;
	}
	
	
	@GET
	@Path("/report")
	@Produces({ "application/xml", "application/json" })
	public PainGoalReport fetchPainGoalReport(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo u, @Context HttpHeaders headers) {
		
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

		PainGoalDataService dataService = new PainGoalDataService();
		PainGoals painGoals = dataService.fetchAllPainGoals(patientIdentifier);

		PainGoalReportGenerator generator = new PainGoalReportGenerator(painGoals);
		PainGoalReport report = generator.generateReport();
		
		return report;
	}
	
	@POST
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	public PainGoal createNewPainGoal(PainGoal painGoal, @PathParam("assigning-authority") String assigningAuthority,
			@PathParam("patient-id") String patientId, @Context UriInfo u, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		painGoal.setPatientIdentifier(patientIdentifier);

		PainGoalDataService dataService = new PainGoalDataService();
		PainGoal createdPainGoal = dataService.createNewPainGoal(patientIdentifier, painGoal);

		if (u != null) {
			PainGoalLinkBuilder linkBuilder = new PainGoalLinkBuilder(u.getBaseUri());
			linkBuilder.fillLinks(createdPainGoal, u.getRequestUri());
		}

		populateNextActionStep(createdPainGoal);
		
		return createdPainGoal;
	}
	
	@PUT
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	@Path("system/{system-id}/id/{id}")
	public PainGoal updatePainGoal(PainGoal painGoal, @PathParam("assigning-authority") String assigningAuthority,
			@PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("id") String id, @Context UriInfo u, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DataIdentifier dataIdentifier = new DataIdentifier(systemId, id);
		painGoal.setPatientIdentifier(patientIdentifier);

		PainGoalDataService dataService = new PainGoalDataService();
		PainGoal updatedPainGoal = dataService.updatePainGoal(patientIdentifier, dataIdentifier, painGoal);

		if (u != null) {
			PainGoalLinkBuilder linkBuilder = new PainGoalLinkBuilder(u.getBaseUri());
			linkBuilder.fillLinks(updatedPainGoal, u.getRequestUri());
		}
		
		populateNextActionStep(updatedPainGoal);
		
		return updatedPainGoal;
	}
	
	@DELETE
	@Path("system/{system-id}/id/{id}")
	public void deletePainGoal(@PathParam("assigning-authority") String assigningAuthority,
			@PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("id") String id, @Context UriInfo u, @Context HttpHeaders headers) {
	
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DataIdentifier dataIdentifier = new DataIdentifier(systemId, id);
		PainGoal painGoal = new PainGoal();
		painGoal.setPatientIdentifier(patientIdentifier);
		painGoal.setDataIdentifier(dataIdentifier);
		
		PainGoalDataService dataService = new PainGoalDataService();
		dataService.deletePainGoal(patientIdentifier, dataIdentifier, painGoal);
	}
	
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("system/{system-id}/id/{id}/pain-goal-entries")
	public PainGoalEntries fetchPainGoalEntries(@PathParam("assigning-authority") String assigningAuthority,
			@PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("id") String goalId, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DataIdentifier dataIdentifier = new DataIdentifier(systemId, goalId);

		PainGoalDataService dataService = new PainGoalDataService();
		PainGoalEntries entries = dataService.fetchAllPainGoalEntries(patientIdentifier, dataIdentifier);
		
		if (u != null) {
			PainGoalEntryLinkBuilder linkBuilder = new PainGoalEntryLinkBuilder(u.getBaseUri(), goalId);
			linkBuilder.fillLinks(entries, u.getRequestUri());
		}
		
		return entries;
	}
	
	@POST
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	@Path("system/{system-id}/id/{id}/pain-goal-entries")
	public PainGoalEntry createNewPainGoalEntry(PainGoalEntry painGoalEntry, @PathParam("assigning-authority") String assigningAuthority,
			@PathParam("patient-id") String patientId,  @PathParam("id") String goalId, @Context UriInfo u, @Context HttpHeaders headers) {
		
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		painGoalEntry.setPatientIdentifier(patientIdentifier);

		PainGoalDataService dataService = new PainGoalDataService();
		PainGoalEntry createdPainGoalEntry = dataService.createNewPainGoalEntry(patientIdentifier, goalId, painGoalEntry);

		if (u != null) {
			PainGoalEntryLinkBuilder linkBuilder = new PainGoalEntryLinkBuilder(u.getBaseUri(), goalId);
			linkBuilder.fillLinks(createdPainGoalEntry, u.getRequestUri());
		}

		return createdPainGoalEntry;
	}
	
	@PUT
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	@Path("system/{system-id}/id/{id}/pain-goal-entries/id/{entry-id}")
	public PainGoalEntry updatePainGoalEntry(PainGoalEntry painGoalEntry, @PathParam("assigning-authority") String assigningAuthority,
			@PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("id") String goalId,  @PathParam("entry-id") String entryId, @Context UriInfo u, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DataIdentifier dataIdentifier = new DataIdentifier(systemId, entryId);
		painGoalEntry.setPatientIdentifier(patientIdentifier);

		PainGoalDataService dataService = new PainGoalDataService();
		PainGoalEntry updatedPainGoalEntry = dataService.updatePainGoalEntry(patientIdentifier, dataIdentifier, goalId, painGoalEntry);

		if (u != null) {
			PainGoalEntryLinkBuilder linkBuilder = new PainGoalEntryLinkBuilder(u.getBaseUri(), goalId);
			linkBuilder.fillLinks(updatedPainGoalEntry, u.getRequestUri());
		}

		return updatedPainGoalEntry;
	}
	
	@DELETE
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	@Path("system/{system-id}/id/{id}/pain-goal-entries/id/{entry-id}")
	public void deletePainGoalEntry(@PathParam("assigning-authority") String assigningAuthority,
			@PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("id") String goalId,  @PathParam("entry-id") String entryId, @Context UriInfo u, @Context HttpHeaders headers) {
		
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DataIdentifier dataIdentifier = new DataIdentifier(systemId, entryId);
		
		PainGoalEntry painGoalEntry = new PainGoalEntry();
		painGoalEntry.setDataIdentifier(dataIdentifier);
		painGoalEntry.setPatientIdentifier(patientIdentifier);
		painGoalEntry.setStatus(PainGoalEntryStatus.PainGoalEntryStatusActive);
		
		PainGoalDataService dataService = new PainGoalDataService();
		dataService.deletePainGoalEntry(patientIdentifier, dataIdentifier, goalId, painGoalEntry);
	}
	
	private void populateNextActionStep(PainGoals painGoals) {
		for (PainGoal goal : painGoals) {
			PainGoalEntry latestEntry = null;

			if (NullChecker.isNotNullish(goal.getPainGoalEntries())) {
				latestEntry = goal.getPainGoalEntries().get(0);

				if (NullChecker.isNotNullish(latestEntry.getComments())) {
					goal.setNextActionStep(latestEntry.getComments());
				} else {
					goal.setNextActionStep("");
				}
			} else {
				goal.setNextActionStep("");
			}
		}
	}
	
	private void populateNextActionStep(PainGoal goal) {

		if (NullChecker.isNotNullish(goal.getPainGoalEntries())) {
			goal.setNextActionStep(getLatestIncompleteAction(goal.getPainGoalEntries()));			
		} else {
			goal.setNextActionStep("");
		}
	}
	
	private String getLatestIncompleteAction(PainGoalEntries entries) {
		String match = "";

		for (PainGoalEntry entry : entries) {
			if (PainGoalEntryStatus.PainGoalEntryStatusActive.toString().equalsIgnoreCase(entry.getStatus().toString())) {
				if (NullChecker.isNotNullish(entry.getComments())) {
					match = entry.getComments();
				}
				break;
			}
		}
		
		return match;
	}
}
