package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.annotation.Resource;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.notification.NotificationDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.Notification;
import com.agilex.healthcare.mobilehealthplatform.domain.NotificationUser;
import com.agilex.healthcare.mobilehealthplatform.domain.Notifications;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.utils.PropertyHelper;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.NotificationUriHelper;

@Component
@Scope("request")
@Path("/notification-service/user/id/{assigning-authority}/{user-id}/notifications")
public class NotificationResource {

	@Resource
	private PropertyHelper propertyHelper;

	/**
	 * A resource that returns all {@link Notifications} that have been sent to this user.
	 * Each {@link Notification} includes the date and time it was sent, and the body of the message.
	 * 
	 * @param userId
	 * @return Returns {@link Notifications} associated with the specified {@link NotificationUser}
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	public Notifications fetchNotifications(@PathParam("user-id") String userId, @Context UriInfo uriInfo) {
		NotificationDataService dataservice = getNotificationDataService();
		DateFilter emptyFilter = DateFilterFactory.createEmptyFilter();
		
		Notifications notifications = dataservice.fetchAllNotifications(userId, emptyFilter);
		NotificationUriHelper.updateNotificationsLinks(notifications, uriInfo);
		return notifications;
	}

	private NotificationDataService getNotificationDataService() {
		NotificationDataService dataservice = new NotificationDataService(propertyHelper);
		return dataservice;
	}

	/**
	 * A resource that returns {@link Notifications}, filtered by today's date, that have been sent to this user.
	 * Each {@link Notification} includes the date and time it was sent, and the body of the message.
	 * 
	 * @param userId
	 * @return Returns {@link Notifications} associated with the specified {@link NotificationUser}
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("/today")
	public Notifications fetchTodaysNotifications(@PathParam("user-id") String userId, @Context UriInfo uriInfo) {
		NotificationDataService dataservice = getNotificationDataService();
		DateFilter todayFilter = DateFilterFactory.createFilterFromToday();
		
		Notifications notifications = dataservice.fetchAllNotifications(userId, todayFilter);
		NotificationUriHelper.updateNotificationsLinks(notifications, uriInfo);
		return notifications;
	}

	/***
	 * A resource that returns a single specified {@Notification} from the list of all notifications associated with this user.
	 * The {@link Notification} includes the date and time that it was sent, and the body of the message.
	 * 
	 * @param notificationId
	 * @param userId
	 * @return The requested {@link Notification} sent to the user.
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("/id/{notification-id}")
	public Notification fetchNotification(@PathParam("notification-id") String notificationId, @PathParam("user-id") String userId,
			@Context UriInfo uriInfo) {
		NotificationDataService dataservice = getNotificationDataService();
		Notification notification = dataservice.fetchNotification(userId, notificationId);
		
		NotificationUriHelper.updateNotificationLinks(notification, uriInfo);
		return notification;
	}

	/***
	 * A resource that allows a client to save a {@link Notification} for a specific user.
	 * 
	 * @param notification
	 * @param userId
	 * @return The {@link Notification} that was saved.
	 */
	@POST
	@Consumes({ "application/xml", "application/json" })
	public Notification saveUserSpecificNotification(Notification notification, @PathParam("user-id") String userId, @Context UriInfo uriInfo) {
		NotificationDataService dataservice = getNotificationDataService();
		notification.setUserId(userId);
		return dataservice.saveNotification(notification);
	}

	/***
	 * A resource that allows a client to remove a {@link Notification} for a specific user
	 * 
	 * @param notificationId
	 * @param userId
	 */
	@DELETE
	@Path("/id/{notification-id}")	
	public void deleteNotification(@PathParam("notification-id") String notificationId, @PathParam("user-id") String userId, @Context UriInfo uriInfo) {
		if (notificationId == null || userId == null) {
			throw new WebApplicationException(Status.BAD_REQUEST);
		}
		
		NotificationDataService dataservice = getNotificationDataService();
		dataservice.deleteNotification(userId, notificationId);
	}
}
