package com.agilex.healthcare.mobilehealthplatform.restservice;

import java.net.URI;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.UriBuilder;
import javax.ws.rs.core.UriInfo;

import com.agilex.healthcare.mobilehealthplatform.datalayer.audit.AuditLogDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.audit.AuditLogFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.AuditLog;
import com.agilex.healthcare.mobilehealthplatform.domain.AuditLogs;

@Path("/notification-service/audits/auditLogs")
public class NotificationAuditLogResource {

	/**
	 * Returns all {@link AuditLogs} associated with the provided user-id, filtering upon the following URL 
	 * parameters:
	 * 	type: If provided, only {@link AuditLogs} matching the specified type will be returned (i.e. "Notification")
	 * 	subtype: If provided, only auditLogs that match the user-id and subType will be returned (i.e "Notification {action}")
	 * 	startDate/endDate: If provided, all AuditLogs for the specified user-id will be returned for the date range provided
	 * 	date: If provided, only {@link AuditLogs} saved on the date provided will be returned
	 * 
	 * @param userId
	 * @param uriInfo
	 * @return
	 */
	@GET
	@Path("user/id/{assigning-authority}/{user-id}")
	@Produces({ "application/xml", "application/json" })
	public AuditLogs fetchAuditLogs(@PathParam("user-id") String userId, @Context UriInfo uriInfo) {
		
		AuditLogFilter emptyFilter = new AuditLogFilter();		
		AuditLogDataService dataservice = new AuditLogDataService();
		AuditLogs retrievedAuditLogs = dataservice.fetchAuditLogs(userId, emptyFilter);
		
		updateAuditLogsLinks(retrievedAuditLogs, uriInfo);
		
		return retrievedAuditLogs;
	}
	
	/**
	 * Returns a specific {@link AuditLog} assocaited with the user-id and audit-log-id provided. Filter parameters are not 
	 * available for this resource.
	 * 
	 * @param userId
	 * @param uriInfo
	 * @return
	 */
	@GET
	@Path("user/id/{assigning-authority}/{user-id}/auditLog/{audit-log-id}")
	@Produces({ "application/xml", "application/json" })
	public AuditLog fetchAuditLog(@PathParam("user-id") String userId, @PathParam("audit-log-id") String auditLogId, @Context UriInfo uriInfo) {
		
		AuditLogDataService dataservice = new AuditLogDataService();
		AuditLog retrievedAuditLog = dataservice.fetchAuditLog(userId, auditLogId);
		
		updateAuditLogLinks(retrievedAuditLog, uriInfo);
		
		return retrievedAuditLog;
	}

	private void updateAuditLogsLinks(AuditLogs retrievedAuditLogs, UriInfo uriInfo) {
		if(retrievedAuditLogs != null && uriInfo != null) {
			for (AuditLog auditLog : retrievedAuditLogs) {
				updateAuditLogLinks(auditLog, uriInfo);
			}
		}
	}

	private void updateAuditLogLinks(AuditLog auditLog, UriInfo uriInfo) {
		if(auditLog != null && uriInfo != null) {
			String userId = auditLog.getUserId();
			URI selfUri = UriBuilder.fromUri(uriInfo.getBaseUri()).path("notification-service").path("audits").path("auditLogs").path("user").path("id").path("EDIPI").path(userId).path("auditLog").path(auditLog.getId()).build();
			auditLog.setSelfUri(selfUri);
		}
	}
}
