package com.agilex.healthcare.mobilehealthplatform.restservice;

import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.util.Date;

import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.StreamingOutput;
import javax.ws.rs.core.UriBuilder;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mbb.roa.ROAImageGenerator;
import com.agilex.healthcare.mobilehealthplatform.datalayer.rightofaccess.RightOfAccessDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.staffuserdisclaimer.StaffUserDisclaimerDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.AtomLink;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.domain.RightOfAccessInfo;
import com.agilex.healthcare.mobilehealthplatform.domain.StaffUserDisclaimerContent;
import com.agilex.healthcare.mobilehealthplatform.domain.StaffUserDisclaimerInfo;
import com.agilex.healthcare.mobilehealthplatform.domain.atom.rel;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.UriHelper;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.PatientLinkBuilder;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.google.common.base.Joiner;
import com.itextpdf.text.pdf.AcroFields;
import com.itextpdf.text.pdf.PdfReader;
import com.itextpdf.text.pdf.PdfStamper;
import com.sun.jersey.api.NotFoundException;

@Path("/mhpuser")
@Component
@Scope("request")
public class MhpUserResource {
  private static final String DOB_FIELD = "vha101349[0].#subform[0].DateTimeField1[0]";
  private static final String NAME_FIELD = "vha101349[0].#subform[0].TextField1[0]";
  private static final String DATE_FIELD = "vha101349[0].#subform[0].TextField2[0]";
  private static final String RESOURCES_LOCATION = "com/agilex/healthcare/mobilehealthplatform/pdf/";

  @GET
  @Produces({ "application/xml", "application/json" })
  public MhpUser getMhpUserWithRightOfAccessFlag(@Context UriInfo uriInfo) {
    MhpUser mhpUser = MhpUserFactory.createFromSecurityContext();
    MhpUser mergedUser = mergeWithRightOfAccessInfo(mhpUser);

    setUpLinks(uriInfo, mergedUser);

    return mergedUser;
  }

  @GET
  @Path("staff-user-disclaimer")
  @Produces({ "application/xml", "application/json" })
  public StaffUserDisclaimerInfo getStaffUserDisclaimerInfo(@Context UriInfo uriInfo) {
    MhpUser mhpUser = MhpUserFactory.createFromSecurityContext();
    String vistaLocation = mhpUser.getVistaLocation();
    
    StaffUserDisclaimerInfo staffUserDisclaimerInfo = null;
    if (NullChecker.isNotNullish(vistaLocation)) {
      String userId = mhpUser.getId();
      
      staffUserDisclaimerInfo = fetchStaffUserDisclaimerInfo(vistaLocation, userId);
    }
    
    if (staffUserDisclaimerInfo == null){
      throw new WebApplicationException(Status.NO_CONTENT);
    }
    
    return staffUserDisclaimerInfo;
  }

  @GET
  @Path("staff-user-disclaimer/content")
  @Produces({ "application/xml", "application/json" })
  public StaffUserDisclaimerContent getStaffUserDisclaimerContent(@Context UriInfo uriInfo) {
    StaffUserDisclaimerDataService dataService = new StaffUserDisclaimerDataService();
    StaffUserDisclaimerContent content = dataService.getLatestStaffUserDisclaimerContent();
    
    return content;
  }
  
  @GET
  @Path("right-of-access")
  @Produces({ "application/xml", "application/json" })
  public RightOfAccessInfo getRightOfAccessInfo(@Context UriInfo uriInfo) {
    String patientId = MhpUserFactory.createFromSecurityContext().getPatientId();
    RightOfAccessInfo rightOfAccessInfo = fetchRightOfAccessInfo(patientId);
    
    if(rightOfAccessInfo == null){
      throw new NotFoundException("Right of Access Not Found");
    }
    
    return rightOfAccessInfo;
  }
  
  
  @GET
  @Path("right-of-access")
  @Produces("application/pdf")
  public Response getRightOfAccessAsPDF(@Context UriInfo uriInfo) {
    MhpUser mhpUser = MhpUserFactory.createFromSecurityContext();
    RightOfAccessInfo rightOfAccessInfo = fetchRightOfAccessInfo(mhpUser.getPatientId());
    StreamingOutput stream = getRightOfAccesPDFAsStream(mhpUser, rightOfAccessInfo);
    return Response.ok(stream).build();
  }

  @GET
  @Path("right-of-access/pdf")
  @Produces("application/pdf")
  public Response getRightOfAccessAsPDFResource(@Context UriInfo uriInfo) {
    MhpUser mhpUser = MhpUserFactory.createFromSecurityContext();
    RightOfAccessInfo rightOfAccessInfo = fetchRightOfAccessInfo(mhpUser.getPatientId());
    StreamingOutput stream = getRightOfAccesPDFAsStream(mhpUser, rightOfAccessInfo);
    return Response.ok(stream).build();
  }

  @GET
  @Path("right-of-access/image")
  @Produces("image/png")
  public Response getNewRightOfAccessAsImage(@Context UriInfo uriInfo, @Context HttpServletResponse response) throws Exception{
    return getROAasImage();
  }


  @POST
  @Path("right-of-access")
  @Consumes({ "application/xml", "application/json" })
  @Produces({ "application/xml", "application/json" })
  public RightOfAccessInfo postRightOfAccessInfo(RightOfAccessInfo rightOfAccessInfo, @Context UriInfo uriInfo) {
    return saveRightOfAccessInfo(rightOfAccessInfo);
  }

  @PUT
  @Path("right-of-access")
  @Consumes({ "application/xml", "application/json" })
  @Produces({ "application/xml", "application/json" })
  public RightOfAccessInfo putRightOfAccessInfo(RightOfAccessInfo rightOfAccessInfo, @Context UriInfo uriInfo) {
    return saveRightOfAccessInfo(rightOfAccessInfo);
  }
  
  private RightOfAccessInfo saveRightOfAccessInfo(RightOfAccessInfo rightOfAccessInfo) {
    MhpUser mhpUser = MhpUserFactory.createFromSecurityContext();
    
    rightOfAccessInfo.setRightOfAccessDate(new Date());
    rightOfAccessInfo.setUserId(mhpUser.getPatientId());
    
    if (rightOfAccessInfo.isRightOfAccessAccepted()){
      ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
      updateRightOfAccessFormWithUserInformation(mhpUser, rightOfAccessInfo, outputStream);
      rightOfAccessInfo.setRightOfAccessPDF(outputStream.toByteArray());
      closeOutputStream(outputStream);
    }
    
    RightOfAccessDataService dataService = new RightOfAccessDataService();
    return dataService.saveRightOfAccessInfo(rightOfAccessInfo);
  }

  private void closeOutputStream(ByteArrayOutputStream outputStream) {
    try {
      outputStream.close();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }
  
  @POST
  @Path("staff-user-disclaimer")
  @Consumes({ "application/xml", "application/json" })
  @Produces({ "application/xml", "application/json" })
  public StaffUserDisclaimerInfo postStaffUserDisclaimerInfo(StaffUserDisclaimerInfo staffUserDisclaimerInfo, @Context UriInfo uriInfo) {
    return saveStaffUserDisclaimerInfo(staffUserDisclaimerInfo);
  }
  
  @PUT
  @Path("staff-user-disclaimer")
  @Consumes({ "application/xml", "application/json" })
  @Produces({ "application/xml", "application/json" })
  public StaffUserDisclaimerInfo putStaffUserDisclaimerInfo(StaffUserDisclaimerInfo staffUserDisclaimerInfo, @Context UriInfo uriInfo) {
    return saveStaffUserDisclaimerInfo(staffUserDisclaimerInfo);
  }
  
  private StaffUserDisclaimerInfo saveStaffUserDisclaimerInfo(StaffUserDisclaimerInfo staffUserDisclaimerInfo) {
    MhpUser mhpUser = MhpUserFactory.createFromSecurityContext();
    
    staffUserDisclaimerInfo.setStaffUserDisclaimerDate(new Date());
    staffUserDisclaimerInfo.setVistaLocation(mhpUser.getVistaLocation());
    staffUserDisclaimerInfo.setUserId(mhpUser.getId());
    
    StaffUserDisclaimerDataService dataService = new StaffUserDisclaimerDataService();
    return dataService.saveStaffUserDisclaimerInfo(staffUserDisclaimerInfo);
  }

  private void setUpLinks(UriInfo uriInfo, MhpUser mhpUser) {
        boolean includePatientLinks = !MhpUserFactory.doesUserHasStaffRole();

    mhpUser.setDisplayName(calculateDisplayName(mhpUser.getLastName(), mhpUser.getFirstName()));
        mhpUser.getPatient().setDisplayName(calculateDisplayName(mhpUser.getPatientLastName(), mhpUser.getPatientFirstName()));

        if (includePatientLinks)
            mhpUser.setPatientUri(createPatientLink(mhpUser, uriInfo));

        addRelatedLink(mhpUser, LinkTitles.StaffUserDisclaimer, UriHelper.mergeUri(uriInfo.getBaseUri(), "mhpuser/staff-user-disclaimer"));
        addRelatedLink(mhpUser, LinkTitles.StaffUserDisclaimerContent, UriHelper.mergeUri(uriInfo.getBaseUri(), "mhpuser/staff-user-disclaimer/content"));
        addRelatedLink(mhpUser, LinkTitles.RightOfAccess, UriHelper.mergeUri(uriInfo.getBaseUri(), "mhpuser/right-of-access"));
        addRelatedLink(mhpUser, LinkTitles.RightOfAccessPDF, UriHelper.mergeUri(uriInfo.getBaseUri(), "mhpuser/right-of-access/pdf"));
        addRelatedLink(mhpUser, LinkTitles.RightOfAccessImage, UriHelper.mergeUri(uriInfo.getBaseUri(), "mhpuser/right-of-access/image"));
        addRelatedLink(mhpUser, LinkTitles.MbbReport, UriHelper.mergeUri(uriInfo.getBaseUri(), "report"));
        addRelatedLink(mhpUser, LinkTitles.PatientContextLink, UriHelper.mergeUri(uriInfo.getBaseUri(), "report"));
        addRelatedLink(mhpUser, LinkTitles.StaffDirectory, UriHelper.mergeUri(uriInfo.getBaseUri(), "staff-directory"));
    
        if (includePatientLinks) {
          final String PATIENT_STRING = "patient";
          String assigningAuthority = mhpUser.getUserIdentifier().getAssigningAuthority();
            addRelatedLink(mhpUser, LinkTitles.SecureMessageUser, UriBuilder.fromUri(uriInfo.getBaseUri()).path("securemessage-service").path(assigningAuthority).path(mhpUser.getPatientId()).path("secure-message-user").build());
            addRelatedLink(mhpUser, LinkTitles.Appointments, UriBuilder.fromUri(uriInfo.getBaseUri()).path("appointment-service").path(PATIENT_STRING).path(assigningAuthority).path(mhpUser.getPatientId()).path("appointments").build());
            addRelatedLink(mhpUser, LinkTitles.PatientInternal, UriBuilder.fromUri(uriInfo.getBaseUri()).path(PATIENT_STRING).path(assigningAuthority).path(mhpUser.getPatientId()).build());
            addRelatedLink(mhpUser, LinkTitles.PrimaryCareProviders, UriBuilder.fromUri(uriInfo.getBaseUri()).path("provider-service").path(PATIENT_STRING).path(assigningAuthority).path(mhpUser.getPatientId()).path("providers").path("primarycare").build());
            addRelatedLink(mhpUser, LinkTitles.MentalHealthProviders, UriBuilder.fromUri(uriInfo.getBaseUri()).path("provider-service").path(PATIENT_STRING).path(assigningAuthority).path(mhpUser.getPatientId()).path("providers").path("mentalhealth").build());
            addRelatedLink(mhpUser, LinkTitles.MostRecentFacilities, UriBuilder.fromUri(uriInfo.getBaseUri()).path("facility-service").path(PATIENT_STRING).path(assigningAuthority).path(mhpUser.getPatientId()).path("facilities").build());
        }
        
        addPatientContextLink(mhpUser);
    }

  private void addPatientContextLink(MhpUser mhpUser) {
    AtomLink link = new AtomLink();
    link.setHref(UriBuilder.fromUri("http://localhost:8080/PatientContext/rest/context/" + mhpUser.getId()).build());
    link.setTitle(LinkTitles.PatientContextLink);
    mhpUser.getLink().setLinkByTitle(link);
  }

    private URI createPatientLink(MhpUser mhpUser, UriInfo uriInfo) {
    PatientLinkBuilder linkBuilder = new PatientLinkBuilder(uriInfo.getBaseUri());
    return linkBuilder.createPatientSelfUri(mhpUser.getPatient().getPatientIdentifier());
    }

    private void addRelatedLink(MhpUser user, String title, URI href) {
        AtomLink link = new AtomLink();
        link.setHref(href);
        link.setRel(rel.related);
        link.setTitle(title);

        user.getLink().setLinkByTitle(link);
    }

  public MhpUser mergeWithRightOfAccessInfo(MhpUser mhpUser) {
    if(MhpUserFactory.doesUserHasStaffRole()){
      mhpUser.setRightOfAccessAccepted(true);
    }else{
      RightOfAccessDataService dataService = new RightOfAccessDataService();
      mhpUser.setRightOfAccessAccepted(dataService.hasAcceptedRightOfAccess(mhpUser.getPatientId()));
    }
    
    return mhpUser;
  }
  
  private void updateRightOfAccessFormWithUserInformation(MhpUser mhpUser, RightOfAccessInfo rightOfAccessInfo, OutputStream outputStream){

    try {
      PdfReader reader = new PdfReader(RESOURCES_LOCATION + "ROA.pdf");
      PdfStamper stamper = new PdfStamper(reader, outputStream);
      
      AcroFields acroFields = stamper.getAcroFields();
      
      acroFields.setField(NAME_FIELD, getPatientDisplayName(mhpUser));
      acroFields.setField(DOB_FIELD, mhpUser.getPatientDateOfBirth());

      if (rightOfAccessInfo != null){
        acroFields.setField(DATE_FIELD, DateHelper.format(rightOfAccessInfo.getRightOfAccessDate(), "MM/dd/yyyy"));
      }else{
        acroFields.setField(DATE_FIELD, DateHelper.format(new Date(), "MM/dd/yyyy"));
      }
      
      stamper.setFormFlattening(true);
      
      stamper.close();
    } catch (Exception e) {
      throw new RuntimeException("Unable to update Right of Access Information", e);
    }
  }
  
  private String getPatientDisplayName(MhpUser user){
        return addMiddleName(getDisplayName(user.getPatientLastName(), user.getPatientFirstName()), user.getPatientMiddleName());
  }

    private String calculateDisplayName(String firstName, String lastName) {
        String displayName = getDisplayName(lastName, firstName);
        return "".equals(displayName) ? "?" : displayName;
    }

    private String getDisplayName(String firstName, String lastName) {
        Joiner joiner = Joiner.on(", ").skipNulls();
        return joiner.join(lastName, firstName);
    }

    private String addMiddleName(String name, String middleName) {
        Joiner joiner = Joiner.on(" ").skipNulls();
        return joiner.join(name, middleName);
    }
  private StreamingOutput getRightOfAccesPDFAsStream(final MhpUser mhpUser, final RightOfAccessInfo rightOfAccessInfo){
    
    StreamingOutput stream = new StreamingOutput() {
      public void write(OutputStream output) throws IOException, WebApplicationException {
        updateRightOfAccessFormWithUserInformation(mhpUser, rightOfAccessInfo, output);
      }
    };
    return stream;
  }

  private StaffUserDisclaimerInfo fetchStaffUserDisclaimerInfo(String vistaLocation, String userId){
    StaffUserDisclaimerInfo staffUserDisclaimerInfo = null;

    if(MhpUserFactory.doesUserHasStaffRole()){
      StaffUserDisclaimerDataService dataService = new StaffUserDisclaimerDataService();
      staffUserDisclaimerInfo = dataService.getStaffUserDisclaimerInfo(vistaLocation, userId);
    } else {
      staffUserDisclaimerInfo = new StaffUserDisclaimerInfo();
      staffUserDisclaimerInfo.setVistaLocation(vistaLocation);
      staffUserDisclaimerInfo.setUserId(userId);
      staffUserDisclaimerInfo.setStaffUserDisclaimerAccepted(true);
      staffUserDisclaimerInfo.setStaffUserDisclaimerDate(new Date());
    }
    
    return staffUserDisclaimerInfo;
  }
  
  private RightOfAccessInfo fetchRightOfAccessInfo(String patientId){
    RightOfAccessInfo rightOfAccessInfo = null;

    if(MhpUserFactory.doesUserHasStaffRole()){
      rightOfAccessInfo = new RightOfAccessInfo();
      rightOfAccessInfo.setUserId(patientId);
      rightOfAccessInfo.setRightOfAccessAccepted(true);
      rightOfAccessInfo.setRightOfAccessDate(new Date());
    }else{
      RightOfAccessDataService dataService = new RightOfAccessDataService();
      rightOfAccessInfo = dataService.getRightOfAccessInfo(patientId);
    }
    
    return rightOfAccessInfo;
  }
  
  private Response getROAasImage() throws IOException {
    MhpUser mhpUser = MhpUserFactory.createFromSecurityContext();
    
    ROAImageGenerator imageGenerator = new ROAImageGenerator();

    BufferedImage imageData = imageGenerator.generate(mhpUser); 
    return Response.ok(imageData, "image/png").build();
  }
}