package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.learn.LearnDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.LearnArticle;
import com.agilex.healthcare.mobilehealthplatform.domain.LearnCategories;
import com.agilex.healthcare.mobilehealthplatform.domain.LearnCategory;
import com.agilex.healthcare.mobilehealthplatform.domain.LearnCondition;
import com.agilex.healthcare.mobilehealthplatform.domain.LearnConditions;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.LearnArticleLinkBuilder;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.LearnCategoryLinkBuilder;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.LearnConditionLinkBuilder;

@Component
@Scope("request")
@Path("public/learn")
public class LearnResource {
	LearnDataService dataservice = new LearnDataService();

	/***
	 * This is the list of conditions supported by the "Learn" subsystem. Conditions refer to the condition that the patient has, such as PTSD or Strain.
	 * @return List of conditions
	 */
	@GET
	@Path("conditions")
	@Produces({ "application/xml", "application/json" })
	public LearnConditions fetchConditions(@Context UriInfo uriInfo, @Context HttpHeaders headers) {
		LearnConditions conditions = dataservice.fetchLearnConditions();

		LearnConditionLinkBuilder linkBuilder = new LearnConditionLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(conditions, uriInfo.getRequestUri());

		return conditions;
	}

	/***
	 * This is the list of categories associated with a given condition
	 * @param condition The condition with which these categories are associated
	 * @return List of categories for this condition
	 */
	@GET
	@Path("conditions/id/{condition-id}/categories")
	@Produces({ "application/xml", "application/json" })
	public LearnCategories fetchCategoriesByCondition(@PathParam("condition-id") String conditionId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		LearnCondition condition = new LearnCondition();
		condition.setId(conditionId);
		LearnCategories categories = dataservice.fetchLearnCategories(condition);
		
		LearnCategoryLinkBuilder linkBuilder = new LearnCategoryLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(categories, uriInfo.getRequestUri());

		return categories;
		
	}

	/***
	 * This is a specific learn category, including its associated sub-categories and pointers to the articles that it contains
	 * @param category-id The id of the category being requested
	 * @return Learn category as requested by id; includes associated sub-categories and pointers to articles
	 */
	@GET
	@Path("categories/id/{category-id}")
	@Produces({ "application/xml", "application/json" })
	public LearnCategory fetchLearnCategory(@PathParam("category-id") String categoryId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		LearnCategory category = dataservice.fetchLearnCategory(categoryId);
		
		LearnCategoryLinkBuilder linkBuilder = new LearnCategoryLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(category, uriInfo.getRequestUri());
		
		return category;
	}

	/***
	 * This is a specific article and its content
	 * @param article-id The id of the article being requested
	 * @return Article and its content
	 */
	@GET
	@Path("articles/id/{article-id}")
	@Produces({ "application/xml", "application/json" })
	public LearnArticle fetchLearnArticle(@PathParam("article-id") String articleId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		LearnArticle article = dataservice.fetchLearnArticle(articleId);

		LearnArticleLinkBuilder linkBuilder = new LearnArticleLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(article, uriInfo.getRequestUri());
		
		return article;
	}
}
