package com.agilex.healthcare.mobilehealthplatform.restservice;

import java.io.*;
import java.net.URISyntaxException;
import java.util.*;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.StreamingOutput;
import javax.ws.rs.core.UriInfo;

import com.agilex.healthcare.mobilehealthplatform.domain.*;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mbb.datalayer.PdfGenerationContext;
import com.agilex.healthcare.mbb.dataservice.PdfDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.healthadvocaterequest.HealthAdvocateRequestDataService;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.HealthAdvocateRequestEligibilityLinkBuilder;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.HealthAdvocateRequestLinkBuilder;

@Path("/patient/{assigning-authority}/{patient-id}/health-advocate-requests")
@Component
@Scope("request")
public class HealthAdvocateRequestsResource {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(HealthAdvocateRequestsResource.class);

	@GET
	@Produces({ "application/xml", "application/json" })
	public HealthAdvocateRequests getHealthAdvocateRequests(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo u,
			@Context HttpHeaders headers) {

		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		HealthAdvocateRequestDataService dataService = new HealthAdvocateRequestDataService();

		HealthAdvocateRequests healthAdvocateRequests = dataService.fetchHealthAdvocateRequests(patientIdentifier);

		if (u != null) {
			HealthAdvocateRequestLinkBuilder linkBuilder = new HealthAdvocateRequestLinkBuilder(u.getBaseUri());
			linkBuilder.fillLinks(healthAdvocateRequests, u.getRequestUri());
		}

		return healthAdvocateRequests;
	}

	@POST
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	public HealthAdvocateRequest createNewHealthAdvocateRequest(HealthAdvocateRequest healthAdvocateRequest, @PathParam("assigning-authority") String assigningAuthority,
			@PathParam("patient-id") String patientId, @Context UriInfo u, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		healthAdvocateRequest.setPatientIdentifier(patientIdentifier);

		HealthAdvocateRequestDataService dataService = new HealthAdvocateRequestDataService();
		HealthAdvocateRequest createdHealthAdvocateRequest = dataService.createNewHealthAdvocateRequest(patientIdentifier, healthAdvocateRequest);

		if (u != null) {
			HealthAdvocateRequestLinkBuilder linkBuilder = new HealthAdvocateRequestLinkBuilder(u.getBaseUri());
			linkBuilder.fillLinks(createdHealthAdvocateRequest, u.getRequestUri());
		}

		return createdHealthAdvocateRequest;
	}

	@PUT
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	@Path("system/{system-id}/id/{id}")
	public HealthAdvocateRequest updateHealthAdvocateRequest(HealthAdvocateRequest healthAdvocateRequest, @PathParam("assigning-authority") String assigningAuthority,
			@PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("id") String id, @Context UriInfo u, @Context HttpHeaders headers) {
		return processUpdateHealthAdvocateRequest(healthAdvocateRequest,
				assigningAuthority, patientId, systemId, id, u);
	}
	
	@GET
	@Produces("application/pdf")
	@Path("system/{system-id}/id/{id}")
	public Response getHealthAdvocateRequestForm(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemId,
			@PathParam("id") String id, @Context UriInfo u, @Context HttpHeaders headers) throws URISyntaxException {

		return processRequestForm(assigningAuthority, patientId, systemId, id);
	}

	@GET
	@Produces("application/pdf")
	@Path("system/{system-id}/id/{id}/Revoked")
	public Response getHealthAdvocateRequestFormRevoked(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemId,
			@PathParam("id") String id, @Context UriInfo u, @Context HttpHeaders headers) throws URISyntaxException {

		return processRequestForm(assigningAuthority, patientId, systemId, id, HealthAdvocateStatusCode.REVOKED);
	}

	@GET
	@Produces("application/pdf")
	@Path("system/{system-id}/id/{id}/Cancelled")
	public Response getHealthAdvocateRequestFormCancelled(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemId,
			@PathParam("id") String id, @Context UriInfo u, @Context HttpHeaders headers) throws URISyntaxException {

		return processRequestForm(assigningAuthority, patientId, systemId, id, HealthAdvocateStatusCode.CANCELLED);
	}
	
	@GET
	@Path("/eligibility")
	@Produces({ "application/xml", "application/json" })
	public HealthAdvocateRequestEligibility fetchHealthAdvocateRequestEligibility(@PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
			@Context UriInfo uriInfo, @Context HttpHeaders headers) {
		
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		HealthAdvocateRequestDataService dataService = new HealthAdvocateRequestDataService();

		HealthAdvocateRequestEligibility availability = dataService.fetchHealthAdvocateRequestEligibility(patientIdentifier);
		
		HealthAdvocateRequestEligibilityLinkBuilder linkbuilder = new HealthAdvocateRequestEligibilityLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(availability, uriInfo.getBaseUri());
		
		return availability;
	}

    private HealthAdvocateRequest processUpdateHealthAdvocateRequest(
            HealthAdvocateRequest healthAdvocateRequest,
            String assigningAuthority, String patientId, String systemId,
            String id, UriInfo u) {

        HealthAdvocateRequest updatedHealthAdvocateRequest = persistUpdatedRequest(
                healthAdvocateRequest, assigningAuthority, patientId, systemId,
                id);

        if (u != null) {
            HealthAdvocateRequestLinkBuilder linkBuilder = new HealthAdvocateRequestLinkBuilder(u.getBaseUri());
            linkBuilder.fillLinks(updatedHealthAdvocateRequest, u.getRequestUri());
        }

        return updatedHealthAdvocateRequest;
    }

    private HealthAdvocateRequest persistUpdatedRequest(
            HealthAdvocateRequest healthAdvocateRequest,
            String assigningAuthority, String patientId, String systemId,
            String id) {
        
    	PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        final DataIdentifier dataIdentifier = new DataIdentifier(systemId, id);

        healthAdvocateRequest.setPatientIdentifier(patientIdentifier);
        HealthAdvocateRequestDataService dataService = new HealthAdvocateRequestDataService();
        HealthAdvocateRequest updatedHealthAdvocateRequest = dataService.updateHealthAdvocateRequest(patientIdentifier, dataIdentifier, healthAdvocateRequest);

        byte[] bytes = getPdfBytes(assigningAuthority, patientId, systemId, id);
        HealthAdvocateRequestForm healthAdvocateRequestForm = new HealthAdvocateRequestForm(updatedHealthAdvocateRequest);
        healthAdvocateRequestForm.setFormDataBytes(bytes);
        healthAdvocateRequestForm.setRequestDate(new Date());
        healthAdvocateRequestForm.setPatientId(patientId);
        healthAdvocateRequestForm.setStatus(healthAdvocateRequest.getStatus());
        HealthAdvocateRequestForm acceptedHealthAdvocateRequestForm = dataService.saveHealthAdvocateRequestForm(patientIdentifier, healthAdvocateRequestForm);

        return updatedHealthAdvocateRequest;
    }

    private byte[] getPdfBytes(String assigningAuthority,
                               String patientId, String systemId, String id) {

        final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        final DataIdentifier dataIdentifier = new DataIdentifier(systemId, id);

        HealthAdvocateRequest healthAdvocateRequest = getHealthAdvocateRequest(patientIdentifier, dataIdentifier);

        return buildPdfResponseAsBytes(patientIdentifier, healthAdvocateRequest);
    }

    private Response processRequestForm(String assigningAuthority,
			String patientId, String systemId, String id) {

		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DataIdentifier dataIdentifier = new DataIdentifier(systemId, id);

		HealthAdvocateRequest healthAdvocateRequest = getHealthAdvocateRequest(patientIdentifier, dataIdentifier);
		
		return buildPdfResponse(patientIdentifier, healthAdvocateRequest);
	}

	//Allows the request status to be specified
	private Response processRequestForm(String assigningAuthority,
			String patientId, String systemId, String id, String requestFormStatus) {

		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DataIdentifier dataIdentifier = new DataIdentifier(systemId, id);

		HealthAdvocateRequest healthAdvocateRequest = getHealthAdvocateRequest(patientIdentifier, dataIdentifier);
		
		healthAdvocateRequest.setStatus(requestFormStatus);
		
		return buildPdfResponse(patientIdentifier, healthAdvocateRequest);
	}
	
	private HealthAdvocateRequest getHealthAdvocateRequest(PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier){

		HealthAdvocateRequestDataService dataService = new HealthAdvocateRequestDataService();

		return dataService.fetchHealthAdvocateRequest(patientIdentifier, dataIdentifier);
	}

	private byte[] buildPdfResponseAsBytes(PatientIdentifier patientIdentifier, HealthAdvocateRequest healthAdvocateRequest){
		final PdfGenerationContext context;

		try {
			context = buildContext(patientIdentifier, healthAdvocateRequest);
		} catch (IllegalStateException e) {
			logger.error("Failed to generate the PDF context prior to writing the HealthAdvocate PDF to bytes", e);
			return null;
		}

        ByteArrayOutputStream os = new ByteArrayOutputStream();
        try {
            writeFormToStream(context, os);
        } catch (IOException e) {
            logger.error("Failed to write HealthAdvocate PDF as bytes", e);
        }
        return os.toByteArray();
	}

	private Response buildPdfResponse(PatientIdentifier patientIdentifier, HealthAdvocateRequest healthAdvocateRequest){
		final PdfGenerationContext context;

		try {
			context = buildContext(patientIdentifier, healthAdvocateRequest);
		} catch (IllegalStateException e) {
			logger.error("Illegal state while building PDF response", e);
			return null;
		}

		StreamingOutput stream = generatePdfToStreamingOutput(context);
		return Response.ok(stream).build();
	}

	

	private PdfGenerationContext buildContext(PatientIdentifier patientIdentifier, HealthAdvocateRequest healthAdvocateRequest) {

		PdfGenerationContext context = new PdfGenerationContext();

		context.setPatientIdentifier(patientIdentifier);
		context.setLocale(Locale.US);

		context.setVisible(Domain.healthAdvocateRequest, true);

		setHealthAdvocateRequestDomain(context, healthAdvocateRequest);

		return context;
	}

	private void setHealthAdvocateRequestDomain(PdfGenerationContext context, HealthAdvocateRequest healthAdvocateRequest) {

		if (context.isVisible(Domain.healthAdvocateRequest)) {
			context.setInformation(Domain.healthAdvocateRequest, healthAdvocateRequest);
		}

	}

    public void writeFormToStream(PdfGenerationContext context, OutputStream output) throws IOException, WebApplicationException {
        try {
            PdfDataService dataService = new PdfDataService();
            dataService.generateHealthAdvocateRequestForm(output, context);
        } catch (Exception e) {
            logger.error("Error generating PDF Report", e);
            throw new WebApplicationException(Status.INTERNAL_SERVER_ERROR);
        }
    }

    private StreamingOutput generatePdfToStreamingOutput(final PdfGenerationContext context) {
		StreamingOutput stream = new StreamingOutput() {
			public void write(OutputStream output) throws IOException, WebApplicationException {
				try {
					PdfDataService dataService = new PdfDataService();
					dataService.generateHealthAdvocateRequestForm(output, context);
				} catch (Exception e) {
					logger.error("Error generating PDF Report", e);
					throw new WebApplicationException(Status.INTERNAL_SERVER_ERROR);
				}
			}
		};
		return stream;
	}
}
