package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.notification.DeviceRegistrationDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistration;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistrations;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.DeviceRegistrationUriHelper;

@Path("/notification-service/user/id/{assigning-authority}/{user-id}/device-registrations")
@Component
@Scope("request")
public class DeviceRegistrationResource {

	/***
	 * A resource that gets all {@link DeviceRegistrations} associated with this
	 * user
	 * 
	 * @param userId
	 * @return The {@link DeviceRegistrations} associated with this user
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	public DeviceRegistrations getDeviceRegistrations(@PathParam("user-id") String userId, @Context UriInfo uriInfo) {
		DeviceRegistrationDataService dataservice = new DeviceRegistrationDataService();
		DeviceRegistrations deviceRegistrations = dataservice.fetchRegistrations(userId);
		DeviceRegistrationUriHelper.updateDeviceRegistrationsLinks(deviceRegistrations, uriInfo);
		return deviceRegistrations;
	}
	
	/***
	 * A resource that gets a specific {@link DeviceRegistration} as specified
	 * by the device-token
	 * 
	 * @param userId
	 * @param deviceToken
	 * @return The requested {@link DeviceRegistration} associated with this
	 *         user
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("/token/{device-token}")
	public DeviceRegistration getDeviceRegistration(@PathParam("user-id") String userId, @PathParam("device-token") String deviceToken, @Context UriInfo uriInfo) {
		DeviceRegistrationDataService dataservice = new DeviceRegistrationDataService();
		DeviceRegistration deviceRegistration = dataservice.fetchRegistration(userId, deviceToken);

		if (deviceRegistration != null) {
			DeviceRegistrationUriHelper.updateDeviceRegistrationLinks(deviceRegistration, uriInfo);
			return deviceRegistration;
		} else {
			throw new WebApplicationException(Status.NOT_FOUND);
		}
	}

	/***
	 * A resource that saves a new {@link DeviceRegistration} for the specified
	 * user
	 * 
	 * @param deviceRegistration
	 * @param userId
	 * @return The {@link DeviceRegistration} that was saved
	 */
	@POST
	@Consumes({ "application/xml", "application/json" })
	@Produces({ "application/xml", "application/json" })
	public DeviceRegistration createDeviceRegistration(DeviceRegistration deviceRegistration, @PathParam("user-id") String userId, @Context UriInfo uriInfo) {
		deviceRegistration.setUserId(userId);

		DeviceRegistrationDataService dataservice = new DeviceRegistrationDataService();
		DeviceRegistration savedDeviceRegistration = dataservice.saveRegistration(deviceRegistration);
		
		if(uriInfo != null && savedDeviceRegistration != null) {
			DeviceRegistrationUriHelper.updateDeviceRegistrationLinks(savedDeviceRegistration, uriInfo);
		}
		
		return savedDeviceRegistration;
	}

	/**
	 * Updates an existing {@link DeviceRegistration}.
	 * 
	 * @param deviceRegistration
	 * @param userId
	 * @param uriInfo
	 * @return
	 */
	@PUT
	@Produces({ "application/xml", "application/json" })
	public DeviceRegistration updateDeviceRegistration(DeviceRegistration deviceRegistration, @PathParam("user-id") String userId, @Context UriInfo uriInfo) {
		deviceRegistration.setUserId(userId);

		DeviceRegistrationDataService dataservice = new DeviceRegistrationDataService();
		dataservice.updateOptStatus(deviceRegistration);

		DeviceRegistrationUriHelper.updateDeviceRegistrationLinks(deviceRegistration, uriInfo);

		return deviceRegistration;
	}
	
	/***
	 * A resource that deletes a specific {@link DeviceRegistration} as
	 * specified by the device-token
	 * 
	 * @throws {@link WebApplicationException} (Status.BAD_REQUEST)
	 * @param userId
	 * @param deviceToken
	 * @return The {@link DeviceRegistration} that has been deleted
	 */
	@DELETE
	@Consumes({ "application/xml", "application/json" })
	@Path("/token/{device-token}")
	public void deleteDeviceRegistration(@PathParam("user-id") String userId, @PathParam("device-token") String deviceToken, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		DeviceRegistrationDataService dataservice = new DeviceRegistrationDataService();
		dataservice.deleteRegistration(userId, deviceToken);
	}

}
