package com.agilex.healthcare.mobilehealthplatform.restservice;

import java.util.Date;
import java.util.Set;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

//import com.agilex.healthcare.mobilehealthplatform.appointmentrequestmessage.AppointmentRequestMessageNotificationGateway;
//import com.agilex.healthcare.mobilehealthplatform.appointmentrequestmessage.MessagingFactory;
import com.agilex.healthcare.mobilehealthplatform.datalayer.appointment.AppointmentMetricsDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.appointment.AppointmentRequestDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.appointment.AppointmentRequestFilter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.appointment.AppointmentRequestInProcessDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.appointment.DetailCodeDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientMetadataDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentMetric;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequest;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestDetailCode;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestInProcess;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestMessage;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestMessages;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequests;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentsMetadata;
import com.agilex.healthcare.mobilehealthplatform.domain.DataIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.DetailCode;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppType;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentMetrics;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestStatus;
import com.agilex.healthcare.mobilehealthplatform.restservice.AbstractUserResource;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.utils.PilotFacilitiesHelper;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.AppointmentRequestLinkBuilder;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.AppointmentRequestMessagesLinkBuilder;
import com.agilex.healthcare.mobilehealthplatform.validator.appointment.AppointmentRequestValidator;

@Path("/appointment-service")
@Component
@Scope("request")
public class AppointmentRequestResource extends AbstractUserResource {

    @Resource
    AppointmentsMetadata appointmentsMetadata;

    @Resource
    PilotFacilitiesHelper pilotFacilitiesHelper;

    @Resource
    String haMode;

    @POST
    @Path("/metrics/no-icn-or-dfn")
    @Produces({"application/xml", "application/json"})
    public AppointmentMetric incrementNoIcnOrDfnMetric(@Context HttpServletRequest request, @Context UriInfo uriInfo) {
        AppointmentMetricsDataService dataService = new AppointmentMetricsDataService();
        return dataService.incrementMetric(AppointmentMetrics.NO_ICN_OR_DFN.getName());
    }

    @POST
    @Path("/metrics/no-dfn-cancel-request")
    @Produces({"application/xml", "application/json"})
    public AppointmentMetric incrementNoDfnCancelRequestMetric(@Context HttpServletRequest request, @Context UriInfo uriInfo) {
        AppointmentMetricsDataService dataService = new AppointmentMetricsDataService();
        return dataService.incrementMetric(AppointmentMetrics.NO_DFN_CANCEL_REQUEST.getName());
    }

    @GET
    @Path("/metadata")
    @Produces({"application/xml", "application/json"})
    public AppointmentsMetadata getAppointmentsMetadata() {
        DetailCodeDataService service = new DetailCodeDataService();
        Set<DetailCode> detailCodes = service.fetchDetailCodes();
        appointmentsMetadata.setDetailCodes(detailCodes);
        return appointmentsMetadata;
    }

    /**
     * A resource that returns all {@link AppointmentRequests} that have been
     * created by this user.
     *
     * @return Returns {@link AppointmentRequests} associated with the user
     *         calling the request
     */
    @GET
    @Path("/appointment-requests")
    @Produces({"application/xml", "application/json"})
    public AppointmentRequests getAppointmentRequestsByCriteria(@QueryParam("parentSiteCode") String parentSiteCode, @QueryParam("noSort") Boolean noSort, @Context UriInfo uriInfo) {
        AppointmentRequests appointmentRequests;
        AppointmentRequestDataService appointmentRequestDataService = new AppointmentRequestDataService();

        AppointmentRequestFilter filter = getAppointmentRequestCriteria(parentSiteCode, uriInfo);

        if (noSort != null && noSort) {
            appointmentRequests = appointmentRequestDataService.getAppointmentRequestsNoSort(filter);

        } else {
            appointmentRequests = appointmentRequestDataService.getAppointmentRequests(filter);

        }

        updateAtomLinks(uriInfo, appointmentRequests);

        return appointmentRequests;
    }

    private AppointmentRequestFilter getAppointmentRequestCriteria(String parentSiteCode, UriInfo uriInfo) {
        DateFilter dateFilter = DateFilterFactory.createFilterFromUri(uriInfo);

        AppointmentRequestFilter appointmentRequestFilter = new AppointmentRequestFilter();
        appointmentRequestFilter.setParentSiteCode(parentSiteCode);
        appointmentRequestFilter.setStartDate(dateFilter.getStartDate());
        appointmentRequestFilter.setEndDate(dateFilter.getEndDate());

        return appointmentRequestFilter;
    }

    /**
     * A resource that returns all {@link AppointmentRequests} that have been
     * created by this user.
     *
     * @return Returns {@link AppointmentRequests} associated with the user
     *         calling the request
     */
    @GET
    @Path("/patient/{assigning-authority}/{patient-id}/appointments")
    @Produces({"application/xml", "application/json"})
    public AppointmentRequests getAppointmentRequests(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo uriInfo) {
        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        DateFilter dateFilter = DateFilterFactory.createFilterFromUri(uriInfo);
        AppointmentRequests appointmentRequests;

        AppointmentRequestDataService appointmentRequestDataService = new AppointmentRequestDataService();
        appointmentRequests = appointmentRequestDataService.getPatientAppointmentRequests(patientIdentifier, dateFilter, ScopeFilter.getInstanceForLongitudinalScope());

        PatientMetadataDataService patientMetadataDataService = new PatientMetadataDataService();
        appointmentRequests.setLastAccessDate(patientMetadataDataService.fetchAndUpdateLastAccessDate(patientIdentifier, ScopeFilter.getInstanceForLongitudinalScope()));

        updateAtomLinks(uriInfo, appointmentRequests);

        return appointmentRequests;
    }

    /**
     * A resource that returns all {@link AppointmentRequestMessages} that have
     * been created by this user.
     *
     * @return Returns {@link AppointmentRequestMessages} associated with the
     *         user calling the request
     */
    @GET
    @Path("/patient/{assigning-authority}/{patient-id}/appointment-requests/system/{system-id}/id/{appointment-request-id}/messages")
    @Produces({"application/xml", "application/json"})
    public AppointmentRequestMessages getAppointmentRequestMessages(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemId,
                                                                    @PathParam("appointment-request-id") String appointmentRequestId, @Context UriInfo uriInfo) {
        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        DataIdentifier dataIdentifier = new DataIdentifier(systemId, appointmentRequestId);

        AppointmentRequestDataService appointmentRequestDataService = new AppointmentRequestDataService();
        AppointmentRequestMessages messages = appointmentRequestDataService.fetchAppointmentRequestMessages(patientIdentifier, dataIdentifier);

        updateAtomLinks(uriInfo, messages);

        return messages;
    }

    @POST
    @Path("/patient/{assigning-authority}/{patient-id}/appointment-requests/system/{system-id}/id/{appointment-request-id}/messages")
    @Consumes({"application/xml", "application/json"})
    @Produces({"application/xml", "application/json"})
    public AppointmentRequestMessage createAppointmentRequestMessage(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("appointment-request-id") String appointmentRequestId, @PathParam("system-id") String systemId ,AppointmentRequestMessage appointmentRequestMessage) {


        if (appointmentRequestId.equals(appointmentRequestMessage.getAppointmentRequestId())) {
            appointmentRequestMessage.setMessageDateTime(new Date());
            appointmentRequestMessage.setPatientIdentifier(new PatientIdentifier(assigningAuthority, patientId));
            appointmentRequestMessage.setDataIdentifier(new DataIdentifier(systemId, appointmentRequestId));

            AppointmentRequestDataService dataService = new AppointmentRequestDataService();
            AppointmentRequestMessage result = dataService.saveAppointmentRequestMessage(appointmentRequestMessage);

            //temp fix for save not returning patientIdentifier
            if ((result.getPatientIdentifier() == null) || (result.getPatientIdentifier().getUniqueId() == null) || (result.getPatientIdentifier().getAssigningAuthority() == null)) {
                result.setPatientIdentifier(new PatientIdentifier(assigningAuthority, patientId));
            }

            //Only send notification if message is staff generated (don't notify veteran of action they just did themselves)
//            if (MhpUserFactory.doesUserHasStaffRole()) {
//                sendNotificationMessage(result);
//            }

            return result;
        } else {
            return null;
        }
    }

    @POST
    @Path("/patient/{assigning-authority}/{patient-id}/appointment-requests/system/{system-id}/id/{appointment-request-id}/messages/read")
    @Consumes({"application/xml", "application/json"})
    @Produces({"application/xml", "application/json"})
    public Response markMessagesAsRead(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("appointment-request-id") String appointmentRequestId) {

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        DataIdentifier dataIdentifier = new DataIdentifier(systemId, appointmentRequestId);

        AppointmentRequestDataService dataService = new AppointmentRequestDataService();
        AppointmentRequest appointmentRequest = dataService.updateAppointmentMessageFlag(appointmentRequestId,patientIdentifier,dataIdentifier);
        appointmentRequest.setPatientIdentifier(patientIdentifier);
        appointmentRequest.setDataIdentifier(dataIdentifier);
        if (appointmentRequest.getAppointmentRequestId().equals(appointmentRequestId)) {
            return Response.ok().build();
        } else {
            return Response.serverError().build();
        }
    }

//    private void sendNotificationMessage(AppointmentRequestMessage result) {
//        AppointmentRequestMessageNotificationGateway messageGateway = MessagingFactory.getAppointmentRequestMessageNotificationGateway();
//
//        messageGateway.sendMessageToPatient(result);
//    }

    private void updateAtomLinks(UriInfo uriInfo, AppointmentRequestMessages messages) {
        if (uriInfo != null) {
            AppointmentRequestMessagesLinkBuilder linkBuilder = new AppointmentRequestMessagesLinkBuilder(uriInfo.getBaseUri());
            linkBuilder.fillLinks(messages, uriInfo.getRequestUri());
        }
    }

    private void updateAtomLinks(UriInfo uriInfo, AppointmentRequests appointmentRequests) {
        if (uriInfo != null) {
            AppointmentRequestLinkBuilder linkBuilder = new AppointmentRequestLinkBuilder(uriInfo.getBaseUri());
            linkBuilder.fillLinks(appointmentRequests, uriInfo.getRequestUri());
        }
    }

    /**
     * A resource that returns a single {@link AppointmentRequest} by user and
     * id
     *
     * @return Returns {@link AppointmentRequest} associated with the id and
     *         user calling the request
     */
    @GET
    @Path("/patient/{assigning-authority}/{patient-id}/appointments/system/{system-id}/id/{appointment-request-id}")
    @Produces({"application/xml", "application/json"})
    public AppointmentRequest getAppointmentRequest(@PathParam("assigning-authority") String assigningAuthority, @PathParam("system-id") String systemId, @PathParam("patient-id") String patientId,
                                                    @PathParam("appointment-request-id") String appointmentRequestId, @Context UriInfo uriInfo) {
        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        DataIdentifier dataIdentifier = new DataIdentifier(systemId, appointmentRequestId);
        AppointmentRequest appointmentRequest = null;

        AppointmentRequestDataService appointmentRequestDataService = new AppointmentRequestDataService();
        appointmentRequest = appointmentRequestDataService.getPatientAppointmentRequest(patientIdentifier, dataIdentifier);
        AppointmentRequestInProcess beingProcessedBy = retrieveAppointmentRequestInProcess(appointmentRequest.getUniqueId());
        appointmentRequest.setBeingProcessedBy(beingProcessedBy);

        updateAtomLinks(appointmentRequest, uriInfo);

        return appointmentRequest;
    }

    private AppointmentRequestInProcess retrieveAppointmentRequestInProcess(String appointmentRequestId) {
        AppointmentRequestInProcessDataService service = new AppointmentRequestInProcessDataService();
        return service.fetchAppointmentRequestInProcessById(appointmentRequestId);
    }

    /**
     * A resource that saves a single {@link AppointmentRequest} by user and id
     *
     * @return Returns {@link AppointmentRequest} associated with the id and
     *         user calling the request
     */
    @PUT
    @Path("/patient/{assigning-authority}/{patient-id}/appointments/system/{system-id}/id/{appointment-request-id}")
    @Consumes({"application/xml", "application/json"})
    @Produces({"application/xml", "application/json"})
    public AppointmentRequest updateAppointmentRequest(@Context HttpServletRequest request, @PathParam("assigning-authority") String assigningAuthority, @PathParam("system-id") String systemId, @PathParam("patient-id") String patientId,
                                                       @PathParam("appointment-request-id") String appointmentRequestId, AppointmentRequest appointmentRequest, @Context UriInfo uriInfo) {
        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        DataIdentifier dataIdentifier = new DataIdentifier(systemId, appointmentRequestId);
        appointmentRequest.setPatientIdentifier(patientIdentifier);
        appointmentRequest.setDataIdentifier(dataIdentifier);
        appointmentRequest.setLastUpdatedDate(new Date());

        for (AppointmentRequestDetailCode ardc : appointmentRequest.getAppointmentRequestDetailCode()) {
            if (ardc.getCreatedDate() == null) {
                ardc.setCreatedDate(new Date());
            }
        }


        performStatusChangeProcessing(appointmentRequest);


        boolean fullValidation = false;
        appointmentRequest = saveAppointmentRequest(appointmentRequest, fullValidation, uriInfo);

        return appointmentRequest;
    }

    private void performStatusChangeProcessing(AppointmentRequest appointmentRequest) {

        if (!MhpUserFactory.doesUserHasStaffRole()) {
            return;  //no need to send Notification if veteran is performing the action
        }

        //retrieve existing AR
        AppointmentRequestDataService dataService = new AppointmentRequestDataService();
        AppointmentRequest existing = dataService.getPatientAppointmentRequest(appointmentRequest.getPatientIdentifier(), appointmentRequest.getDataIdentifier());

        //if status has changed
        if (!existing.getStatus().equals(appointmentRequest.getStatus())) {
            //generate notification message
//            AppointmentRequestMessageNotificationGateway messageGateway = MessagingFactory.getAppointmentRequestMessageNotificationGateway();
//
//            messageGateway.sendMessageToPatient(appointmentRequest.getPatientIdentifier());
        }
    }

    private void setCreateDateInDetailCodesOfParentAppointmentRequest(AppointmentRequest appointmentRequest) {
        AppointmentRequest parentAppointmentRequest = appointmentRequest.getParentRequest();

        if (parentAppointmentRequest != null) {
            for (AppointmentRequestDetailCode ardc : parentAppointmentRequest.getAppointmentRequestDetailCode()) {
                if (ardc.getCreatedDate() == null) {
                    ardc.setCreatedDate(new Date());
                }
            }
        }
    }

    /**
     * A resource that allows a client to save an {@link AppointmentRequest}.
     * See {@link AppointmentRequestValidator} for val idation logic.
     *
     * @param AppointmentRequest
     * @return The {@link AppointmentRequest} that was saved.
     */
    @POST
    @Path("/patient/{assigning-authority}/{patient-id}/appointments")
    @Consumes({"application/xml", "application/json"})
    @Produces({"application/xml", "application/json"})
    public AppointmentRequest submitAppointmentRequest(@Context HttpServletRequest request, @PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, AppointmentRequest appointmentRequest,
                                                       @Context UriInfo uriInfo) {
        prepareAppointmentRequestForSubmission(request.getSession(), appointmentRequest, assigningAuthority, patientId);

        boolean fullValidation = true;
        setCreateDateInDetailCodesOfParentAppointmentRequest(appointmentRequest);
        appointmentRequest = saveAppointmentRequest(appointmentRequest, fullValidation, uriInfo);

        return appointmentRequest;
    }

    /**
     * @param AppointmentRequestInProcess
     * @return The {@link AppointmentRequestInProcess} that was saved.
     */
    @POST
    @Path("/patient/{assigning-authority}/{patient-id}/appointments/system/{system-id}/id/{appointment-request-id}/inprocess")
    @Produces({"application/xml", "application/json"})
    public Response saveAppointmentRequestInProcess(@PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("appointment-request-id") String appointmentRequestId) {
        Response response = null;
        AppointmentRequestInProcess beingProcessedBy = null;
        beingProcessedBy = retrieveAppointmentRequestInProcess(appointmentRequestId);

        if (beingProcessedBy == null) {
            MhpUser mhpUser = MhpUserFactory.createFromSecurityContext();
            beingProcessedBy = new AppointmentRequestInProcess();
            beingProcessedBy.setAppointmentRequestId(appointmentRequestId);
            beingProcessedBy.setUserId(mhpUser.getId());
            beingProcessedBy.setFirstName(mhpUser.getFirstName());
            beingProcessedBy.setLastName(mhpUser.getLastName());

            AppointmentRequestInProcessDataService service = new AppointmentRequestInProcessDataService();
            beingProcessedBy = service.saveAppointmentRequestInProcess(beingProcessedBy);

            response = Response.ok(beingProcessedBy).build();
        } else if (beingProcessedBy.getUserId().equals(getCurrentUserId())) {
            response = Response.ok(beingProcessedBy).build();
        } else {
            response = Response.status(400).entity(beingProcessedBy).build();
        }

        return response;
    }

    /**
     * @param AppointmentRequestInProcess
     * @return The {@link AppointmentRequestInProcess} that was saved.
     */
    @DELETE
    @Path("/patient/{assigning-authority}/{patient-id}/appointments/system/{system-id}/id/{appointment-request-id}/inprocess")
    @Produces({"application/xml", "application/json"})
    public Response deleteAppointmentRequestInProcess(@PathParam("system-id") String systemId, @PathParam("appointment-request-id") String appointmentRequestId) {

        AppointmentRequestInProcessDataService service = new AppointmentRequestInProcessDataService();
        service.deleteAppointmentRequestInProcessById(appointmentRequestId);

        return Response.ok().build();
    }

    @PUT
    @Path("/patient/{assigning-authority}/{patient-id}/appointment-requests/system/{system-id}/id/{appointment-request-id}/appointment/provider-read")
    @Consumes({"application/xml", "application/json"})
    @Produces({"application/xml", "application/json"})
    public Response markAppointmentRequestIsSeenByStaff(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("appointment-request-id") String appointmentRequestId) {

        if (MhpUserFactory.doesUserHasStaffRole() && appointmentRequestId != null) {
            PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
            DataIdentifier dataIdentifier = new DataIdentifier(systemId, appointmentRequestId);
            AppointmentRequestDataService dataService = new AppointmentRequestDataService();
            AppointmentRequest updatedAppointmentRequest = dataService.updateProviderSeeAppointmentRequestFlag(appointmentRequestId, patientIdentifier, dataIdentifier);
            if(appointmentRequestId.equals(updatedAppointmentRequest.getAppointmentRequestId())){
            	return Response.ok().build();
            }
            return Response.serverError().build();
        } else {
            return Response.status(401).build();
        }
    }


    private AppointmentRequest saveAppointmentRequest(AppointmentRequest appointmentRequest, boolean fullValidation, UriInfo uriInfo) {
        AppointmentRequestDataService appointmentRequestDataService = new AppointmentRequestDataService();
        appointmentRequest = appointmentRequestDataService.saveAppointmentRequest(pilotFacilitiesHelper.getPilotSites(AppType.VARCAR.getName()), appointmentRequest, haMode, fullValidation, ScopeFilter.getInstanceForLongitudinalScope());
        updateAtomLinks(appointmentRequest, uriInfo);
        return appointmentRequest;
    }

    private void updateAtomLinks(AppointmentRequest appointmentRequest, UriInfo uriInfo) {
        AppointmentRequestLinkBuilder linkBuilder = new AppointmentRequestLinkBuilder(uriInfo.getBaseUri());
        linkBuilder.fillLinks(appointmentRequest, uriInfo.getRequestUri());
    }

    private void prepareAppointmentRequestForSubmission(HttpSession session, AppointmentRequest appointmentRequest, String assigningAuthority, String patientId) {
        Date now = new Date();

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        appointmentRequest.setPatientIdentifier(patientIdentifier);
        appointmentRequest.setStatus(AppointmentRequestStatus.SUBMITTED.getName());
        appointmentRequest.setCreatedDate(now);
        appointmentRequest.setLastUpdatedDate(now);
        appointmentRequest.setDeletedDate(null);

        if (!appointmentRequest.isTextMessagingAllowed()) {
            appointmentRequest.setTextMessagingPhoneNumber("");
        }
    }

}
