package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.antibiogram.AntibiogramDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.AntibiogramURI;
import com.agilex.healthcare.mobilehealthplatform.domain.antibiogram.AntibiogramMatrix;
import com.agilex.healthcare.mobilehealthplatform.domain.antibiogram.AntibiogramSummaries;
import com.agilex.healthcare.mobilehealthplatform.domain.antibiogram.OrganismGroups;
import com.agilex.healthcare.mobilehealthplatform.domain.antibiogram.Organisms;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.AntibiogramLinkBuilder;

@Path("/public/antibiogram/facility")
@Component
@Scope("request")
public class AntibiogramResource {

	/**
	 * This resource is used to get to the facility specific antibiogram
	 * resource. It is expected that the client is aware of the facility site
	 * code or will discover it via the facility search resource.
	 * 
	 * Note: it was a deliberate action to not make this a facility search
	 * resource as that would require run-time coupling between the facility
	 * search and antibiogram services. If desired in the future, then the
	 * antibiogram service should have its own copy of the facility list rather
	 * than making calls to the facility services.
	 * 
	 * @param facilitySiteCode
	 *            this query param is the facility code representing the
	 *            facility. Example for selecting DCVAMC:
	 *            "?facilitySiteCode=688"
	 * @param uriInfo
	 * @param headers
	 * @return Returns a set of links representing facility specific antibiogram
	 *         resources. These links include 
	 *         (fetchOrganismsSummaryBySite) "antibiogram-facility-organism"
	 *         (fetchOrganismsBySite) "antibiogram-facility-matrix"
	 * 
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	public AntibiogramURI fetchAntibiogramFacilityByFacilityCode(@QueryParam("facilitySiteCode") String facilitySiteCode, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		AntibiogramURI antibiogramURI = new AntibiogramURI();
		antibiogramURI.setFacilityCode(facilitySiteCode);
		AntibiogramLinkBuilder linkBuilder = new AntibiogramLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(antibiogramURI, uriInfo.getRequestUri());

		return antibiogramURI;
	}

	/**
	 * This resource is used to get the facility specific organism collection.
	 * It is expected that the client is sending facility site
	 * code as path param and organism term (first two chars of Organism name) as the query param
	 * or organism group term (first two chars of OrganismGroup name) as the query param
	 * 
	 * @param facilityCode
	 *            this path param is the facility code representing the
	 *            facility. Example for passing facility(DCVAMC) as path param:
	 *            "/688/organism"
	 * @param term
	 * 			first two chars of Organism name or Organism group name,
	 * 			Example for sending first two chars of organism name or organism group name: 
	 * 			"?term=CE"            
	 * @param uriInfo
	 * @param headers
	 * @return Returns organism collection {@link Organisms}
	 */	
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("/{facilityCode}/organism-group")
	public Organisms fetchOrganismsBySite(@PathParam("facilityCode") String facilitySiteCode, @QueryParam("term") String term, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		AntibiogramDataService dataService = new AntibiogramDataService();
		Organisms organisms = dataService.fetchOrganisms(facilitySiteCode, term);
		fillLinks(organisms, uriInfo);

		OrganismGroups organismGroups = dataService.fetchOrganismGroups(facilitySiteCode, term);
		fillLinks(organismGroups, uriInfo);
		
		organisms.setOrganismGroups(organismGroups);
		return organisms;
	}

	/**
	 * This resource is used to get the Organism to Antibiotics summaries for a specific facility and organism.
	 * It is expected that the client is sending facility site code
	 * as and organism id as path params.
	 * 
	 * @param facilityCode
	 *            this path param is the facility code representing the
	 *            facility. Example for passing facility(DCVAMC) as path param:
	 *            "/688/organism"
	 * @param organism-id
	 *            this path param is the organism id of the selected organism. 
	 *            Example for passing facility(DCVAMC) and organism-id as path params:
	 *            "/688/organism/1400002234"	 
	 * @param uriInfo
	 * @param headers
	 * @return Returns organism to antibiotics summaries collection {@link AntibiogramSummaries}
	 */		
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("/{facilityCode}/organism/{organism-id}/antibiotic")
	public AntibiogramSummaries fetchAntibioticsForOrganism(@PathParam("facilityCode") String facilitySiteCode, @PathParam("organism-id") String organismId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(uriInfo);

		AntibiogramDataService dataService = new AntibiogramDataService();
		AntibiogramSummaries summaries = dataService.fetchSummaries(facilitySiteCode, organismId, dateFilter);

		return summaries;
	}

	/**
	 * This resource is used to get the Organism to Antibiotics summaries for a specific facility and organism group.
	 * It is expected that the client is sending facility site code
	 * as and organism group id as path params.
	 * 
	 * @param facilityCode
	 *            this path param is the facility code representing the
	 *            facility. Example for passing facility(DCVAMC) as path param:
	 *            "/688/organism-group"
	 * @param organism-id
	 *            this path param is the organism id of the selected organism. 
	 *            Example for passing facility(DCVAMC) and organism-id as path params:
	 *            "/688/organism-group/1"	 
	 * @param uriInfo
	 * @param headers
	 * @return Returns organism to antibiotics summaries collection {@link AntibiogramSummaries}
	 */		
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("/{facilityCode}/organism-group/{organismgroup-id}/antibiotic")
	public AntibiogramSummaries fetchAntibioticsForOrganismGroup(@PathParam("facilityCode") String facilitySiteCode, @PathParam("organismgroup-id") String organismGroupId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(uriInfo);

		AntibiogramDataService dataService = new AntibiogramDataService();
		AntibiogramSummaries summaries = dataService.fetchSummariesByGroup(facilitySiteCode, organismGroupId, dateFilter);
		

		return summaries;
	}
	
	/**
	 * This resource is used to get to the facility specific organisms to antibiotics matrix.
	 * It is expected that the client is sending facility site code as path param.
	 * 
	 * @param facilityCode
	 *            this path param is the facility code representing the
	 *            facility. Example for sending the DCVAMC as path param:
	 *            "/688/matrix"
	 * @param uriInfo
	 * @param headers
	 * @return Returns organisms to antiobiotics matrix collection {@link AntibiogramMatrix}
	 */		
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("/{facilityCode}/matrix")
	public AntibiogramMatrix fetchOrganismsSummaryBySite(@PathParam("facilityCode") String facilitySiteCode, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(uriInfo);

		AntibiogramDataService dataService = new AntibiogramDataService();
		AntibiogramMatrix antibiogramMatrix = dataService.fetchMatrix(facilitySiteCode, dateFilter);

		return antibiogramMatrix;
	}

	private void fillLinks(Organisms organisms, UriInfo uriInfo) {
		AntibiogramLinkBuilder linkBuilder = new AntibiogramLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(organisms, uriInfo.getRequestUri());
	}

	private void fillLinks(OrganismGroups organismGroups, UriInfo uriInfo) {
		AntibiogramLinkBuilder linkBuilder = new AntibiogramLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(organismGroups, uriInfo.getRequestUri());
	}
	
}
