package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import com.agilex.healthcare.mobilehealthplatform.datalayer.JournalEventsDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.DataIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientData;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientDataCollection;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.MhpObjectFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;

//@Path("/patient/{assigning-authority}/{patient-id}/diet")
//@Component
//@Scope("request")
public abstract class AbstractJournalResource<Ts extends PatientDataCollection<T>, T extends PatientData> {
	private JournalEventsDataService<Ts, T> dataService;
	private Class<T> dataitemType;

	public AbstractJournalResource(JournalEventsDataService<Ts, T> dataservice, Class<T> dataitemType) {
		this.dataService = dataservice;
		this.dataitemType = dataitemType;
	}

	@GET
	@Produces({ "application/xml", "application/json" })
	public Ts fetchDataList(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(u);
		return fetchDataList(patientIdentifier, dateFilter, ScopeFilter.getInstanceForLongitudinalScope(), u);
	}

	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("scope/{scope}")
	public Ts fetchDataListByScope(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("scope") String scope, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(u);
		return fetchDataList(patientIdentifier, dateFilter, ScopeFilter.getInstanceForScope(scope), u);
	}

	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("system/{system-id}")
	public Ts fetchDataListBySystem(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemIdentifier, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(u);
		return fetchDataList(patientIdentifier, dateFilter, ScopeFilter.getInstanceForSystem(systemIdentifier), u);
	}

	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("system/{system-id}/id/{id}")
	public T fetchData(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("id") String id, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DataIdentifier dataIdentifier = new DataIdentifier(systemId, id);

		T data = dataService.fetchData(patientIdentifier, dataIdentifier);

		fillLinks(data, u);

		if (data == null)
			throw ExceptionFactory.generateNoDataFoundException();

		return data;
	}

	private Ts fetchDataList(final PatientIdentifier patientIdentifier, DateFilter dateFilter, ScopeFilter scopeFilter, UriInfo u) {
		Ts datalist = dataService.fetchDataList(patientIdentifier, dateFilter, scopeFilter);
		fillLinks(datalist, u);
		return datalist;
	}

	@POST
	@Consumes({ "application/xml", "application/json" })
	public T createData(T data, @PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		data.setPatientIdentifier(patientIdentifier);

		// this really should be part of the path
		ScopeFilter scopeFilter = ScopeFilter.getInstanceForLongitudinalScope();

		T savedData = dataService.createData(data, scopeFilter);

		fillLinks(savedData, u);
		return savedData;
	}

	@PUT
	@Path("system/{system-id}/id/{id}")
	@Consumes({ "application/xml", "application/json" })
	@Produces({ "application/xml", "application/json" })
	public T saveData(T data, @PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("id") String id, @Context UriInfo u,
			@Context HttpHeaders headers) {

		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		data.setPatientIdentifier(patientIdentifier);

		final DataIdentifier eventIdentifier = new DataIdentifier(systemId, id);
		data.setDataIdentifier(eventIdentifier);

		T savedData = dataService.updateData(data);

		fillLinks(savedData, u);

		return savedData;
	}

	@DELETE
	@Path("system/{system-id}/id/{id}")
	public void deleteData(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("id") String id, @Context UriInfo u, @Context HttpHeaders headers) {

		T data = MhpObjectFactory.initializeByType(dataitemType);

		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		data.setPatientIdentifier(patientIdentifier);

		final DataIdentifier eventIdentifier = new DataIdentifier(systemId, id);
		data.setDataIdentifier(eventIdentifier);

		dataService.deleteData(data);
	}

	protected abstract void fillLinks(Ts datalist, UriInfo u);

	protected abstract void fillLinks(T data, UriInfo u);
}
