package com.agilex.healthcare.mobilehealthplatform.pdf;

import com.agilex.healthcare.mobilehealthplatform.domain.DailyEvent;
import com.agilex.healthcare.mobilehealthplatform.domain.Documents;
import com.agilex.healthcare.pdf.*;
import com.agilex.healthcare.utility.DateHelper;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Image;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfWriter;

import java.io.IOException;
import java.net.MalformedURLException;

public class DailyEventPage extends AbstractPage {
    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(DailyEventPage.class);

    public DailyEventPage(Document document, PageEvent event, PdfWriter writer, PdfGenerationContext context) {
        super(document, event, writer, context);
    }

    public void create() {
        try {
            setupReportHeaders();
            addHeading(context.getReportHeading(), "Heading");
            addGeneratedFor();
            addGenerationTime();

            DailyEvent dailyEvent = context.getDailyEvent();
            if(dailyEvent == null) {
                addNoDataFound();
                return;
            }
            addDailyEventInfo();
            addAttachmentListing();
            addNote();
            addAttachments();

        } catch (Exception e) {
            logger.error("Error in creating DailyEventPage", e);
            throw new PDFException("Error generating PDF DailyEventPage");
        }
    }

    private void addAttachments() throws DocumentException, IOException{
        Documents attachments = context.getAttachments();
        if(attachments == null || attachments.size() == 0) return;

        for(com.agilex.healthcare.mobilehealthplatform.domain.Document attachment : attachments) {
            addImage(attachment);
        }
    }

    private void addImage(com.agilex.healthcare.mobilehealthplatform.domain.Document attachment)
        throws DocumentException, IOException{
        PdfPTable table = new PdfPTable(1);
        table.setWidthPercentage(90);
        table.setSpacingBefore(20);
        table.setSpacingAfter(0);
        table.setHorizontalAlignment(PdfPTable.ALIGN_LEFT);

        PdfPCell cell = new PdfPCell();
        cell.setBorderWidth(0);
        cell.addElement(new Phrase(attachment.getDocumentTitle()));
        PdfPTable spacing = new PdfPTable(1);
        spacing.setSpacingAfter(5);
        spacing.getDefaultCell().setBorderWidth(0);
        spacing.addCell(new Phrase(""));
        cell.addElement(spacing);
        Image image = Image.getInstance(attachment.getBody().getContents());
        if(image.getScaledWidth() > 400 || image.getScaledHeight() > 300) {
            image.scaleToFit(400, 300);
        }
        cell.addElement(image);

        table.addCell(cell);
        document.add(table);
    }

    private void addNote() throws DocumentException{
        addPhrase(new Phrase("Note:", Stylesheet.getBoldLarge()), 10, 0, PdfPTable.ALIGN_LEFT, 0);
        addPhrase(new Phrase(""), 5, 0, PdfPTable.ALIGN_LEFT, 1);
        addPhrase(new Phrase(context.getDailyEvent().getNotes()));
    }

    private void addAttachmentListing() throws DocumentException{
        Documents attachments = context.getAttachments();
        if(attachments == null || attachments.size() == 0) return;

        addPhrase(new Phrase("Attachments:", Stylesheet.getBoldLarge()), 10, 0, PdfPTable.ALIGN_LEFT, 0);
        for(com.agilex.healthcare.mobilehealthplatform.domain.Document attachment : attachments) {
            addPhrase(new Phrase(attachment.getDocumentTitle()), 3, 0, PdfPTable.ALIGN_LEFT, 0);
        }
        addPhrase(new Phrase(""), 0, 15, PdfPTable.ALIGN_LEFT, 0);
    }

    private void addDailyEventInfo() throws DocumentException{
        PdfPTable table = new PdfPTable(2);
        table.setWidthPercentage(100);
        table.setWidths(new int[] {3, 25});
        table.setSpacingBefore(25);
        table.setSpacingAfter(10);
        table.setHorizontalAlignment(PdfPTable.ALIGN_LEFT);

        PdfPCell defaultCell = table.getDefaultCell();
        defaultCell.setPadding(2);
        defaultCell.setBorderWidth(0);
        defaultCell.setHorizontalAlignment(PdfPTable.ALIGN_LEFT);
        defaultCell.setVerticalAlignment(PdfPTable.ALIGN_CENTER);

        DailyEvent dailyEvent = context.getDailyEvent();
        table.addCell(new Phrase("Date:", Stylesheet.getBoldLarge()));
        table.addCell(DateHelper.formatDate(dailyEvent.getDate()));

        table.addCell(new Phrase("Time:", Stylesheet.getBoldLarge()));
        table.addCell(DateHelper.format(dailyEvent.getDate(), PatientEnteredVitalsMapper.timeFormat));

        table.addCell(new Phrase("Title:", Stylesheet.getBoldLarge()));
        table.addCell(dailyEvent.getTitle());
        document.add(table);
    }

    private void setupReportHeaders() {
        context.setReportHeading("Daily Event - From My VA Health");
    }
}
