package com.agilex.healthcare.mbb.restservice;

import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Locale;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.StreamingOutput;
import javax.ws.rs.core.UriInfo;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mbb.datalayer.PdfGenerationContext;
import com.agilex.healthcare.mbb.dataservice.PdfDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.admission.AdmissionDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.allergy.AllergyDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.appointment.AppointmentDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.lab.ChemistryLabDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.lab.micro.MicrobiologyLabDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationFilter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.problem.ProblemDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.radiology.RadiologyDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.surgery.SurgeryDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.VitalDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.VitalEntryFilter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.VitalsFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.domain.Admissions;
import com.agilex.healthcare.mobilehealthplatform.domain.Allergies;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointments;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResults;
import com.agilex.healthcare.mobilehealthplatform.domain.Medications;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientDemographics;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.Problems;
import com.agilex.healthcare.mobilehealthplatform.domain.RadiologyResults;
import com.agilex.healthcare.mobilehealthplatform.domain.Surgeries;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.restservice.AbstractUserResource;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.utils.SessionStateHelper;

@Path("/report")
@Component
@Scope("request")
public class ReportResource extends AbstractUserResource {
	protected static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(ReportResource.class);
	
	@Autowired
	MessageSource messageSource;
	
	@Resource
	SessionStateHelper sessionStateHelper;
	
	@GET
	@Produces("application/pdf")
	public Response getReport(
			@Context HttpServletRequest request,
			@Context UriInfo uriInfo) throws URISyntaxException {
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(uriInfo.getRequestUri());
		PatientIdentifier patientIdentifier = getPatientIdentifier();
		final PdfGenerationContext context;
		
		try {
			context = buildContext(request, patientIdentifier, dateFilter, uriInfo);
		} catch (IllegalStateException  e) {
			return Response.temporaryRedirect(new URI("../share")).build();
		}

		StreamingOutput stream = generatePdfToStreamingOutput(context);
		
		return Response.ok(stream).build();
	}
	
	private StreamingOutput generatePdfToStreamingOutput(final PdfGenerationContext context) {
		StreamingOutput stream = new StreamingOutput() {
			public void write(OutputStream output) throws IOException, WebApplicationException {
				try {
					PdfDataService dataService = new PdfDataService();
					dataService.generateReport(output, context);
				} catch (Exception e) {
					LOGGER.error("Error generating PDF Report", e);
					throw new WebApplicationException(Status.INTERNAL_SERVER_ERROR);
				}
			}
		};
		return stream;
	}

	private PdfGenerationContext buildContext(
			HttpServletRequest req, 
			PatientIdentifier patientIdentifier, 
			DateFilter dateFilter, 
			UriInfo uriInfo) {
		MultivaluedMap<String, String> parameters = uriInfo.getQueryParameters();
		
		boolean demographics = Boolean.valueOf(parameters.getFirst(Domain.demographic));
		boolean allergies = Boolean.valueOf(parameters.getFirst(Domain.allergy));
		boolean problems = Boolean.valueOf(parameters.getFirst(Domain.problem));
		boolean medications = Boolean.valueOf(parameters.getFirst(Domain.medication));
		boolean medicationHistory = Boolean.valueOf(parameters.getFirst(Domain.medicationHistory));
		boolean surgeries = Boolean.valueOf(parameters.getFirst(Domain.surgery));
		boolean radiology = Boolean.valueOf(parameters.getFirst(Domain.radiologyResult));
		boolean labs = Boolean.valueOf(parameters.getFirst(Domain.labResult));
		boolean appointments = Boolean.valueOf(parameters.getFirst(Domain.appointment));
		boolean admissions = Boolean.valueOf(parameters.getFirst(Domain.admission));
		boolean vitals = Boolean.valueOf(parameters.getFirst(Domain.vital));
		
		PdfGenerationContext context = new PdfGenerationContext();
		
		context.setPatientIdentifier(patientIdentifier);
		context.setDateFilter(dateFilter);
		context.setMessageSource(messageSource);
		context.setLocale(Locale.US);
		
		context.setVisible(Domain.demographic, demographics);
		context.setVisible(Domain.allergy, allergies);
		context.setVisible(Domain.problem, problems);
		context.setVisible(Domain.medication, medications);
		context.setVisible(Domain.medicationHistory, medicationHistory);
		context.setVisible(Domain.surgery, surgeries);
		context.setVisible(Domain.radiologyResult, radiology);
		context.setVisible(Domain.labResult, labs);
		context.setVisible(Domain.appointment, appointments);
		context.setVisible(Domain.admission, admissions);
		context.setVisible(Domain.vital, vitals);

		retrieveSessionState(req.getSession(), context, patientIdentifier);
		
		return context;
	}

	private void retrieveSessionState(HttpSession session, PdfGenerationContext context, PatientIdentifier patientIdentifier) {
		if (sessionStateHelper.isCacheEnabled()) {
			setPatientInformation(session, context, patientIdentifier);
			setDomain(session, context, patientIdentifier, Domain.allergy, Allergies.class);
			setTimeSensitiveDomain(session, context, patientIdentifier, Domain.appointment, Appointments.class);
			setTimeSensitiveDomain(session, context, patientIdentifier, Domain.admission, Admissions.class);
			setDomain(session, context, patientIdentifier, Domain.problem, Problems.class);
			setTimeSensitiveDomain(session, context, patientIdentifier, Domain.radiologyResult, RadiologyResults.class);
			setDomain(session, context, patientIdentifier, Domain.surgery, Surgeries.class);
			setTimeSensitiveDomain(session, context, patientIdentifier, Domain.vital, VitalEntries.class);
			setDomain(session, context, patientIdentifier, Domain.medication, Medications.class);
			setTimeSensitiveDomain(session, context, patientIdentifier, Domain.medicationHistory, Medications.class);
			setTimeSensitiveDomain(session, context, patientIdentifier, Domain.labResult, LabResults.class);
		} else {
			PatientDataService patientDataService = new PatientDataService();
			Patient patient = MhpUserFactory.createFromSecurityContext().getPatient();
			if (patient != null) {
				context.setPatient(patient);
			}
			PatientDemographics demographics = patientDataService.getPatientDemographics(patientIdentifier, ScopeFilter.getInstanceForLongitudinalScope());
			context.setInformation(Domain.demographic, demographics);
			if (context.isVisible(Domain.allergy)) {
				AllergyDataService dataservice = new AllergyDataService();
				Allergies allergies = dataservice.fetchPatientAllergies(patientIdentifier, DateFilterFactory.createEmptyFilter(), ScopeFilter.getInstanceForLongitudinalScope());
				context.setInformation(Domain.allergy, allergies);
			}
			if (context.isVisible(Domain.appointment)) {
				AppointmentDataService dataservice = new AppointmentDataService();
				Appointments appointments = dataservice.getPatientAppointmentsWithDetails(patientIdentifier, context.getDateFilter());
				Appointments sortedAppointments = dataservice.sortByDate(appointments, true);
				context.setInformation(Domain.appointment, sortedAppointments);
			}
			if (context.isVisible(Domain.admission)) {
				AdmissionDataService dataservice = new AdmissionDataService();
				Admissions admissions = dataservice.getPatientAdmissionsWithDetails(patientIdentifier, context.getDateFilter(), ScopeFilter.getInstanceForLongitudinalScope());
				context.setInformation(Domain.admission, admissions);
			}
			if (context.isVisible(Domain.problem)) {

				ProblemDataService dataservice = new ProblemDataService();
				Problems problems = dataservice.fetchProblems(patientIdentifier, DateFilterFactory.createEmptyFilter(), ScopeFilter.getInstanceForLongitudinalScope());
				context.setInformation(Domain.problem, problems);
			}
			if (context.isVisible(Domain.radiologyResult)) {
				RadiologyDataService dataservice = new RadiologyDataService();
				RadiologyResults radiologyResults = dataservice.getRadiologyResults(patientIdentifier, context.getDateFilter(), ScopeFilter.getInstanceForLongitudinalScope());
				context.setInformation(Domain.radiologyResult, radiologyResults);
			}
			if (context.isVisible(Domain.surgery)) {
				SurgeryDataService dataservice = new SurgeryDataService();
				Surgeries surgeries = dataservice.getPatientSurgeries(patientIdentifier, DateFilterFactory.createEmptyFilter(), ScopeFilter.getInstanceForLongitudinalScope());
				context.setInformation(Domain.surgery, surgeries);
			}
			if (context.isVisible(Domain.vital)) {
				VitalEntryFilter filter = VitalsFilterFactory.createFilterFromDateFilter(context.getDateFilter());
				VitalDataService dataservice = new VitalDataService();
				VitalEntries vitalEntries = dataservice.fetchVitalEntries(patientIdentifier, filter, ScopeFilter.getInstanceForLongitudinalScope());
				context.setInformation(Domain.vital, vitalEntries);
			}
			if (context.isVisible(Domain.medication)) {
				MedicationDataService dataservice = new MedicationDataService();
				MedicationFilter filter = MedicationFilterFactory.createEmptyFilter();
				Medications medications = dataservice.getActiveAndRecentlyExpiredPatientMedications(patientIdentifier, filter, ScopeFilter.getInstanceForLongitudinalScope());
				context.setInformation(Domain.medication, medications);
			}
			if (context.isVisible(Domain.medicationHistory)) {
				MedicationDataService dataservice = new MedicationDataService();
				MedicationFilter filter = MedicationFilterFactory.createEmptyFilter();
				filter.setDateFilter(context.getDateFilter());
				Medications medications = dataservice.getPatientMedicationHistory(patientIdentifier, filter, ScopeFilter.getInstanceForLongitudinalScope());
				context.setInformation(Domain.medicationHistory, medications);
			}
			if (context.isVisible(Domain.labResult)) {
				ChemistryLabDataService dataservice = new ChemistryLabDataService();
				LabResults labResults = dataservice.getLabResults(patientIdentifier, context.getDateFilter(), ScopeFilter.getInstanceForLongitudinalScope());

                // MBB-18: Microbiology does not have a proper data source for veterans as of early 2014
//				MicrobiologyLabDataService microDataservice = new MicrobiologyLabDataService();
//				LabResults microLabResults = microDataservice.getLabResultsWithTextResults(patientIdentifier, context.getDateFilter(), ScopeFilter.getInstanceForLongitudinalScope());
//				labResults.addAll(microLabResults);
				
				context.setInformation(Domain.labResult, labResults);
			}
		}
	}
	
	private void setPatientInformation(HttpSession session, PdfGenerationContext context, PatientIdentifier patientIdentifier) {
		Object cache = sessionStateHelper.checkCache(session, "patients");
		Patient patient = (Patient)cache;
		if (patient != null) {
			context.setPatient(patient);
		}
		cache = sessionStateHelper.checkCache(session, Domain.demographic);
		context.setInformation(Domain.demographic, (PatientDemographics)cache);
	}

	private <T> void setDomain(HttpSession session, PdfGenerationContext context, PatientIdentifier patientIdentifier, String domain, T type) {
		if (context.isVisible(domain)) {
			@SuppressWarnings("unchecked")
			T cache = (T)sessionStateHelper.checkCache(session, domain, 15);
			
			if (cache == null) {
				throw new IllegalStateException(domain + " not found in session");
			}

			context.setInformation(domain, cache);
		}
	}
	
	private <T> void setTimeSensitiveDomain(HttpSession session, PdfGenerationContext context, PatientIdentifier patientIdentifier, String domain, T type) {
		if (context.isVisible(domain)) {
			@SuppressWarnings("unchecked")
			T cache = (T)sessionStateHelper.checkCache(session, domain, context.getDateFilter(), 15);
			
			if (cache == null) {
				throw new IllegalStateException(domain + " not found in session");
			}

			context.setInformation(domain, cache);
		}
	}
}
