package com.agilex.healthcare.mbb.pdf.securemessage;

import java.util.Date;
import java.util.Locale;

import org.springframework.context.MessageSource;

import com.agilex.healthcare.mbb.pdf.PageEvent;
import com.agilex.healthcare.mbb.pdf.Stylesheet;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessage;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.Rectangle;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfWriter;

public class SecureMessagePDF {
  private Document document;
  private SecureMessage secureMessage;
  private MessageSource messageSource;
  private Locale locale = Locale.ENGLISH;

  public SecureMessagePDF(Document document, PageEvent event, PdfWriter writer, MessageSource messageSource, SecureMessage secureMessage) {
    this.document = document;
    this.secureMessage = secureMessage;
    this.messageSource = messageSource;
  }

  public void create() throws DocumentException {
    addSenderAndReceiver(secureMessage.getFrom(), secureMessage.getTo());
    addSubject(secureMessage.getSubject());

    if (NullChecker.isNotNullish(secureMessage.getDateSent())) {
      addSentDate(secureMessage.getDateSent());
    }

    addDivider();
    addMessage(secureMessage.getMessageBody());
  }

  private void addSenderAndReceiver(String from, String to) throws DocumentException {
    PdfPTable senderAndReceiverTable = new PdfPTable(2);
    senderAndReceiverTable.setWidthPercentage(100);
    senderAndReceiverTable.setSpacingBefore(0);
    senderAndReceiverTable.setSpacingAfter(10);
    senderAndReceiverTable.setWidths(new int[] { 20, 80 });

    PdfPCell cell;

    cell = new PdfPCell(new Phrase(messageSource.getMessage("mbb.secure-message.from", null, locale), Stylesheet.getNormal()));
    cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
    cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
    cell.setBorderWidth(0);
    senderAndReceiverTable.addCell(cell);

    cell = new PdfPCell(new Phrase(from, Stylesheet.getNormal()));
    cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
    cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
    cell.setBorderWidth(0);
    senderAndReceiverTable.addCell(cell);

    cell = new PdfPCell(new Phrase(messageSource.getMessage("mbb.secure-message.to", null, locale), Stylesheet.getNormal()));
    cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
    cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
    cell.setBorderWidth(0);
    senderAndReceiverTable.addCell(cell);

    cell = new PdfPCell(new Phrase(to, Stylesheet.getNormal()));
    cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
    cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
    cell.setBorderWidth(0);
    senderAndReceiverTable.addCell(cell);

    document.add(senderAndReceiverTable);
  }

  public void addSubject(String title) throws DocumentException {
    PdfPTable heading = new PdfPTable(1);
    heading.setWidthPercentage(100);
    heading.setSpacingBefore(10);
    heading.setSpacingAfter(5);

    PdfPCell cell = new PdfPCell(new Phrase(title, Stylesheet.getH2()));
    cell.setBorderWidth(0);
    heading.addCell(cell);

    document.add(heading);
  }

  private void addSentDate(Date sentDate) throws DocumentException {
    String dateSent = DateHelper.formatDateTime(sentDate);

    PdfPTable sentDateTable = new PdfPTable(1);
    sentDateTable.setWidthPercentage(100);
    sentDateTable.setSpacingBefore(0);
    sentDateTable.setSpacingAfter(0);

    PdfPCell cell;

    cell = new PdfPCell(new Phrase(dateSent, Stylesheet.getNormal()));
    cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
    cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
    cell.setBorderWidth(0);
    sentDateTable.addCell(cell);

    document.add(sentDateTable);
  }

  private void addDivider() throws DocumentException {
    PdfPTable messageTable = new PdfPTable(1);
    messageTable.setWidthPercentage(100);
    messageTable.setSpacingBefore(5);
    messageTable.setSpacingAfter(10);

    PdfPCell cell;

    cell = new PdfPCell();
    cell.setPadding(5);
    cell.setBorder(Rectangle.BOTTOM);
    messageTable.addCell(cell);

    document.add(messageTable);
  }

  private void addMessage(String message) throws DocumentException {
    PdfPTable messageTable = new PdfPTable(1);
    messageTable.setWidthPercentage(100);
    messageTable.setSpacingBefore(0);
    messageTable.setSpacingAfter(10);

    PdfPCell cell;

    cell = new PdfPCell(new Phrase(message, Stylesheet.getNormal()));
    cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
    cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
    cell.setBorderWidth(0);
    messageTable.addCell(cell);

    document.add(messageTable);
  }
}
