package com.agilex.healthcare.mbb.pdf.pages;

import com.agilex.healthcare.mbb.datalayer.PdfGenerationContext;
import com.agilex.healthcare.mbb.pdf.PageEvent;
import com.agilex.healthcare.mbb.pdf.Stylesheet;
import com.agilex.healthcare.mobilehealthplatform.domain.Medication;
import com.agilex.healthcare.mobilehealthplatform.domain.Medications;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.mobilehealthplatform.utils.StringUtil;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.itextpdf.text.Chunk;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Paragraph;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfWriter;

public class MedicationHistoryPage extends AbstractPage {
  //private static final Logger LOGGER = Logger.getLogger(MedicationHistoryPage.class);

  public MedicationHistoryPage(Document document, PageEvent event, PdfWriter writer, PdfGenerationContext context) {
    super(document, event, writer, context);
  }

  public void create() throws DocumentException {
    String domain = Domain.medicationHistory;
    addHeading(getMessage("mbb.report.medicationHistory.title", null), domain);
    addUpdate(getMessage("mbb.report.medicationHistory.updateFrequency", null), domain);
    addTextDateRange(context.getDateFilter());
    addBlankLines(2);

    Medications medications = (Medications)context.getInformation(domain);
    display(medications);
  }

  private void display(Medications medications) throws DocumentException {
    if (NullChecker.isNotNullish(medications)) {
      for (Medication medication : medications) {
        addMedication(document, medication);
      }
    } else {
      addNoDataFound();
    }
  }

  private void addMedication(Document document, Medication medication) throws DocumentException {
    //LOGGER.info(LogUtil.getInfo("medication=", medication));

    final String NAME = StringUtil.blankIfNull(medication.getDrugName());
    final String DESCRIPTION  = StringUtil.blankIfNull(medication.getMedicationDetail());
    final String FULL_NAME    = StringUtil.concatWithDesc(NAME, DESCRIPTION);
    final String STATUS     = StringUtil.blankIfNull(medication.getStatus());
    final String INSTRUCTIONS = StringUtil.blankIfNull(medication.getSig());
    final String QUANTITY   = StringUtil.blankIfNull(medication.getQuantity());
    final String REFILLS    = StringUtil.blankIfNull(StringUtil.blankIfNull(medication.getRefillsRemaining()));
    //final String REFILLS    = StringUtil.blankIfNull(medication.getRefills());
    final String LAST_FILLED  = StringUtil.blankIfNull(DateHelper.formatDate(medication.getLastFilledDate()));
    final String INIT_ORDER   = StringUtil.blankIfNull(DateHelper.formatDate(medication.getStartDate()));
    final String SOURCE     = StringUtil.blankIfNull(medication.getSourceSystem());

    document.add(new Paragraph(new Phrase(FULL_NAME, Stylesheet.getBoldLarge())));
    document.add(new Paragraph(getLabelAndData("mbb.report.medicationHistory.status",STATUS)));
    if (!StringUtil.isEmpty(INSTRUCTIONS)) {
      document.add(new Paragraph(getLabelAndData("mbb.report.medicationHistory.directions",INSTRUCTIONS)));
    }

    PdfPTable table = createMedicationTable();
    PdfPCell cell;
    for (String data : new String[] {QUANTITY, REFILLS, LAST_FILLED, INIT_ORDER, SOURCE}) {
      cell = createCell(new Phrase(data, Stylesheet.getNormal()), 0, false, true, PdfPCell.ALIGN_LEFT);
      table.addCell(cell);
    }
    document.add(table);

  }

  private PdfPTable createMedicationTable() throws DocumentException {
    PdfPTable table = new PdfPTable(5);
    table.setWidthPercentage(100);
    table.setWidths(new int[] { 20, 20, 20, 20, 20 });
    table.setSpacingBefore(10);
    table.setSpacingAfter(10);
    table.setHeaderRows(1);

    PdfPCell cell;
    for (String key : new String[] {"quantity", "refills", "lastFilled", "initialOrder", "source"}) {
      cell = createTableHeader(getMessage("mbb.report.medicationHistory." + key, null), false, PdfPCell.ALIGN_LEFT);
      table.addCell(cell);
    }
    return table;
  }
}
