package com.agilex.healthcare.mbb.pdf.pages;

import java.util.Date;

import com.agilex.healthcare.mbb.datalayer.PdfGenerationContext;
import com.agilex.healthcare.mbb.pdf.PageEvent;
import com.agilex.healthcare.mbb.pdf.Stylesheet;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResult;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResults;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.mobilehealthplatform.utils.StringUtil;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.itextpdf.text.Chunk;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Paragraph;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfWriter;

public class LabPage extends AbstractPage {

  public LabPage(Document document, PageEvent event, PdfWriter writer, PdfGenerationContext context) {
    super(document, event, writer, context);
  }

  public void create() throws DocumentException {
    String domain = Domain.labResult;
    addHeading(context.getMessageSource().getMessage("mbb.report.lab.title", null, context.getLocale()), domain);
    addUpdate(context.getMessageSource().getMessage("mbb.report.lab.updateFrequency", null, context.getLocale()), domain);
    addTextDateRange(context.getDateFilter());

    LabResults results = (LabResults)context.getInformation(domain);

    if (NullChecker.isNotNullish(results)) {
      int i = 0;
      boolean lastRow = false;
      for (LabResult result : results) {
        if (result.getValue().contains("MICROBIOLOGY")) {
          addMicrobiologyLabResult(document, result, i, lastRow);
        } else {
          addChemistryLabResult(document, result, i, lastRow);
        }
      }
    } else {
      addNoDataFound();
    }
  }

  private PdfPTable createChemistryTable() throws DocumentException {
    PdfPTable table = new PdfPTable(2);
    table.setWidthPercentage(100);
    table.setWidths(new int[] { 40, 60 });
    table.setSpacingBefore(10);
    table.setSpacingAfter(10);
    table.setHeaderRows(1);

    PdfPCell cell;
    for (String key : new String[] {"result", "refRange"}) {
      cell = createTableHeader(getMessage("mbb.report.lab." + key, null), false, PdfPCell.ALIGN_LEFT);
      table.addCell(cell);
    }
    return table;
  }

  private void addChemistryLabResult(Document document, LabResult result, int rowNum, boolean lastRow) throws DocumentException {
    final String DATE         = DateHelper.format(result.getResultedDate(), DateHelper.REPORT_DETAILED_DATE_FORMAT_NO_TIME);
    final String TEST         = StringUtil.blankIfNull(result.getTestname());
    final String SPECIMEN     = StringUtil.blankIfNull(result.getSpecimenName());
    final String RESULT       = String.format("%s %s", StringUtil.blankIfNull(result.getValue()), StringUtil.blankIfNull(result.getValueUnits()));
    final String REFERENCE_HIGH   = StringUtil.blankIfNull(result.getReferenceHigh());
    final String REFERENCE_LOW    = StringUtil.blankIfNull(result.getReferenceLow());
    final String FACILITY     = StringUtil.blankIfNull(result.getSourceSystem());

    String refRange = generateRefRange(REFERENCE_LOW, REFERENCE_HIGH);

    document.add(new Paragraph(new Phrase(String.format("%s (%s)", TEST, SPECIMEN), Stylesheet.getBoldLarge())));
    document.add(new Paragraph(getLabelAndData("mbb.report.lab.resultedDate",DATE)));
    document.add(new Paragraph(getLabelAndData("mbb.report.lab.facility",FACILITY)));

    PdfPTable table = createChemistryTable();
    PdfPCell cell;
    for (String data : new String[] {RESULT, refRange}) {
      cell = createCell(new Phrase(data, Stylesheet.getNormal()), 0, false, lastRow, PdfPCell.ALIGN_LEFT);
      table.addCell(cell);
    }
    document.add(table);
  }

  private void addMicrobiologyLabResult(Document document, LabResult result, int rowNum, boolean lastRow) throws DocumentException {
    final String DATE   = DateHelper.format(result.getResultedDate(), DateHelper.REPORT_DETAILED_DATE_FORMAT_NO_TIME);
    final String TEST   = StringUtil.blankIfNull(result.getTestname());
    final String SPECIMEN = StringUtil.blankIfNull(result.getSpecimenName());
    final String RESULT   = String.format("%s %s", StringUtil.blankIfNull(result.getValue()), StringUtil.blankIfNull(result.getValueUnits()));
    final String FACILITY = StringUtil.blankIfNull("no data found");

    document.newPage();
    document.add(new Paragraph(new Phrase(String.format("%s (%s)", TEST, SPECIMEN), Stylesheet.getBoldLarge())));
    document.add(new Paragraph(getLabelAndData("mbb.report.lab.resultedDate",DATE)));
    document.add(new Paragraph(getLabelAndData("mbb.report.lab.facility",FACILITY)));

    addBlankLines(1);
    document.add(new Paragraph(new Phrase(getMessage("mbb.report.micro.additionalnfo",null), Stylesheet.getBoldLarge())));
    document.add(new Paragraph(new Phrase(RESULT, Stylesheet.getNormal())));

  }

  private String generateRefRange(String referenceLow, String referenceHigh) {
    String refRange;
    if (NullChecker.isNotNullish(referenceLow) && NullChecker.isNotNullish(referenceHigh)) {
      refRange = String.format("%s - %s", referenceLow, referenceHigh);
    } else if (NullChecker.isNotNullish(referenceLow)) {
      refRange = referenceLow;
    } else if (NullChecker.isNotNullish(referenceHigh)) {
      refRange = referenceHigh;
    } else {
      refRange = "--";
    }
    return refRange;
  }
}
