package com.agilex.healthcare.mbb.pdf.pages;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import com.agilex.healthcare.mbb.datalayer.PdfGenerationContext;
import com.agilex.healthcare.mbb.pdf.PageEvent;
import com.agilex.healthcare.mobilehealthplatform.domain.HealthAdvocateRequest;
import com.agilex.healthcare.mobilehealthplatform.domain.HealthAdvocateStatusCode;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.StreamHelper;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.pdf.AcroFields;
import com.itextpdf.text.pdf.BaseFont;
import com.itextpdf.text.pdf.PdfContentByte;
import com.itextpdf.text.pdf.PdfImportedPage;
import com.itextpdf.text.pdf.PdfReader;
import com.itextpdf.text.pdf.PdfStamper;
import com.itextpdf.text.pdf.PdfWriter;

public class HealthAdvocateRequestPage extends AbstractPage {
    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(HealthAdvocateRequestPage.class);
	private static final String BASE_LOCATION = "com/agilex/healthcare/mobilehealthplatform/pdf/";

	private OutputStream outputStream;

	public HealthAdvocateRequestPage(Document document, PageEvent event, PdfWriter writer, PdfGenerationContext context, OutputStream outputStream){
		super(document, event, writer, context);
		this.outputStream = outputStream;
	}

	@Override
	public void create() throws DocumentException {
		String domain = Domain.healthAdvocateRequest;

		HealthAdvocateRequest healthAdvocateRequest = (HealthAdvocateRequest)context.getInformation(domain);

			PdfStamper stamper;
			try {
				stamper = createStamper(outputStream);

		        fill(stamper.getAcroFields(), healthAdvocateRequest);

		        stamper.setFormFlattening(true);
		        stamper.close();

			} catch (FileNotFoundException e) {
                logger.error("Creation of HealthAdvocate Request Form Failed", e);
				e.printStackTrace();
			} catch (IOException e) {
                logger.error("Creation of HealthAdvocate Request Form Failed", e);
			}
	}

	public static void concatPDFs(List<PdfReader> readers,
            OutputStream outputStream, boolean paginate) {

        Document document = new Document();
        try {
            int totalPages = 2;

            PdfWriter writer = PdfWriter.getInstance(document, outputStream);

            document.open();
            BaseFont bf = BaseFont.createFont(BaseFont.HELVETICA,
                    BaseFont.CP1252, BaseFont.NOT_EMBEDDED);
            PdfContentByte cb = writer.getDirectContent(); // Holds the PDF
            // data

            PdfImportedPage page;
            int currentPageNumber = 0;
            int pageOfCurrentReaderPDF = 0;
            Iterator<PdfReader> iteratorPDFReader = readers.iterator();

            // Loop through the PDF files and add to the output.
            while (iteratorPDFReader.hasNext()) {
                PdfReader pdfReader = iteratorPDFReader.next();

                // Create a new page in the target for each source page.
                while (pageOfCurrentReaderPDF < pdfReader.getNumberOfPages()) {
                    document.newPage();
                    pageOfCurrentReaderPDF++;
                    currentPageNumber++;
                    page = writer.getImportedPage(pdfReader,
                            pageOfCurrentReaderPDF);
                    cb.addTemplate(page, 0, 0);

                    // Code for pagination.
                    if (paginate) {
                        cb.beginText();
                        cb.setFontAndSize(bf, 9);
                        cb.showTextAligned(PdfContentByte.ALIGN_CENTER, ""
                                + currentPageNumber + " of " + totalPages, 520,
                                5, 0);
                        cb.endText();
                    }
                }
                pageOfCurrentReaderPDF = 0;
            }
            outputStream.flush();
            document.close();
            outputStream.close();
        } catch (Exception e) {
            logger.debug("Creation of HealthAdvocate Request Failed during concatenation", e);
        } finally {
            if (document.isOpen())
                document.close();
            try {
                if (outputStream != null)
                    outputStream.close();
            } catch (IOException ioe) {
                logger.debug("Creation of HealthAdvocate Request Failed during concatenation", ioe);
            }
        }
	}



	//Create the stamper using the existing pdf as a template
	private PdfStamper createStamper(OutputStream outputStream) throws IOException, DocumentException,
		FileNotFoundException {

		PdfReader reader = new PdfReader(BASE_LOCATION + "AdvocateAuthorization.pdf");
		PdfStamper stamper = new PdfStamper(reader,outputStream);

		return stamper;
	}





	private static String formatStringValue(String input){
		String value = input;

		if (null == value){
			value = "";
		}
		return value;
	}

	private static String formatDateValue(Date input){
		String value;

		if (null != input){
			value = String.format("%1$tD", input);
		}
		else{
			value = "";
		}
		return value;

	}

	//use known field names to populate from the existing info
	 private static void fill(AcroFields form, HealthAdvocateRequest domainInformation)
		        throws IOException, DocumentException {

		if (null != domainInformation){
			String advocateLastName = formatStringValue(domainInformation.getLastName());
			String advocateFirstName = formatStringValue(domainInformation.getFirstName());
			String advocateEmail = formatStringValue(domainInformation.getEmail());
			String patientName = formatStringValue(domainInformation.getPatientName());
			String patientDisplayName = formatStringValue(domainInformation.getPatientDisplayName());
			String advocateAddress1 = formatStringValue(domainInformation.getAddressFirstLine());
			String advocateAddress2 = formatStringValue(domainInformation.getAddressSecondLine());
			String ssn = formatStringValue(domainInformation.getSsn());

			String city = formatStringValue(domainInformation.getCity());
			String state = formatStringValue(domainInformation.getState());
			String zipCode = formatStringValue(domainInformation.getZipCode());
			String advocateCityStateZip = String.format("%1s, %2s %3s", city, state, zipCode);

			String formattedRequestDate = formatDateValue(domainInformation.getRequestDate());


			String line1 = String.format("%1$s %2$s   %3$s", advocateFirstName, advocateLastName, advocateEmail);
			String line2;
			if (advocateAddress2.isEmpty()){
				line2 = String.format("%1$s / %2$s", advocateAddress1, advocateCityStateZip);
			}
			else{
				line2 = String.format("%1$s / %2$s / %3$s", advocateAddress1, advocateAddress2, advocateCityStateZip);
			}

			String releasedText = String.format("%1$s%n%2$s", line1, line2);

//			form.setField("vha105345form[0].#subform[0].va_infoa[0]",  "va-infoa");
			form.setField("vha105345form[0].#subform[0].patient_name[0]", 	patientDisplayName);
			form.setField("vha105345form[0].#subform[0].ssn[0]",  ssn);

			form.setField("vha105345form[0].#subform[0].organization[0]",  releasedText);
//
//			form.setField("vha105345form[0].#subform[0].drug_abuse[0]",  "YES");
//			form.setField("vha105345form[0].#subform[0].alcoholism[0]",  "YES");
//			form.setField("vha105345form[0].#subform[0].hiv_abuse[0]",  "YES");
//			form.setField("vha105345form[0].#subform[0].sicle_cell[0]",  "YES");

			form.setField("vha105345form[0].#subform[0].hospital_summary[0]",  "Health Advocate System");
//			form.setField("vha105345form[0].#subform[0].treatment_notes[0]",  "treatment_notes");
//			form.setField("vha105345form[0].#subform[0].other[0]", "YES" ); //without the explicit set, this field is being checked


//			form.setField("vha105345form[0].#subform[0].specification[0]",  "specification");
			String driversLicenseNumber = domainInformation.getDriversLicenseNumber();
			String driversLicenseState = domainInformation.getDriversLicenseState();
			String purpose = String.format("To authorize %1$s %2$s to act as my VA family caregiver for online services accessed via DS Logon.%n%3$s %4$s, Health Advocate's Driver's License: %5$s, State: %6$s", advocateFirstName, advocateLastName, advocateFirstName, advocateLastName, driversLicenseNumber, driversLicenseState);
			form.setField("vha105345form[0].#subform[0].purpose[0]", purpose);

//			form.setField("vha105345form[0].#subform[0].expiration_date[0]",  "N/A");
			String status = domainInformation.getStatus();
			String conditions = "";
			if ( HealthAdvocateStatusCode.CANCELLED.equals(status) || HealthAdvocateStatusCode.REVOKED.equals(status)){
				conditions = String.format("Request revocation of this request originally made by me on %1$s after processing by VBA.", DateHelper.formatDate(domainInformation.getRequestDate()));
				form.setField("vha105345form[0].#subform[0].conditions[0]",  conditions);
			}
			else{
				conditions = "This authorization covers diagnosis and related health information I may have upon signing the Authorization and those I may acquire in the future, including those protected by Title 38 U.S.C 7332. This Authorization will expire upon processing of the revocation request.";
				form.setField("vha105345form[0].#subform[0].conditions[0]",  conditions);
			}

			form.setField("vha105345form[0].#subform[0].signature_date[0]",  formattedRequestDate);

//			form.setField("vha105345form[0].#subform[0].textfield[3]",  "textfield[3]");
//			form.setField("vha105345form[0].#subform[0].textfield[0]",  "textfield[0]");
//			form.setField("vha105345form[0].#subform[0].textfield[1]",  "textfield[1]");
//			form.setField("vha105345form[0].#subform[0].textfield[2]",  "textfield[2]");

			form.setField("vha105345form[0].#subform[0].signature[0]",  String.format("%1$s /ES-DS Logon/", patientName));

		}


    }
}
