package com.agilex.healthcare.mbb.pdf.pages;

import java.util.Date;

import com.agilex.healthcare.mbb.datalayer.PdfGenerationContext;
import com.agilex.healthcare.mbb.pdf.PageEvent;
import com.agilex.healthcare.mbb.pdf.Stylesheet;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientDemographics;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import java.util.Date;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.mobilehealthplatform.domain.DOBDate;
import com.agilex.healthcare.utility.NullChecker;
import com.itextpdf.text.Chunk;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfWriter;
import com.itextpdf.text.pdf.PdfDiv;

public class DemographicPage extends AbstractPage {
    
  private final float KEYW = 0.15f;
  private final float VALW = 0.85f;

  public DemographicPage(Document document, PageEvent event, PdfWriter writer, PdfGenerationContext context) {
    super(document, event, writer, context);
  }

  public void create() throws DocumentException {
    addHeading(context.getMessageSource().getMessage("mbb.report.demographics.title", null, context.getLocale()), Domain.demographic);

    addVeteranIdentification(context);
    addNextOfKin(context);
  }

  private void addVeteranIdentification(PdfGenerationContext context) throws DocumentException {
    PatientDemographics demographics = (PatientDemographics)context.getInformation(Domain.demographic);
    final String LAST_NAME = context.getPatient().getLastName();
    final String FIRST_NAME = context.getPatient().getFirstName();
    final String GENDER = context.getPatient().getGender();
    final Date DATE_OF_BIRTH = context.getPatient().getDateOfBirth().getDate();
    final String STREET_ADDRESS_1 = getReportValue(demographics.getAddress().getStreetAddressLine1());
    final String STREET_ADDRESS_2 = demographics.getAddress().getStreetAddressLine2();
    final String STREET_ADDRESS_3 = demographics.getAddress().getStreetAddressLine3();
    final String ROOM_BED = context.getPatient().getRoombed();
    final String WARD_LOCATION = context.getPatient().getWardLocation();
    final String CITY = demographics.getAddress().getCity();
    final String STATE = demographics.getAddress().getState();
    final String ZIP_CODE = demographics.getAddress().getZipCode();
    final String HOME = getReportValue(demographics.getPhoneNumberHome());
    final String CELL = getReportValue(demographics.getPhoneNumberMobile());
    final String WORK = getReportValue(demographics.getPhoneNumberWork());
    final String PAGER = getReportValue(demographics.getPhoneNumberPager());
    final String EMAIL = getReportValue(demographics.getEmailAddress());
    
    final String LOCATION;
    if (context.getPatient().isInpatient()) {
      if (NullChecker.isNotNullish(WARD_LOCATION) && NullChecker.isNotNullish(ROOM_BED)) {
        LOCATION = String.format("%s, %s", WARD_LOCATION, ROOM_BED);
      } else if (NullChecker.isNotNullish(WARD_LOCATION)) {
        LOCATION = WARD_LOCATION;
      } else if (NullChecker.isNotNullish(ROOM_BED)) {
        LOCATION = ROOM_BED;
      } else {
        LOCATION = getReportValue(null);
      }
    } else {
      LOCATION = getReportLocationValue(null);
    }
    
    addInfoRow(context.getMessageSource().getMessage("mbb.report.demographics.veteran", null, context.getLocale()), String.format("%s, %s", LAST_NAME, FIRST_NAME));
    
    addInfoRow(context.getMessageSource().getMessage("mbb.report.demographics.gender", null, context.getLocale()), GENDER);
    
    addInfoRow(context.getMessageSource().getMessage("mbb.report.demographics.dob", null, context.getLocale()), DateHelper.format(DATE_OF_BIRTH, DateHelper.REPORT_DETAILED_DATE_FORMAT_NO_TIME));

    Phrase phrase = new Phrase();
    phrase.add(new Chunk(STREET_ADDRESS_1, Stylesheet.getNormal()));
    if (NullChecker.isNotNullish(STREET_ADDRESS_2)) {
      phrase.add(Chunk.NEWLINE);
      phrase.add(new Chunk(STREET_ADDRESS_2, Stylesheet.getNormal()));
    }
    if (NullChecker.isNotNullish(STREET_ADDRESS_3)) {
      phrase.add(Chunk.NEWLINE);
      phrase.add(new Chunk(STREET_ADDRESS_3, Stylesheet.getNormal()));
    }
    phrase.add(Chunk.NEWLINE);

    StringBuilder sb = new StringBuilder();
    if (NullChecker.isNotNullish(CITY) || NullChecker.isNotNullish(STATE) || NullChecker.isNotNullish(ZIP_CODE)) {
      if (NullChecker.isNotNullish(CITY)) {
        sb.append(CITY);
        if (NullChecker.isNotNullish(STATE)) {
          sb.append(", ");
        } else {
          sb.append(" ");
        }
      }
      sb.append(NullChecker.isNotNullish(STATE) ? STATE + " " : "");
      sb.append(NullChecker.isNotNullish(ZIP_CODE) ? ZIP_CODE : "");
    } else {
      sb.append(getReportValue(null));
    }

    phrase.add(new Chunk(sb.toString(), Stylesheet.getNormal()));
    
    addInfoRow(context.getMessageSource().getMessage("mbb.report.demographics.address", null, context.getLocale()), phrase);
    
    addInfoRow(context.getMessageSource().getMessage("mbb.report.demographics.work", null, context.getLocale()), WORK);
    
    addInfoRow(context.getMessageSource().getMessage("mbb.report.demographics.home", null, context.getLocale()), HOME);
    
    addInfoRow(context.getMessageSource().getMessage("mbb.report.demographics.cell", null, context.getLocale()), CELL);
    
    addInfoRow(context.getMessageSource().getMessage("mbb.report.demographics.pager", null, context.getLocale()), PAGER);
    
    addInfoRow(context.getMessageSource().getMessage("mbb.report.demographics.email", null, context.getLocale()), EMAIL);
    
    addInfoRow(context.getMessageSource().getMessage("mbb.report.demographics.location", null, context.getLocale()), LOCATION);
  }

  private String getReportValue(String value) {
    return NullChecker.isNullish(value) ? context.getMessageSource().getMessage("mbb.report.noDataFound", null, context.getLocale()) : value;
  }

  private String getReportLocationValue(String value) {
    return NullChecker.isNullish(value) ? context.getMessageSource().getMessage("mbb.report.demographics.notCurrentlyAdmitted", null, context.getLocale()) : value;
  }

  private void addNextOfKin(PdfGenerationContext context) throws DocumentException {
    PatientDemographics demographics = (PatientDemographics)context.getInformation(Domain.demographic);

        final String noDataFound = context.getMessageSource().getMessage("mbb.report.noDataFound", null, context.getLocale());

        String NAME = noDataFound;
        String HOME_PHONE = noDataFound;
        String RELATIONSHIP = noDataFound;

    if(demographics != null && demographics.getNextOfKin() != null ) {
            if(NullChecker.isNotNullish(demographics.getNextOfKin().getName())) NAME = demographics.getNextOfKin().getName();
            if(NullChecker.isNotNullish(demographics.getNextOfKin().getPhoneNumber())) HOME_PHONE = demographics.getNextOfKin().getPhoneNumber();
            if(NullChecker.isNotNullish(demographics.getNextOfKin().getRelationship())) RELATIONSHIP = demographics.getNextOfKin().getRelationship();
        }
    
    PdfDiv nextOfKinDiv = new PdfDiv();
    nextOfKinDiv.setPaddingTop(10);
    nextOfKinDiv.addElement(new Chunk(context.getMessageSource().getMessage("mbb.report.demographics.nextOfKin", null, context.getLocale()), Stylesheet.getBoldLarge()));
    document.add(nextOfKinDiv);    
        
    addInfoRow(context.getMessageSource().getMessage("mbb.report.demographics.nextOfKin.name", null, context.getLocale()), NAME);
    
    addInfoRow(context.getMessageSource().getMessage("mbb.report.demographics.home", null, context.getLocale()), HOME_PHONE);

    addInfoRow(context.getMessageSource().getMessage("mbb.report.demographics.nextOfKin.relationship", null, context.getLocale()), RELATIONSHIP);
  }
  
  private PdfDiv getStyleDiv(String text, boolean isBold, float percentageWidth) {
      Chunk chunk = new Chunk(text, isBold ? Stylesheet.getBold() : Stylesheet.getNormal());
      PdfDiv div = new PdfDiv();
      
      div.setFloatType(PdfDiv.FloatType.LEFT);
      div.setPercentageWidth(percentageWidth);
      div.addElement(chunk);
      
      return div;
  }
  
  private PdfDiv getStyleDiv(Phrase phrase, float percentageWidth) {
      PdfDiv div = new PdfDiv();
      
      div.setFloatType(PdfDiv.FloatType.LEFT);
      div.setPercentageWidth(percentageWidth);
      div.addElement(phrase);
      
      return div;
  }
  
  private void addInfoRow(String key, String value) throws DocumentException {
    PdfDiv row = new PdfDiv();
    PdfDiv section = getStyleDiv(key, true, KEYW);
    row.addElement(section);

    section = getStyleDiv(value, false, VALW);
    row.addElement(section);

    document.add(row);
  }
  
  private void addInfoRow(String key, Phrase value) throws DocumentException {
    PdfDiv row = new PdfDiv();
    PdfDiv section = getStyleDiv(key, true, KEYW);
    row.addElement(section);

    section = getStyleDiv(value, VALW);
    row.addElement(section);

    document.add(row);      
  }
}
