package com.agilex.healthcare.mbb.pdf.pages;

import java.awt.Color;
import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.io.IOUtils;

import com.agilex.healthcare.mbb.datalayer.PdfGenerationContext;
import com.agilex.healthcare.mbb.pdf.PageEvent;
import com.agilex.healthcare.mbb.pdf.PdfRoundCorners;
import com.agilex.healthcare.mbb.pdf.Stylesheet;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.utility.*;
import com.itextpdf.text.Anchor;
import com.itextpdf.text.BaseColor;
import com.itextpdf.text.Chunk;
import com.itextpdf.text.List;
import com.itextpdf.text.ListItem;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Image;
import com.itextpdf.text.Paragraph;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.pdf.PdfDiv;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfWriter;
import com.itextpdf.text.pdf.PdfAction;

import java.util.Date;

import com.agilex.healthcare.mobilehealthplatform.domain.DOBDate;

public class CoverPage extends AbstractPage {
  private static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(CoverPage.class);

  public CoverPage(Document document, PageEvent event, PdfWriter writer, PdfGenerationContext context) {
    super(document, event, writer, context);
  }

  public void create() throws DocumentException {
    addNote();
    addReportInformation(context);

    addLogo();
    document.newPage();
    addHeading(context.getMessageSource().getMessage("mbb.report.cover.toc.title", null, context.getLocale()), "toc");
    addTableOfContents(context);
  }

  private void addNote() throws DocumentException {
    Color noteColor = new Color(0x54, 0x69, 0x93);
	BaseColor baseNoteColor = new BaseColor(noteColor.getRed(), noteColor.getGreen(), noteColor.getBlue());
    
    PdfPTable table = new PdfPTable(1);
    table.setWidthPercentage(100);
    table.setSpacingBefore(20);
    table.setSpacingAfter(20);
    table.setHeaderRows(1);

    List unorderedList = new List(false, 20);
    unorderedList.add(new ListItem(context.getMessageSource().getMessage("mbb.report.cover.generalPointOne", null, context.getLocale()), Stylesheet.getNormal()));
    unorderedList.add(new ListItem(context.getMessageSource().getMessage("mbb.report.cover.generalPointTwo", null, context.getLocale()), Stylesheet.getNormal()));
    unorderedList.add(new ListItem(context.getMessageSource().getMessage("mbb.report.cover.generalPointThree", null, context.getLocale()), Stylesheet.getNormal()));

    PdfPCell cell = new PdfPCell(new Phrase(context.getMessageSource().getMessage("mbb.report.title", null, context.getLocale()), Stylesheet.getH1()));
    
    cell.setBorderColor(baseNoteColor);
    cell.setBorderWidth(0);
    cell.setPaddingTop(5);
    cell.setPaddingBottom(5);
    cell.setPaddingLeft(15);
    cell.setPaddingRight(15);
    cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
    cell.setBorderWidthBottom(0);
    cell.setCellEvent(new PdfRoundCorners(noteColor, true, true, false, false));
    
    table.addCell(cell);

    Paragraph note = new Paragraph();
    note.add(new Paragraph(context.getMessageSource().getMessage("mbb.report.cover.confidential", null, context.getLocale()), Stylesheet.getBold()));
    note.add(Chunk.NEWLINE);
    note.add(new Paragraph(context.getMessageSource().getMessage("mbb.report.cover.producedBy", null, context.getLocale()), Stylesheet.getNormal()));
    note.add(Chunk.NEWLINE);
    note.add(new Paragraph(context.getMessageSource().getMessage("mbb.report.cover.summary", null, context.getLocale()), Stylesheet.getNormal()));
    note.add(Chunk.NEWLINE);
    note.add(new Paragraph(context.getMessageSource().getMessage("mbb.report.cover.generalTitle", null, context.getLocale()), Stylesheet.getNormal()));

    Paragraph bottomNote = new Paragraph();
    bottomNote.add(Chunk.NEWLINE);
    bottomNote.add(new Paragraph(context.getMessageSource().getMessage("mbb.report.cover.questions", null, context.getLocale()), Stylesheet.getNormal()));

    cell = new PdfPCell();
    cell.addElement(note);
    cell.addElement(unorderedList);
    cell.addElement(bottomNote);
    cell.setBorderWidth(2);
    cell.setBorderColor(baseNoteColor);
    cell.setPaddingTop(5);
    cell.setPaddingBottom(15);
    cell.setPaddingLeft(15);
    cell.setPaddingRight(15);
    cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
    table.addCell(cell);

    document.add(table);
  }

  private void addReportInformation(PdfGenerationContext context) throws DocumentException {
    final String FIRST_NAME = context.getPatient().getFirstName();
    final String LAST_NAME = context.getPatient().getLastName();
    String REPORT_DATE = DateHelper.formatDateTimeInReportDetailedFormat(context.getReportDate());
    REPORT_DATE = REPORT_DATE.replaceFirst("@", "at");

    final Date temporaryBirthDate = context.getPatient().getDateOfBirth().getDate();

    String birthDate = "";
    if(temporaryBirthDate != null){
      birthDate = DateHelper.format(temporaryBirthDate, DateHelper.REPORT_DETAILED_DATE_FORMAT_NO_TIME);
    }

    final String DATE_OF_BIRTH = birthDate;

    document.add(new Paragraph(new Phrase(context.getMessageSource().getMessage("mbb.report.cover.generatedFor", null, context.getLocale()), Stylesheet.getNormal())));

    PdfDiv row = new PdfDiv();
    PdfDiv name = new PdfDiv();
    name.setFloatType(PdfDiv.FloatType.LEFT);
    name.setPercentageWidth(0.4f);

    name.addElement(new Phrase(String.format("%s %s", FIRST_NAME, LAST_NAME), Stylesheet.getBold()));
    row.addElement(name);

    Phrase phrase = new Phrase();
    phrase.add(new Chunk(context.getMessageSource().getMessage("mbb.report.cover.dateOfBirth", null, context.getLocale()), Stylesheet.getNormal()));
    phrase.add(new Chunk(" "));
    phrase.add(new Chunk(DATE_OF_BIRTH, Stylesheet.getBold()));
    row.addElement(phrase);
    document.add(row);

    phrase = new Phrase();
    phrase.add(new Chunk(context.getMessageSource().getMessage("mbb.report.cover.on", null, context.getLocale()), Stylesheet.getNormal()));
    phrase.add(new Chunk(" "));
    phrase.add(new Chunk(REPORT_DATE, Stylesheet.getBold()));

    document.add(new Paragraph(phrase));
  }

  private void addTableOfContents(PdfGenerationContext context) throws DocumentException {
    if (includeCurrentActiveInformation(context) || includeTimeLimitedInformation(context)) {
        PdfDiv section;
        
        Paragraph title;

      if (includeCurrentActiveInformation(context)) {
        title = new Paragraph(new Phrase(context.getMessageSource().getMessage("mbb.report.cover.toc.active", null, context.getLocale()), Stylesheet.getBold()));
        document.add(title);
        
        section = new PdfDiv();

        if (context.isVisible(Domain.demographic)) {
          section.addElement(createLinkedTitle(context.getMessageSource().getMessage("mbb.report.demographics.title", null, context.getLocale()), Domain.demographic));
        }
        if (context.isVisible(Domain.problem)) {
          section.addElement(createLinkedTitle(context.getMessageSource().getMessage("mbb.report.problems.title", null, context.getLocale()), Domain.problem));
        }
        if (context.isVisible(Domain.allergy)) {
          section.addElement(createLinkedTitle(context.getMessageSource().getMessage("mbb.report.allergies.title", null, context.getLocale()), Domain.allergy));
        }
        if (context.isVisible(Domain.medication)) {
          section.addElement(createLinkedTitle(context.getMessageSource().getMessage("mbb.report.medications.title", null, context.getLocale()), Domain.medication));
        }
        if (context.isVisible(Domain.surgery)) {
          section.addElement(createLinkedTitle(context.getMessageSource().getMessage("mbb.report.surgeries.title", null, context.getLocale()), Domain.surgery));
        }
        document.add(section);
      }

      if (includeTimeLimitedInformation(context)) {
        title = new Paragraph(new Phrase(context.getMessageSource().getMessage("mbb.report.cover.toc.limited", null, context.getLocale()), Stylesheet.getBold()));
        document.add(title);

        section = new PdfDiv();
        
        if (context.isVisible(Domain.vital)) {
          section.addElement(createLinkedTitle(context.getMessageSource().getMessage("mbb.report.vitals.title", null, context.getLocale()), Domain.vital));
        }
        if (context.isVisible(Domain.medicationHistory)) {
          section.addElement(createLinkedTitle(context.getMessageSource().getMessage("mbb.report.medicationHistory.title", null, context.getLocale()), Domain.medicationHistory));
        }
        if (context.isVisible(Domain.labResult)) {
          section.addElement(createLinkedTitle(context.getMessageSource().getMessage("mbb.report.lab.title", null, context.getLocale()), Domain.chemLabResult));
        }
        if (context.isVisible(Domain.radiologyResult)) {
          section.addElement(createLinkedTitle(context.getMessageSource().getMessage("mbb.report.radiology.title", null, context.getLocale()), Domain.radiologyResult));
        }
        if (context.isVisible(Domain.appointment)) {
          section.addElement(createLinkedTitle(context.getMessageSource().getMessage("mbb.report.appointments.title", null, context.getLocale()), Domain.appointment));
        }
        if (context.isVisible(Domain.admission)) {
          section.addElement(createLinkedTitle(context.getMessageSource().getMessage("mbb.report.admissions.title", null, context.getLocale()), Domain.admission));
        }
        document.add(section);
      }
    } else {
      addNoDataFound();
    }
  }

  private void addLogo() throws DocumentException {
    InputStream inputStream = null;
    try {
      inputStream = PageEvent.class.getClassLoader().getResourceAsStream("com/agilex/healthcare/mobilehealthplatform/pdf/mbb.png");
      Image image = Image.getInstance(IOUtils.toByteArray(inputStream));
      image.setAlt("VA My Mobile Blue Button");
      image.setCompressionLevel(0);
      image.scalePercent(25f);
      image.setAbsolutePosition(280, 700);
      document.add(image);
    } catch (IOException e) {
      LOGGER.error(context.getMessageSource().getMessage("mbb.report.cover.logo.error", null, context.getLocale()), e);
    } catch (NullPointerException npe) {
            LOGGER.error(context.getMessageSource().getMessage("mbb.report.cover.logo.error", null, context.getLocale()), npe);
        } finally {
      StreamHelper.safeClose(inputStream);
    }
  }

  private boolean includeCurrentActiveInformation(PdfGenerationContext context) {
    boolean result = false;
    if (context.isVisible(Domain.demographic) || context.isVisible(Domain.medication) || context.isVisible(Domain.allergy) || context.isVisible(Domain.problem) || context.isVisible(Domain.surgery)) {
      result = true;
    }
    return result;
  }

  private boolean includeTimeLimitedInformation(PdfGenerationContext context) {
    boolean result = false;
    if (context.isVisible(Domain.vital) || context.isVisible(Domain.medicationHistory) || context.isVisible(Domain.labResult) || context.isVisible(Domain.radiologyResult) || context.isVisible(Domain.appointment) || context.isVisible(Domain.admission)) {
      result = true;
    }
    return result;
  }

  private Paragraph createLinkedTitle(String value, String anchorName) {
    Chunk anchor = new Chunk(value, Stylesheet.getBold());
    anchor.setAction(PdfAction.gotoLocalPage(anchorName, false));
    Paragraph result = new Paragraph();
    result.setIndentationLeft(20.0f);
    result.add(anchor);

    return result;
  }
}
