package com.agilex.healthcare.mbb.pdf.pages;

import java.util.Date;

import com.agilex.healthcare.mbb.datalayer.PdfGenerationContext;
import com.agilex.healthcare.mbb.pdf.PageEvent;
import com.agilex.healthcare.mbb.pdf.Stylesheet;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResult;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResults;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfWriter;

public class ChemistryHematologyLabPage extends AbstractPage {

  public ChemistryHematologyLabPage(Document document, PageEvent event, PdfWriter writer, PdfGenerationContext context) {
    super(document, event, writer, context);
  }

  public void create() throws DocumentException {
    String domain = Domain.chemLabResult;
    addHeading(context.getMessageSource().getMessage("mbb.report.chemistry.title", null, context.getLocale()), domain);
    addTextDateRange(context.getDateFilter());

    PdfPTable table = createTable();

    LabResults results = (LabResults)context.getInformation(domain);

    if (NullChecker.isNotNullish(results)) {
      int i = 0;
      for (LabResult result : results) {
        boolean lastRow = (i == (results.size() - 1));
        addLabResult(table, result, i++, lastRow);
      }
    } else {
      addNoDataFound();
    }

    document.add(table);
  }

  private PdfPTable createTable() throws DocumentException {
    PdfPTable table = new PdfPTable(4);
    table.setWidthPercentage(100);
    table.setWidths(new int[] { 20, 30, 25, 25 });
    table.setSpacingBefore(10);
    table.setSpacingAfter(10);
    table.setHeaderRows(1);

    PdfPCell cell;
    cell = createTableHeader(context.getMessageSource().getMessage("mbb.report.chemistry.date", null, context.getLocale()), false, PdfPCell.ALIGN_LEFT);
    table.addCell(cell);
    cell = createTableHeader(context.getMessageSource().getMessage("mbb.report.chemistry.test", null, context.getLocale()), false, PdfPCell.ALIGN_LEFT);
    table.addCell(cell);
    cell = createTableHeader(context.getMessageSource().getMessage("mbb.report.chemistry.result", null, context.getLocale()), false, PdfPCell.ALIGN_LEFT);
    table.addCell(cell);
    cell = createTableHeader(context.getMessageSource().getMessage("mbb.report.chemistry.refRange", null, context.getLocale()), true, PdfPCell.ALIGN_LEFT);
    table.addCell(cell);

    return table;
  }

  private void addLabResult(PdfPTable table, LabResult result, int rowNum, boolean lastRow) {
    final Date DATE = result.getResultedDate();
    final String TEST = result.getTestname();
    final String RESULT = String.format("%s %s", result.getValue(), result.getValueUnits());
    final String REFERENCE_HIGH = result.getReferenceHigh();
    final String REFERENCE_LOW = result.getReferenceLow();

    String refRange = generateRefRange(REFERENCE_LOW, REFERENCE_HIGH);

    PdfPCell cell;
    cell = createCell(new Phrase(DATE == null ? "" : DateHelper.formatDate(DATE), Stylesheet.getNormal()), rowNum, false, lastRow, PdfPCell.ALIGN_LEFT);
    table.addCell(cell);
    cell = createCell(new Phrase(TEST, Stylesheet.getNormal()), rowNum, false, lastRow, PdfPCell.ALIGN_LEFT);
    table.addCell(cell);
    cell = createCell(new Phrase(RESULT, Stylesheet.getNormal()), rowNum, false, lastRow, PdfPCell.ALIGN_LEFT);
    table.addCell(cell);
    cell = createCell(new Phrase(refRange, Stylesheet.getNormal()), rowNum, true, lastRow, PdfPCell.ALIGN_LEFT);
    table.addCell(cell);
  }

  private String generateRefRange(String referenceLow, String referenceHigh) {
    String refRange;
    if (NullChecker.isNotNullish(referenceLow) && NullChecker.isNotNullish(referenceHigh)) {
      refRange = String.format("%s - %s", referenceLow, referenceHigh);
    } else if (NullChecker.isNotNullish(referenceLow)) {
      refRange = referenceLow;
    } else if (NullChecker.isNotNullish(referenceHigh)) {
      refRange = referenceHigh;
    } else {
      refRange = "--";
    }
    return refRange;
  }
}
