package com.agilex.healthcare.mbb.pdf.pages;

import java.util.Date;

import com.agilex.healthcare.mbb.datalayer.PdfGenerationContext;
import com.agilex.healthcare.mbb.pdf.PageEvent;
import com.agilex.healthcare.mbb.pdf.Stylesheet;
import com.agilex.healthcare.mobilehealthplatform.domain.Admission;
import com.agilex.healthcare.mobilehealthplatform.domain.Admissions;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Paragraph;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfWriter;

public class AdmissionPage extends AbstractPage {
  //private static final Logger LOGGER = Logger.getLogger(AdmissionPage.class);

  public AdmissionPage(Document document, PageEvent event, PdfWriter writer, PdfGenerationContext context) {
    super(document, event, writer, context);
  }

  public void create() throws DocumentException {
    String domain = Domain.admission;
    addHeading(getMessage("mbb.report.admissions.title", null), domain);
    addUpdate(getMessage("mbb.report.admissions.updateFrequency", null), domain);
    addTextDateRange(context.getDateFilter());

    Admissions admissions = (Admissions)context.getInformation(domain);

    if (NullChecker.isNotNullish(admissions)) {
      for (Admission admission : admissions) {
        addAdmission(document, admission);
      }
    } else {
      addNoDataFound();
    }
  }

  private void addAdmission(Document document, Admission admission) throws DocumentException {
    final Date ADMISSION_DATE = admission.getAdmissionDate();
    final Date DISCHARGE_DATE = admission.getDischargeDate();
    final String FACILITY = admission.getFacilityName();
    final String ADMITTING_PROVIDER = admission.getAdmittingProviderName();
    final String DISCHARGE_SUMMARY = NullChecker.isNullish(admission.getDischargeSummary()) ? getMessage("mbb.report.noNoteFound", null) : admission.getDischargeSummary();

    PdfPTable table = createTable();
    document.add(table);

    PdfPTable resultInformationTable = createInformationTable();
    addInformationRow(resultInformationTable, false, getMessage("mbb.report.admissions.admissionDate", null), DateHelper.formatDate(ADMISSION_DATE));
    addInformationRow(resultInformationTable, false, getMessage("mbb.report.admissions.dischargeDate", null), DateHelper.formatDate(DISCHARGE_DATE));
    addInformationRow(resultInformationTable, false, getMessage("mbb.report.admissions.facility", null), FACILITY);
    addInformationRow(resultInformationTable, false, getMessage("mbb.report.admissions.admittingProvider", null), ADMITTING_PROVIDER);
    document.add(resultInformationTable);

    Paragraph dischargeSummary = new Paragraph(DISCHARGE_SUMMARY, Stylesheet.getNormal());
    setupParagraph(dischargeSummary);
    document.add(dischargeSummary);
    addSolidLine();
    addBlankLines(1);
  }

  private PdfPTable createTable() throws DocumentException {
    PdfPTable table = new PdfPTable(1);
    table.setWidthPercentage(100);
    table.setSpacingBefore(0);
    table.setSpacingAfter(10);
    table.setKeepTogether(false);
    table.setSplitRows(false);
    table.setHeaderRows(1);

    PdfPCell cell = createTableHeader(getMessage("mbb.report.admissions.dischargeSummary", null), true, PdfPCell.ALIGN_LEFT);
    table.addCell(cell);

    return table;
  }
}
