package com.agilex.healthcare.mbb.pdf.pages;

import org.springframework.context.NoSuchMessageException;

import com.agilex.healthcare.mbb.datalayer.PdfGenerationContext;
import com.agilex.healthcare.mbb.pdf.PageEvent;
import com.agilex.healthcare.mbb.pdf.Stylesheet;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.utility.DateHelper;
import com.itextpdf.text.Anchor;
import com.itextpdf.text.BaseColor;
import com.itextpdf.text.Chunk;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Font;
import com.itextpdf.text.Paragraph;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfWriter;
import com.itextpdf.text.pdf.draw.LineSeparator;

public abstract class AbstractPage {
  protected Document document;
  protected PageEvent event;
  protected PdfWriter writer;
  protected PdfGenerationContext context;

  public AbstractPage(Document document, PageEvent event, PdfWriter writer, PdfGenerationContext context) {
    this.document = document;
    this.event = event;
    this.writer = writer;
    this.context = context;
  }

  public abstract void create() throws DocumentException;

  public void addHeading(String title, String anchorName) throws DocumentException {
    Paragraph heading = new Paragraph();
    heading.setSpacingBefore(10);

    Anchor anchorTarget = new Anchor(title.toUpperCase(), Stylesheet.getH2());
        anchorTarget.setName(anchorName);

    heading.add(anchorTarget);

    document.add(heading);
  }

  public void addUpdate(String title, String anchorName) throws DocumentException {
    Paragraph heading = new Paragraph();
    heading.setSpacingBefore(0);
    heading.setSpacingAfter(0);

    Anchor anchorTarget = new Anchor(title.toUpperCase(), Stylesheet.getNormal());
    anchorTarget.setName(anchorName);

    heading.add(anchorTarget);

    document.add(heading);
  }

  public void addDateRange(DateFilter dateFilter) throws DocumentException {
    if (dateFilter != null && dateFilter.getStartDate() != null && dateFilter.getEndDate() != null) {
      String startDate = DateHelper.formatDate(dateFilter.getStartDate());
      String endDate = DateHelper.formatDate(dateFilter.getEndDate());
      Paragraph dateRange = new Paragraph();
      dateRange.setSpacingBefore(0);
      dateRange.setSpacingAfter(5);

      Phrase phrase =
          new Phrase(
              context.getMessageSource().getMessage(
                  "mbb.report.dateRange",
                  new String[] {
                      startDate,
                      endDate
                  },
                  context.getLocale()
              ), 
              Stylesheet.getH3()
          );
      dateRange.add(phrase);

      document.add(dateRange);
    }
  }

  public void addTextDateRange(DateFilter dateFilter) throws DocumentException {
    if (dateFilter != null && dateFilter.getStartDate() != null && dateFilter.getEndDate() != null) {
      String startDate = DateHelper.format(dateFilter.getStartDate(), DateHelper.REPORT_DETAILED_DATE_FORMAT_NO_TIME);
      String endDate = DateHelper.format(dateFilter.getEndDate(), DateHelper.REPORT_DETAILED_DATE_FORMAT_NO_TIME);
      Paragraph dateRange = new Paragraph();
      dateRange.setSpacingBefore(0);
      dateRange.setSpacingAfter(5);

      Phrase phrase =
          new Phrase(
              context.getMessageSource().getMessage(
                  "mbb.report.dateRange",
                  new String[] {
                      startDate,
                      endDate
                  },
                  context.getLocale()
              ), 
              Stylesheet.getH3()
          );
      dateRange.add(phrase);

      document.add(dateRange);
    }
  }

  public void addNoDataFound() throws DocumentException {
    Paragraph noDataFound = new Paragraph();
    noDataFound.setSpacingBefore(10);
    noDataFound.setSpacingAfter(10);
    noDataFound.setAlignment(Paragraph.ALIGN_CENTER);

    noDataFound.add(new Phrase(context.getMessageSource().getMessage("mbb.report.noDataFound", null, context.getLocale()), Stylesheet.getItalicLarge()));

    document.add(noDataFound);
  }

  public PdfPTable createInformationTable() throws DocumentException {
    PdfPTable resultInformationTable = new PdfPTable(2);
    resultInformationTable.setWidthPercentage(100);
    resultInformationTable.setWidths(new int[] { 20, 85 });
    resultInformationTable.setSpacingBefore(10);
    resultInformationTable.setSpacingAfter(10);

    return resultInformationTable;
  }

  public void addInformationRow(PdfPTable table, boolean bold, String key, String value) {
    Font font = null;
    if (bold) {
      font = Stylesheet.getBold();
    } else {
      font = Stylesheet.getNormal();
    }

    PdfPCell cell = new PdfPCell(new Phrase(key, font));
    cell.setBorderWidth(0.5f);
    cell.setBackgroundColor(Stylesheet.getInfoTableKey());
    table.addCell(cell);
    cell = new PdfPCell(new Phrase(value, font));
    cell.setBorderWidth(0.5f);
    cell.setBackgroundColor(Stylesheet.getInfoTableValue());
    table.addCell(cell);
  }

  public PdfPCell createTableHeader(String title, boolean lastCol, int horizontalAlignment) {
    PdfPCell cell = new PdfPCell(new Phrase(title, Stylesheet.getBold()));

    cell.setHorizontalAlignment(horizontalAlignment);


    cell.setBorderWidth(0);

    if (lastCol) {
      cell.setBorderWidthRight(0);
    } else {
      cell.setBorderWidthRight(0);
    }
    cell.setBorderWidthBottom(1);
    cell.setBackgroundColor(Stylesheet.getTableHeader());
    cell.setPadding(5);

    return cell;
  }

  public PdfPCell createCell(Phrase phrase, int rowNum, boolean lastCol, boolean lastRow, int horizontalAlignment) {
    PdfPCell cell = new PdfPCell(phrase);
    cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
    cell.setHorizontalAlignment(horizontalAlignment);

    if (rowNum % 2 == 0) {
      cell.setBackgroundColor(Stylesheet.getTableRow1());
    } else {
      cell.setBackgroundColor(Stylesheet.getTableRow2());
    }

    cell.setBorderWidthTop(0);
    if (lastRow) {
      cell.setBorderWidthBottom(0);
    } else {
      cell.setBorderWidthBottom(0);
    }

    cell.setBorderWidthLeft(0);
    if (lastCol) {
      cell.setBorderWidthRight(0);
    } else {
      cell.setBorderWidthRight(0);
    }

    cell.setPadding(5);

    return cell;
  }

  public void addSolidLine() throws DocumentException {
    LineSeparator sep = new LineSeparator();
    sep.setAlignment(PdfPCell.ALIGN_LEFT);
    sep.setLineColor(BaseColor.BLACK);
    document.add(sep);
  }

  public void addBlankLines(int num) throws DocumentException {
    for (int i=0; i<num; i++) {
      document.add(Chunk.NEWLINE);
    }
  }

  public void setupParagraph(Paragraph paragraph) {
    paragraph.setIndentationLeft(5);
    paragraph.setIndentationRight(5);
    paragraph.setLeading(0f, 1.0f);
    paragraph.setSpacingAfter(5.0f);
  }

  public String getMessage(String key, Object[] args) {
    try {
      return context.getMessageSource().getMessage(key, null, context.getLocale());
    } catch (NoSuchMessageException e) {
      return key;
    }
  }

  public Phrase getLabelAndData(String labelKey, String data) {
    Phrase phrase = new Phrase();

    Chunk chunk = new Chunk(getMessage(labelKey, null) + " ", Stylesheet.getNormal());
    phrase.add(chunk);

    chunk = new Chunk(data, Stylesheet.getBold());
    phrase.add(chunk);
    return phrase;

  }
}
