package com.agilex.healthcare.mbb.pdf;

import java.awt.Color;

import com.itextpdf.text.BaseColor;
import com.itextpdf.text.Rectangle;
import com.itextpdf.text.pdf.PdfContentByte;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPCellEvent;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfShading;
import com.itextpdf.text.pdf.PdfShadingPattern;

public class PdfRoundCorners implements PdfPCellEvent {

	private Color[] colorFill;
    private boolean topLeft;
    private boolean topRight;
    private boolean bottomRight;
    private boolean bottomLeft;

    /**
     * Create a new PdfRoundCorners event while specifying a single fill color
     * and which corners will be rounded.
     *
     * @param colorFill The BaseColor to use as the fill color.
     * @param topLeft If true, top left corner will be rounded.
     * @param topRight If true, top right corner will be rounded.
     * @param bottomRight If true, bottom right corner will be rounded.
     * @param bottomLeft If true, bottom left corner will be rounded.
     */
    public PdfRoundCorners(
            Color colorFill,
            boolean topLeft,
            boolean topRight,
            boolean bottomRight,
            boolean bottomLeft) {
        this.colorFill = new Color[]{colorFill};
        this.topLeft = topLeft;
        this.topRight = topRight;
        this.bottomRight = bottomRight;
        this.bottomLeft = bottomLeft;
    }

    /**
     * Create a new PdfRoundCorners event while specifying a vertical background
     * gradient and which corners will be rounded.
     *
     * @param colorFill An array of two BaseColors which will be used to create
     * the gradient.
     * @param topLeft If true, top left corner will be rounded.
     * @param topRight If true, top right corner will be rounded.
     * @param bottomRight If true, bottom right corner will be rounded.
     * @param bottomLeft If true, bottom left corner will be rounded.
     */
    public PdfRoundCorners(
            Color[] colorFill,
            boolean topLeft,
            boolean topRight,
            boolean bottomRight,
            boolean bottomLeft) {
        this.colorFill = colorFill;
        this.topLeft = topLeft;
        this.topRight = topRight;
        this.bottomRight = bottomRight;
        this.bottomLeft = bottomLeft;
    }

    public void cellLayout(PdfPCell cell, Rectangle position, PdfContentByte[] canvases) {
    	int borderWidth = 2;

        // Get the cell's background canvas.
        PdfContentByte cb = canvases[PdfPTable.BACKGROUNDCANVAS];

        // Adjust left and right positions to fix visible gaps.
        float left = position.getLeft() - borderWidth/2;
        float top = position.getTop();
        float right = position.getRight() + borderWidth/2;
        float bottom = position.getBottom();

        // Set the fill color or gradient.
        if(colorFill.length < 2) {
        	BaseColor color = new BaseColor(colorFill[0].getRGB());
            cb.setColorFill(color);
        } else {
            PdfShading shading = PdfShading.simpleAxial(
                    cb.getPdfWriter(), left, top, left, bottom, new BaseColor(colorFill[0].getRGB()), new BaseColor(colorFill[1].getRGB()));
            PdfShadingPattern shadingPattern = new PdfShadingPattern(shading);
            cb.setShadingFill(shadingPattern);
        }

        // Define the background box including rounded corners.
        if (topLeft) {
            cb.moveTo(left, top - 4f);
            cb.curveTo(left, top, left + 4f, top);
        } else cb.moveTo(left, top);

        if (topRight) {
            cb.lineTo(right - 4f, top);
            cb.curveTo(right, top, right, top - 4f);
        } else cb.lineTo(right, top);

        if (bottomRight) {
            cb.lineTo(right, bottom + 4f);
            cb.curveTo(right, bottom, right - 4f, bottom);
        } else cb.lineTo(right, bottom);

        if (bottomLeft) {
            cb.lineTo(left + 4f, bottom);
            cb.curveTo(left, bottom, left, bottom + 4f);
        } else cb.lineTo(left, bottom);

        if (topLeft) cb.lineTo(left, top - 4f);
        else cb.lineTo(left, top);

        cb.closePath();

        // Fill it up!
        cb.fill();
    }
}
