package com.agilex.healthcare.mobilehealthplatform.restservice;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.Paging;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.Patients;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.utility.NullChecker;

public class PatientsSearchTest {
	private PatientDataService dataservice = new PatientDataService();

	@Test
	public void validateNoMatchReturnsEmptyCollection() {
		Patient prototype = new Patient();
		prototype.setLastName("XYZ");
		Patients patients = dataservice.patientSearch(prototype, ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(0, patients.size());
	}

	@Test
	public void validateOneMatchReturnsCollectionWithSizeOne() {
		Patient prototype = new Patient();
		prototype.setLastName("Patient01");
		Patients patients = dataservice.patientSearch(prototype, ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(1, patients.size());
	}

	@Test
	public void validateOneMatchReturnsCollectionWithCorrectPatient() {
		Patient prototype = new Patient();
		prototype.setLastName("Patient01");
		Patients patients = dataservice.patientSearch(prototype, ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(prototype.getLastName().toLowerCase(), patients.get(0).getLastName().toLowerCase());
	}

	@Test
	public void validateTwoMatchesReturnsCollectionWithSizetTwo() {
		Patient prototype = new Patient();
		prototype.setLastName("Patient02");
		Patients patients = dataservice.patientSearch(prototype, ScopeFilter.getInstanceForLongitudinalScope());
		assertTrue(patients.size() > 1);
	}

	@Test
	public void validateTwoMatchReturnsCollectionWithCorrectPatients() {
		Patient prototype = new Patient();
		prototype.setLastName("Patient26");
		Patients patients = dataservice.patientSearch(prototype, ScopeFilter.getInstanceForLongitudinalScope());
		for (Patient patient : patients) {
			assertEquals(prototype.getLastName().toLowerCase(), patient.getLastName().toLowerCase());
		}
	}

	@Test
	public void validateOneMatchByFirstAndLastReturnsCollectionWithCorrectPatient() {
		Patient prototype = new Patient();
		prototype.setFirstName("zztest");
		prototype.setLastName("Patient57");
		Patients patients = dataservice.patientSearch(prototype, ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(1, patients.size());
		assertEquals("zztest", patients.get(0).getFirstName().toLowerCase());
		assertEquals("patient57", patients.get(0).getLastName().toLowerCase());
	}

	@Test
	public void validateOneMatchIgnoringCaseReturnsCollectionWithCorrectPatient() {
		Patient prototype = new Patient();
		prototype.setLastName("patient01");
		Patients patients = dataservice.patientSearch(prototype, ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals("patient01", patients.get(0).getLastName().toLowerCase());
	}

	@Test
	public void validatePatientHasId() {
		Patient prototype = new Patient();
		prototype.setLastName("Patient01");
		Patients patients = dataservice.patientSearch(prototype, ScopeFilter.getInstanceForLongitudinalScope());
		assertNotNull(patients.get(0).getId());
	}

	@Test
	public void validateSearchResultsAreLimitedTo20() {
		int maxrecords = Paging.DEFAULT_MAX_RECORDS;

		Patients patients = dataservice.patientSearch(",zztest", "500", ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(maxrecords, patients.size());
	}

	@Test
	public void validateSearchBySsnPattern() {
		String searchString = "P9901";
		Patients patients = dataservice.patientSearch(searchString, "500", ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(1, patients.size());
		assertEquals("patient01", patients.get(0).getLastName().toLowerCase());
	}

	@Test
	public void validateSearchByLastFourSsn() {
		String searchString = "9901";
		Patients patients = dataservice.patientSearch(searchString, "500", ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(1, patients.size());
		assertEquals("patient01", patients.get(0).getLastName().toLowerCase());
	}

    //
    // There are no patients in the test data with dashes in their last name, so this test can no longer be run.
    //
//	@Test
//	public void validateSearchByLastnameWithDash() {
//		String searchString = "Jones-     ,Mirka";
//		Patients patients = dataservice.patientSearch(searchString, "500", ScopeFilter.getInstanceForLongitudinalScope());
//		assertEquals(1, patients.size());
//	}

	@Test
	public void validateSearchBySsn() {
		String searchString = "999999901";
		Patients patients = dataservice.patientSearch(searchString, "500", ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(1, patients.size());
	}

	@Test
	public void validateSearchBySsnWithDashBySearchString() {
		String searchString = "999-99-9901";
		Patients patients = dataservice.patientSearch(searchString, "500", ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(1, patients.size());
	}

	@Test
	public void validateSearchBySsnWithDash() {
		String searchString = "999999901";
		Patient prototype = new Patient();
		prototype.setSsn(searchString);
		Patients patients = dataservice.patientSearch(prototype, ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(1, patients.size());
	}

	@Test
	public void searchWithInsufficientCriteria() {
		String searchString = "";
		Patients patients = dataservice.patientSearch(searchString, "500", ScopeFilter.getInstanceForLongitudinalScope());
		assertNotNull(patients);
		assertEquals(0, patients.size());
	}

	@Test
	public void validatePatientHasAge() {
		Patient prototype = new Patient();
		prototype.setLastName("Patient01");
		Patients patients = dataservice.patientSearch(prototype, ScopeFilter.getInstanceForLongitudinalScope());
		Patient patient = patients.get(0);
		assertTrue(NullChecker.isNotNullish(patient.getAge()));
	}
}
