package com.agilex.healthcare.mbb.controller.soc;

import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.forwardedUrl;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.model;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.view;

import org.junit.Test;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import org.springframework.test.context.web.WebAppConfiguration;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.test.context.ContextConfiguration;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.web.servlet.MockMvc;
import org.junit.Before;

import com.agilex.healthcare.mobilehealthplatform.datalayer.allergy.AllergyDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationFilter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.problem.ProblemDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.surgery.SurgeryDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.Allergies;
import com.agilex.healthcare.mobilehealthplatform.domain.Medications;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientDemographics;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.Problems;
import com.agilex.healthcare.mobilehealthplatform.domain.Surgeries;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.test.SpringBasedIntegration;

@RunWith(SpringJUnit4ClassRunner.class)
@WebAppConfiguration
@ContextConfiguration("classpath:applicationContext.xml")
public class PatientSummaryClinicalDomainControllerTest extends SpringBasedIntegration {
	PatientIdentifier patientIdentifier = getPatientIdentifier();

  @Autowired  private WebApplicationContext wac;
  private MockMvc mockMvc;

  @Before
  public void setup () {
    this.mockMvc = MockMvcBuilders.webAppContextSetup(this.wac).build();
  }

	@Test
	public void testController() throws Exception {
		PatientDataService dataservice = new PatientDataService();
		PatientDemographics demographics = getSourceDemographics(dataservice);
		Patient patient = getSourcePatient(dataservice);
		Allergies allergies = getSourceAllergies();
		Medications medications = getSourceMedications();
		Problems problems = getSourceProblems();
		Surgeries surgeries = getSourceSurgeries();

		this.mockMvc
			.perform(get("/summary"))
			.andExpect(status().isOk())
			.andExpect(forwardedUrl("soc/patientSummary"))
			.andExpect(view().name("soc/patientSummary"))
			.andExpect(model().attributeExists("demographics", "patient", "allergies", "medications", "problems", "surgeries"))
			.andExpect(model().attribute("allergies", allergies))
			//.andExpect(model().attribute("medications", medications))
			.andExpect(model().attributeExists("medications"))
			.andExpect(model().attribute("problems", problems))
			.andExpect(model().attribute("surgeries", surgeries));
	}

	private PatientDemographics getSourceDemographics(PatientDataService dataservice) {
		PatientDemographics demographics = dataservice.getPatientDemographics(patientIdentifier, ScopeFilter.getInstanceForLongitudinalScope());

		return demographics;
	}

	private Patient getSourcePatient(PatientDataService dataservice) {
		Patient patient = dataservice.getPatient(patientIdentifier, ScopeFilter.getInstanceForLongitudinalScope());

		return patient;
	}

	private Allergies getSourceAllergies() {
		AllergyDataService allergyDataservice = new AllergyDataService();
		Allergies allergies = allergyDataservice.fetchPatientAllergies(patientIdentifier, ScopeFilter.getInstanceForLongitudinalScope());

		return allergies;
	}

	private Medications getSourceMedications() {
		MedicationDataService dataservice = new MedicationDataService();
		MedicationFilter filter = MedicationFilterFactory.createEmptyFilter();
		Medications medications = dataservice.getActiveAndRecentlyExpiredPatientMedications(patientIdentifier, filter, ScopeFilter.getInstanceForLongitudinalScope());

		return medications;
	}

	private Problems getSourceProblems() {
		ProblemDataService dataservice = new ProblemDataService();
		Problems problems = dataservice.fetchProblems(patientIdentifier, DateFilterFactory.createEmptyFilter(), ScopeFilter.getInstanceForLongitudinalScope());

		return problems;
	}

	private Surgeries getSourceSurgeries() {
		SurgeryDataService dataservice = new SurgeryDataService();
		Surgeries surgeries = dataservice.getPatientSurgeries(patientIdentifier, DateFilterFactory.createEmptyFilter(), ScopeFilter.getInstanceForLongitudinalScope());

		return surgeries;
	}
}
