package com.agilex.healthcare.mobilehealthplatform.utils;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientSearchPaging;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.Patients;

public class PagingTest {

	@Test
	public void nullInput() {
		Paging paging = new Paging();
		Patients result = paging.enforceMaxRecords(null, new PatientSearchPaging());
		assertNull(result);
	}

	@Test
	public void emptyInput() {
		Patients patients = new Patients();
		Paging paging = new Paging();
		Patients result = paging.enforceMaxRecords(patients, new PatientSearchPaging());
		assertEquals(0, result.getSize());
	}

	@Test
	public void numberOfPatientsLessThanPageSize() {
		runTest(2, 5, 1, 2, "1", "2");
	}

	@Test
	public void numberOfPatientsLessThanPageSizeWithNullPageNumber() {
		runTest(2, 5, 0, 2, "1", "2");
	}

	@Test
	public void numberOfPatientsEqualsPageSize() {
		runTest(5, 5, 1, 5, "1", "5");
	}

	@Test
	public void numberOfPatientsEqualsPageSizeWithNullPageNumber() {
		runTest(5, 5, 0, 5, "1", "5");
	}

	@Test
	public void numberOfPatientsGreaterThanPageSize() {
		runTest(10, 5, 1, 5, "1", "5");
	}

	@Test
	public void numberOfPatientsGreaterThanPageSizeNullPageNumber() {
		runTest(10, 5, 0, 5, "1", "5");
	}

	@Test
	public void page2() {
		runTest(20, 5, 2, 5, "6", "10");
	}

	@Test
	public void lastPage() {
		runTest(11, 5, 3, 1, "11", "11");
	}

	@Test
	public void noMaxRecords() {
		runTest(20, 0, 0, 20, "1", "20");
	}
	@Test
	public void noPatientsForThisPage() {
		runTest(10, 5, 3, 0, null, null);
	}

	private void runTest(int numberOfPatients, int maxPatients, int pageNumber, int expectedNumberOfPatients, String firstPatientId, String lastPatientId) {
		Patients patients = getPatientList(numberOfPatients);
		PatientSearchPaging pagingOptions = getPatientSearchPagingOptions(maxPatients, pageNumber);

		Paging paging = new Paging();
		Patients result = paging.enforceMaxRecords(patients, pagingOptions);

		assertEquals(expectedNumberOfPatients, result.getSize());
		if (expectedNumberOfPatients > 0) {
			assertEquals(firstPatientId, result.get(0).getId());
			assertEquals(lastPatientId, result.get(expectedNumberOfPatients - 1).getId());
		}

		assertEquals("The original patient list has been altered.  The original list should be left alone.", numberOfPatients, patients.getSize());
	}

	private PatientSearchPaging getPatientSearchPagingOptions(int maxRecords, int pageNumber) {
		PatientSearchPaging patientSearchPagingOptions = new PatientSearchPaging();
		patientSearchPagingOptions.setMaxRecords(maxRecords);
		patientSearchPagingOptions.setPageNumber(pageNumber);
		return patientSearchPagingOptions;

	}

	private Patients getPatientList(int numberOfPatients) {
		Patients patientList = new Patients();
		for (int i = 1; i <= numberOfPatients; i++) {
			patientList.add(createPatient(Integer.toString(i)));
		}
		return patientList;
	}

	private Patient createPatient(String num) {
		Patient p = new Patient();
		p.setLastName(num);
		p.setId(num);
		return p;
	}
}
