package com.agilex.healthcare.mobilehealthplatform.serviceregistry;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import org.junit.Before;
import org.junit.Test;

public class ServiceRegistryTest {

	private ServiceRegistry serviceRegistry;
	private Community communityA = new Community("c-A", "a1");
	private Community communityB = new Community("c-B", "a2");
	private Community communityC = new Community("c-c", "a3");

	@Before
	public void beforeTest() {
		serviceRegistry = ServiceRegistry.getLocalInstance().clear();
	}

	@Test
	public void clearServiceRegistry() {
		serviceRegistry.registerCommunity(communityA);
		serviceRegistry.clear();
		assertEquals(0, serviceRegistry.getCommunities().size());
		serviceRegistry.reset();
	}

	@Test
	public void resetServiceRegistry() {
		serviceRegistry.registerCommunity(communityA);
		final int initialSize = serviceRegistry.getCommunities().size();
		assertTrue("For this to be an effective test, need to start off with a populated service registry", initialSize > 0);
		serviceRegistry.clear();
		final int clearedSize = serviceRegistry.getCommunities().size();
		assertEquals(0, clearedSize);
		serviceRegistry.reset();
		final int resetSize = serviceRegistry.getCommunities().size();
		assertEquals(initialSize, resetSize);
	}

	@Test
	public void registerCommunity() {
		serviceRegistry.clear();
		final int initialSize = serviceRegistry.getCommunities().size();
		serviceRegistry.registerCommunity(new Community("cid", "aa"));
		final int sizeAfterAdd = serviceRegistry.getCommunities().size();
		assertEquals(initialSize + 1, sizeAfterAdd);
	}

	@Test(expected = IllegalArgumentException.class)
	public void registerCommunityRequiresHomeCommunityId() {
		serviceRegistry.registerCommunity(new Community("", "aa"));
	}

	@Test(expected = IllegalArgumentException.class)
	public void registerCommunityRequiresAssigningAuthority() {
		serviceRegistry.registerCommunity(new Community("hc", ""));
	}

	@Test
	public void registerDuplicateCommunityDoesnotCreateDuplicate() {
		final int initialSize = serviceRegistry.getCommunities().size();
		serviceRegistry.registerCommunity(communityA);
		assertEquals(initialSize + 1, serviceRegistry.getCommunities().size());
		serviceRegistry.registerCommunity(communityA);
		assertEquals(initialSize + 1, serviceRegistry.getCommunities().size());
	}

	@Test
	public void registerDuplicateCommunityCanRetrieveByHc() {
		serviceRegistry.clear();
		Community a = new Community("hc1", "aa1");
		serviceRegistry.registerCommunity(a);
		Community b = new Community("hc1", "aa2");
		serviceRegistry.registerCommunity(b);
		assertEquals(b, serviceRegistry.getCommunityById("hc1"));
	}

	@Test
	public void registerDuplicateCommunityCanRetrieveByAa() {
		serviceRegistry.clear();
		Community a = new Community("hc1", "aa1");
		serviceRegistry.registerCommunity(a);
		Community b = new Community("hc1", "aa2");
		serviceRegistry.registerCommunity(b);
		assertEquals(b, serviceRegistry.getCommunityByAssigningAuthority("aa2"));
	}

	@Test
	public void registerDuplicateCommunityCannotRetrieveByPreviousAa() {
		serviceRegistry.clear();
		Community a = new Community("hc1", "aa1");
		serviceRegistry.registerCommunity(a);
		Community b = new Community("hc1", "aa2");
		serviceRegistry.registerCommunity(b);
		assertEquals(null, serviceRegistry.getCommunityByAssigningAuthority("aa1"));
	}

	@Test
	public void getCommunityByCommunityId() {
		serviceRegistry.clear();
		serviceRegistry.registerCommunity(communityA).registerCommunity(communityB);
		assertEquals(communityA, serviceRegistry.getCommunityById(communityA.getId()));
	}

	@Test
	public void getCommunityByCommunityIdGetCommuityNotFirstInList() {
		serviceRegistry.clear();
		serviceRegistry.registerCommunity(communityA).registerCommunity(communityB);
		assertEquals(communityB, serviceRegistry.getCommunityById(communityB.getId()));
	}

	@Test
	public void getCommunityNotFound() {
		serviceRegistry.clear();
		serviceRegistry.registerCommunity(communityA).registerCommunity(communityB);
		assertEquals(null, serviceRegistry.getCommunityById(communityC.getId()));
	}

	@Test
	public void getCommunityOnEmptyList() {
		serviceRegistry.clear();
		assertEquals(null, serviceRegistry.getCommunityById(communityC.getId()));
	}

	@Test
	public void registerPrimaryCommunity() {
		serviceRegistry.clear();
		serviceRegistry.registerCommunity(communityA).registerCommunity(communityC);
		serviceRegistry.registerPrimaryCommunity(communityB);
		assertEquals(communityB, serviceRegistry.getPrimaryCommunity());
	}

	@Test
	public void setPrimaryCommunityId() {
		serviceRegistry.clear();
		serviceRegistry.registerCommunity(communityA).registerCommunity(communityB).registerCommunity(communityC);
		serviceRegistry.setPrimaryCommunityId(communityB.getId());
		assertEquals(communityB, serviceRegistry.getPrimaryCommunity());
	}

	@Test(expected = IllegalArgumentException.class)
	public void setPrimaryCommunityIdNotAlreadyRegistered() {
		serviceRegistry.clear();
		serviceRegistry.registerCommunity(communityA).registerCommunity(communityC);
		serviceRegistry.setPrimaryCommunityId(communityB.getId());
	}

	@Test
	public void getCommunityByAssigningAuthority() {
		serviceRegistry.clear();
		serviceRegistry.registerCommunity(communityA).registerCommunity(communityB).registerCommunity(communityC);
		assertEquals(communityB, serviceRegistry.getCommunityByAssigningAuthority(communityB.getAssigningAuthority()));
	}

	@Test
	public void getCommunityByAssigningAuthorityNotFound() {
		serviceRegistry.clear();
		serviceRegistry.registerCommunity(communityA).registerCommunity(communityC);
		assertEquals(null, serviceRegistry.getCommunityByAssigningAuthority(communityB.getAssigningAuthority()));
	}
}
