package com.agilex.healthcare.mobilehealthplatform.datalayer.patient;

import static org.junit.Assert.assertEquals;

import org.junit.Ignore;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.MockRouterHelper;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.Router;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.Patients;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;

@Ignore
public class PatientDataServicePagingTest {
	private final static org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientDataServicePagingTest.class);

	@Test
	public void retrieveSinglePatient() {
		PatientDataService patientdataservice = setupDataServiceWithSinglePatient();

		PatientSearchPaging pagingInfo = new PatientSearchPaging();
		pagingInfo.setMaxRecords(5);

		Patients results = patientdataservice.patientSearch("Mock     ", "500", pagingInfo, ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(1, results.getSize());
		assertEquals("Mock     ", results.get(0).getLastName());
	}

	private PatientDataService setupDataServiceWithSinglePatient() {
		Patients patients = new Patients();
		patients.add(createPatient("1", "Mock     ", "a"));
		return setupDataService(patients);
	}

	@Test
	public void retrievePatientsLimitedByCustomMaxRecords() {
		PatientDataService patientdataservice = setupDataServiceWithManyPatients();

		PatientSearchPaging pagingInfo = new PatientSearchPaging();
		pagingInfo.setMaxRecords(5);

		Patients result = patientdataservice.patientSearch("     ", "500", pagingInfo, ScopeFilter.getInstanceForLongitudinalScope());

		assertEquals(5, result.size());
	}

	@Test
	public void retrievePatientsFirstPageContainsCorrectPatients() {
		PatientDataService patientdataservice = setupDataServiceWithManyPatients();

		PatientSearchPaging pagingInfo = new PatientSearchPaging();
		pagingInfo.setMaxRecords(5);

		Patients result = patientdataservice.patientSearch("     ", "500", pagingInfo, ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(5, result.size());
		assertEquals("1", result.get(0).getId());
		assertEquals("2", result.get(1).getId());
		assertEquals("3", result.get(2).getId());
		assertEquals("4", result.get(3).getId());
		assertEquals("5", result.get(4).getId());
	}

	@Test
	public void retrievePatientsSecondPageContainsCorrectPatients() {
		PatientDataService patientdataservice = setupDataServiceWithManyPatients();

		PatientSearchPaging pagingInfo = new PatientSearchPaging();
		pagingInfo.setMaxRecords(5);
		pagingInfo.setPageNumber(2);

		Patients result = patientdataservice.patientSearch("     ", "500", pagingInfo, ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(5, result.size());
		assertEquals("6", result.get(0).getId());
		assertEquals("7", result.get(1).getId());
		assertEquals("8", result.get(2).getId());
		assertEquals("9", result.get(3).getId());
		assertEquals("10", result.get(4).getId());
	}

	@Test
	public void retrievePatientsLastPageContainsCorrectPatients() {
		PatientDataService patientdataservice = setupDataServiceWithManyPatients();

		PatientSearchPaging pagingInfo = new PatientSearchPaging();
		pagingInfo.setMaxRecords(5);
		pagingInfo.setPageNumber(3);

		Patients result = patientdataservice.patientSearch("     ", "500", pagingInfo, ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(1, result.size());
		assertEquals("11", result.get(0).getId());
	}

	@Test
	public void retrievePatientsBeyondLastPage() {
		PatientDataService patientdataservice = setupDataServiceWithManyPatients();

		PatientSearchPaging pagingInfo = new PatientSearchPaging();
		pagingInfo.setMaxRecords(5);
		pagingInfo.setPageNumber(4);

		Patients result = patientdataservice.patientSearch("     ", "500", pagingInfo, ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(0, result.size());
	}

	@Test
	public void retrievePatientsNoRecordsNoPagingInfo() {
		PatientDataService patientdataservice = setupDataServiceWithNoPatients();

		Patients patients = patientdataservice.patientSearch("     ", "500", ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(0, patients.getSize());
	}

	@Test
	public void retrievePatientsNoRecordsWithPagingInfo() {
		PatientDataService patientdataservice = setupDataServiceWithNoPatients();

		PatientSearchPaging pagingInfo = new PatientSearchPaging();
		pagingInfo.setMaxRecords(5);

		Patients patients = patientdataservice.patientSearch("     ", "500", pagingInfo, ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(0, patients.getSize());
	}

	@Test
	public void retrievePatientsAskForSecondPageWhenDataFitsInOnePage() {
		PatientDataService patientdataservice = setupDataServiceWithSinglePatient();

		PatientSearchPaging pagingInfo = new PatientSearchPaging();
		pagingInfo.setMaxRecords(5);
		pagingInfo.setPageNumber(2);

		Patients patients = new Patients();
		patients.add(createPatient("1", "     ", "a"));

		Patients results = patientdataservice.patientSearch("     ", "500", pagingInfo, ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(0, results.getSize());
	}

	@Test
	public void retrieveLimitedByDefaultMaxRecords() {
		PatientDataService patientdataservice = setupDataServiceWithThirtyPatients();

		PatientSearchPaging paging = new PatientSearchPaging();

		Patients results = patientdataservice.patientSearch("     ", "500", paging, ScopeFilter.getInstanceForLongitudinalScope());
		logger.debug("results count " + results.getSize());
		assertEquals(Paging.DEFAULT_MAX_RECORDS, results.getSize());
	}

	private PatientDataService setupDataServiceWithManyPatients() {
		Patients patients = new Patients();
		patients.add(createPatient("11", "     ", "k"));
		patients.add(createPatient("10", "     ", "j"));
		patients.add(createPatient("9", "     ", "i"));
		patients.add(createPatient("1", "     ", "a"));
		patients.add(createPatient("2", "     ", "b"));
		patients.add(createPatient("3", "     ", "c"));
		patients.add(createPatient("4", "     ", "d"));
		patients.add(createPatient("5", "     ", "e"));
		patients.add(createPatient("6", "     ", "f"));
		patients.add(createPatient("7", "     ", "g"));
		patients.add(createPatient("8", "     ", "h"));

		Router router = MockRouterHelper.createMockRouterToReturnPatients(patients);
		return new PatientDataService(router);
	}

	private PatientDataService setupDataServiceWithNoPatients() {
		Patients patients = new Patients();

		Router router = MockRouterHelper.createMockRouterToReturnPatients(patients);
		return new PatientDataService(router);
	}

	private PatientDataService setupDataServiceWithThirtyPatients() {
		Patients patients = new Patients();
		for (int i = 1; i < 30; i++) {
			patients.add(createPatient(Integer.toString(i), "     ", "John"));
		}

		Router router = MockRouterHelper.createMockRouterToReturnPatients(patients);
		return new PatientDataService(router);
	}

	private PatientDataService setupDataService(Patients patients) {
		Router router = MockRouterHelper.createMockRouterToReturnPatients(patients);
		return new PatientDataService(router);
	}

	private Patient createPatient(String id, String lastname, String firstname) {
		Patient patient = new Patient();
		patient.setId(id);
		patient.setLastName(lastname);
		patient.setFirstName(firstname);
		return patient;
	}
}
