package com.agilex.healthcare.mobilehealthplatform.datalayer.notification;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import java.util.Date;
import java.util.LinkedList;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response.Status;

import org.junit.Before;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.datalayer.audit.AuditLogDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.AuditLog;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistration;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistrations;
import com.agilex.healthcare.mobilehealthplatform.domain.Notification;
import com.agilex.healthcare.mobilehealthplatform.domain.Notifications;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.utils.PropertyHelper;

public class NotificationDataServiceTest {
	private static final String TEST_NOTIFICATION_ID = "1";
	private static final String TEST_USER_ID = "PATID26";
	private static final String TEST_NOTIFICATION_BODY = "test body";
	private static final String TEST_DEVICE_TOKEN = "test body";
	private DateFilter emptyFilter = DateFilterFactory.createEmptyFilter();
	private static final String NOTIFICATIONS_OPTOUT_ENABLED = "notifications.option_enabled";
	
	private NotificationDataService notificationDataService;
	private Notification notification;
	
	
	@Before
	public void setUp() {
		notification = createNewNotification(TEST_NOTIFICATION_ID, TEST_USER_ID, TEST_NOTIFICATION_BODY, new Date(), true);
		notificationDataService = setupDataServiceWithMockData();
	}
	
	@Test
	public void retrieveOfTruncatedNotificationsWithResults() {
		Notifications notifications = notificationDataService.fetchAllNotifications(TEST_USER_ID, emptyFilter);
		assertEquals(50, notifications.size());
	}
	
	@Test
	public void retrieveSpecificNotification() {
		Notification notification = notificationDataService.fetchNotification(TEST_USER_ID, TEST_NOTIFICATION_ID);
		assertEquals(notification.getBody(), TEST_NOTIFICATION_BODY);
		assertEquals(notification.getUserId(), TEST_USER_ID);
	}
	
	@Test
	public void saveNotification() {
		Notification savedNotification = notificationDataService.saveNotification(notification);
		assertEquals(notification.getNotificationId(), savedNotification.getNotificationId());
	}

	@Test
	public void deleteNotification() {
		boolean testResult = true;
		try {
			notificationDataService.deleteNotification(notification);
		} catch (WebApplicationException e) {
			if (e.getResponse().getStatus() == Status.NOT_FOUND.getStatusCode()) {
				testResult = false;
			}
		} finally {
			assertTrue(testResult);
		}
	}
	
	@Test
	public void deleteNotificationByUserIdAndNotificationid() {
		boolean testResult = true;
		try {
			notificationDataService.deleteNotification(notification.getUserId(), notification.getNotificationId());
		} catch (WebApplicationException e) {
			if (e.getResponse().getStatus() == Status.NOT_FOUND.getStatusCode()) {
				testResult = false;
			}
		} finally {
			assertTrue(testResult);
		}
	}

	@Test
	public void retrieveOfNotificationsWithNoResultsFromDao() {
		NotificationDataService notificationDataServiceNoMock = setupDataServiceWithNoMockData();
		Notifications notifications = notificationDataServiceNoMock.fetchAllNotifications(TEST_USER_ID, emptyFilter);
		assertEquals(0, notifications.size());
	}
	
	private NotificationDataService setupDataServiceWithMockData() {
		Notifications notifications = new Notifications();
		notifications.add(notification);

		Notifications oneHundredNotifications = new Notifications();
		populateOneHundredNotifications(oneHundredNotifications);
		
		NotificationDataLayer mockDao = mock(NotificationDataLayer.class);
		when(mockDao.getActiveNotifications(TEST_USER_ID, emptyFilter)).thenReturn(oneHundredNotifications);
		when(mockDao.getNotification(TEST_USER_ID, TEST_NOTIFICATION_ID)).thenReturn(notification);
		
		DateFilter todayFilter = DateFilterFactory.createFilterFromToday();
		when(mockDao.getActiveNotifications(TEST_USER_ID, todayFilter)).thenReturn(oneHundredNotifications);
		when(mockDao.saveNotification(notification)).thenReturn(notification);
		
		PropertyHelper propertyHelper = mock(PropertyHelper.class);
		when(propertyHelper.getProperty(NOTIFICATIONS_OPTOUT_ENABLED)).thenReturn("true");
		
		NotificationDataService notificationDataService = new NotificationDataService(mockDao, setupNotificationCommunicationDataLayerWithMockData(), setupDeviceRegistrationDataServiceWithMockData(), setupAuditLogDataService(), propertyHelper);
		return notificationDataService;
	}

	private NotificationDataService setupDataServiceWithNoMockData() {
		Notifications notifications = new Notifications();
		NotificationDataLayer mockDao = mock(NotificationDataLayer.class);
		when(mockDao.getActiveNotifications(TEST_USER_ID, emptyFilter)).thenReturn(notifications);

		PropertyHelper propertyHelper = mock(PropertyHelper.class);
		when(propertyHelper.getProperty(NOTIFICATIONS_OPTOUT_ENABLED)).thenReturn("true");
		
		NotificationDataService notificationDataService = new NotificationDataService(mockDao, setupNotificationCommunicationDataLayerWithMockData(), setupDeviceRegistrationDataServiceWithMockData(), setupAuditLogDataService(), propertyHelper);
		return notificationDataService;
	}

	private DeviceRegistrationDataService setupDeviceRegistrationDataServiceWithMockData() {
		DeviceRegistrations deviceRegistrations = new DeviceRegistrations();
		DeviceRegistration deviceRegistration = new DeviceRegistration();
		deviceRegistration.setDeviceToken(TEST_DEVICE_TOKEN);
		deviceRegistration.setUserId(TEST_USER_ID);
		deviceRegistration.setRegisteredDate(new Date());
		deviceRegistrations.add(deviceRegistration);

		DeviceRegistrationDataService mockDao = mock(DeviceRegistrationDataService.class);
		when(mockDao.fetchRegistrations(TEST_USER_ID)).thenReturn(deviceRegistrations);
		when(mockDao.fetchRegistration(TEST_USER_ID, TEST_DEVICE_TOKEN)).thenReturn(deviceRegistration);
		
		return mockDao;
	}
	
	private NotificationCommunicationDataLayer setupNotificationCommunicationDataLayerWithMockData() {
		NotificationCommunicationDataLayer mockDao = mock(NotificationCommunicationDataLayer.class);
		String[] deviceTokens = { TEST_DEVICE_TOKEN };
		when(mockDao.send(deviceTokens, TEST_NOTIFICATION_BODY)).thenReturn(new LinkedList<String>());
		when(mockDao.fetchInvalidDeviceTokens()).thenReturn(new LinkedList<String>());
		
		return mockDao;
	}
	
	private AuditLogDataService setupAuditLogDataService() {
		AuditLog auditLog = new AuditLog();
		AuditLogDataService mockAuditLog = mock(AuditLogDataService.class);
		when(mockAuditLog.saveAudit(auditLog)).thenReturn(auditLog);
		
		return mockAuditLog;
	}
	
	private Notification createNewNotification(String id, String userId, String body, Date date, boolean active) {
		Notification notification = new Notification();
		notification.setNotificationId(id);
		notification.setUserId(userId);
		notification.setBody(body);
		notification.setDate(date);
		notification.setActiveFlag(active);
		
		return notification;
	}
	
	public void populateOneHundredNotifications(Notifications notifications) {
		for (int i = 0; i < 100; i++) {
			notifications.add(createNewNotification(String.valueOf(i), TEST_USER_ID, TEST_NOTIFICATION_BODY, new Date(), true));
		}
	}
}
