package com.agilex.healthcare.mobilehealthplatform.datalayer.notification;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import java.util.Date;
import java.util.LinkedList;
import java.util.List;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response.Status;

import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.security.AppUser;
import org.junit.*;
import org.springframework.security.authentication.TestingAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.context.SecurityContextImpl;

import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistration;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistrations;

public class DeviceRegistrationDataServiceTest {
	private static final String TEST_USER_ID = "PATID26";
	private static final String TEST_FIRST_NAME = "Aldie";
	private static final String TEST_LAST_NAME = "     ";
	private static final String TEST_CURRENT_LAST_NAME = "New     ";
	private static final String TEST_CURRENT_FIRST_NAME = "NewAldie";
	private static final String TEST_DEVICE_REGISTRATION_ID="testId";
	private static final String TEST_DEVICE_TOKEN = "test token";
	private static final String TEST_DEVICE_TOKEN2 = "test nonexistent device";
	private static final String TEST_NOTIFICATION_BODY = "test notification";

	private DeviceRegistrationDataService deviceRegistrationDataService;
	private DeviceRegistrations deviceRegistrations;
	private List<String> userIds;
	private DeviceRegistration deviceRegistration;
	private DeviceRegistration savedDeviceRegistration;
	private DeviceRegistration nonexistentDeviceRegistration;
	private DeviceRegistration savedNonexistentDeviceRegistration;

	@Before
	public void setUp() {
		createSpringSecurityContext();
		
		deviceRegistration = createNewDeviceRegistration(TEST_USER_ID, TEST_DEVICE_TOKEN);
		savedDeviceRegistration = createNewDeviceRegistration(TEST_USER_ID, TEST_DEVICE_TOKEN);
		savedDeviceRegistration.setDeviceRegistrationId(TEST_DEVICE_REGISTRATION_ID);
		savedDeviceRegistration.setRegisteredDate(new Date());
		savedDeviceRegistration.setFirstName(TEST_FIRST_NAME);
		savedDeviceRegistration.setLastName(TEST_LAST_NAME);

		nonexistentDeviceRegistration = createNewDeviceRegistration(TEST_USER_ID, TEST_DEVICE_TOKEN2);
		savedNonexistentDeviceRegistration = createNewDeviceRegistration(TEST_USER_ID, TEST_DEVICE_TOKEN2);
		savedNonexistentDeviceRegistration.setDeviceRegistrationId("testId");
		savedNonexistentDeviceRegistration.setRegisteredDate(new Date());
		savedNonexistentDeviceRegistration.setFirstName(TEST_FIRST_NAME);
		savedNonexistentDeviceRegistration.setLastName(TEST_LAST_NAME);
		
		deviceRegistrationDataService = setupDataServiceWithMockData();
	}
	
	@After
	public void clearContext() {
		SecurityContextHolder.clearContext();
	}

	private void createSpringSecurityContext() {
		SecurityContext securityContext = new SecurityContextImpl();
        MhpUser user = new MhpUser();
        user.setId(TEST_USER_ID);
		user.setFirstName(TEST_CURRENT_FIRST_NAME);
		user.setLastName(TEST_CURRENT_LAST_NAME);
        AppUser appUser = new AppUser("testuser", "testpassword", new LinkedList<GrantedAuthority>(), user);
        Authentication authentication = new TestingAuthenticationToken(appUser, null);
		securityContext.setAuthentication(authentication);
		SecurityContextHolder.setContext(securityContext);
	}
	
	@Test
	public void verifySave() {
		DeviceRegistrationDataLayer mockDl = mock(DeviceRegistrationDataLayer.class);
		when(mockDl.saveRegistration(deviceRegistration)).thenReturn(savedDeviceRegistration);
		when(mockDl.getRegistration(TEST_USER_ID, TEST_DEVICE_TOKEN)).thenReturn(null);
		when(mockDl.getRegistrations(TEST_USER_ID)).thenReturn(deviceRegistrations);
		
		DeviceRegistrationDataService registrationService = new DeviceRegistrationDataService(mockDl, setupNotificationCommunicationDataLayerWithMockData());
		DeviceRegistration savedRegistration = registrationService.saveRegistration(deviceRegistration);
		assertNotNull(savedRegistration);
		assertEquals(TEST_CURRENT_FIRST_NAME, savedRegistration.getFirstName());
		assertEquals(TEST_CURRENT_LAST_NAME, savedRegistration.getLastName());
		assertNotNull(savedRegistration.getDeviceRegistrationId());
	}
	
//	@Test
//	public void verifyUpdate() {
//		deviceRegistration.setOptedIn(false);
//		DeviceRegistrationDataLayer mockDl = mock(DeviceRegistrationDataLayer.class);
//		when(mockDl.getRegistration(TEST_USER_ID, TEST_DEVICE_TOKEN)).thenReturn(savedDeviceRegistration);
//		when(mockDl.getRegistrations(TEST_USER_ID)).thenReturn(deviceRegistrations);
//		savedDeviceRegistration.setOptedIn(false);
//		when(mockDl.saveRegistration(savedDeviceRegistration)).thenReturn(savedDeviceRegistration);
//		
//		DeviceRegistrationDataService registrationService = new DeviceRegistrationDataService(mockDl, setupNotificationCommunicationDataLayerWithMockData());
//		DeviceRegistration savedRegistration = registrationService.saveRegistration(deviceRegistration);
//		assertNotNull(savedRegistration);
//		assertEquals(false, savedRegistration.isOptedIn());
//	}

	@Test
	public void fetchDeviceRegistrationsByUserIds() {
		DeviceRegistrations fetchedDeviceRegistrations = deviceRegistrationDataService.fetchRegistrations(userIds);
		assertEquals(deviceRegistrations.size(), fetchedDeviceRegistrations.size());
	}

	@Test
	public void deleteDeviceRegistrationByUserIdAndToken() {
		boolean testResult = true;
		try {
			deviceRegistrationDataService.deleteRegistration(deviceRegistration.getUserId(), deviceRegistration.getDeviceToken());
		} catch (WebApplicationException e) {
			if (e.getResponse().getStatus() == Status.NOT_FOUND.getStatusCode()) {
				testResult = false;
			}
		} finally {
			assertTrue(testResult);
		}
	}
	
	@Test
	public void deleteNonexistentDeviceRegistration() {
		boolean testResult = false;
		try {
			deviceRegistrationDataService.deleteRegistration(nonexistentDeviceRegistration.getUserId(), nonexistentDeviceRegistration.getDeviceToken());
		} catch (WebApplicationException e) {
			if (e.getResponse().getStatus() == Status.NOT_FOUND.getStatusCode()) {
				testResult = true;
			}
		} finally {
			assertTrue(testResult);
		}
	}
	
	@Test
	public void cleanUpDeviceTokens() {
		deviceRegistrationDataService.cleanUpDeviceTokens();
	}
	
	@Test
	public void retrieveOfDeviceRegistrationsWithNoResultsFromDao() {
		DeviceRegistrationDataService deviceRegistrationDataServiceWithNoMockData = setupDataServiceWithNoMockData();
		DeviceRegistrations deviceRegistrations = deviceRegistrationDataServiceWithNoMockData.fetchRegistrations(TEST_USER_ID);
		assertEquals(0, deviceRegistrations.size());
	}

	@Test
	public void retrieveOfDeviceRegistrationsWithResults() {
		DeviceRegistrations deviceRegistrations = deviceRegistrationDataService.fetchRegistrations(TEST_USER_ID);
		assertEquals(1, deviceRegistrations.size());
		
		DeviceRegistration deviceRegistration = deviceRegistrationDataService.fetchRegistration(TEST_USER_ID, TEST_DEVICE_TOKEN);
		assertEquals(deviceRegistration.getDeviceToken(), TEST_DEVICE_TOKEN);
		assertEquals(deviceRegistration.getUserId(), TEST_USER_ID);
	}
	
	private DeviceRegistrationDataService setupDataServiceWithMockData() {
		deviceRegistrations = new DeviceRegistrations();
		deviceRegistrations.add(savedDeviceRegistration);
		
		userIds = new LinkedList<String>();
		userIds.add(TEST_USER_ID);

		DeviceRegistrationDataLayer mockDao = mock(DeviceRegistrationDataLayer.class);
		when(mockDao.getRegistrations(TEST_USER_ID)).thenReturn(deviceRegistrations);
		when(mockDao.getRegistration(TEST_USER_ID, TEST_DEVICE_TOKEN)).thenReturn(savedDeviceRegistration);
		doThrow(new WebApplicationException(Status.NOT_FOUND)).when(mockDao).getRegistration(TEST_USER_ID, TEST_DEVICE_TOKEN2);
		when(mockDao.fetchDeviceRegistrations(userIds)).thenReturn(deviceRegistrations);
		when(mockDao.saveRegistration(deviceRegistration)).thenReturn(savedDeviceRegistration);
		when(mockDao.saveRegistration(nonexistentDeviceRegistration)).thenReturn(savedNonexistentDeviceRegistration);
		
		DeviceRegistrationDataService deviceRegistrationDataService = new DeviceRegistrationDataService(mockDao, setupNotificationCommunicationDataLayerWithMockData());
		return deviceRegistrationDataService;
	}

	private DeviceRegistrationDataService setupDataServiceWithNoMockData() {
		DeviceRegistrations deviceRegistrations = new DeviceRegistrations();
		DeviceRegistrationDataLayer mockDao = mock(DeviceRegistrationDataLayer.class);
		when(mockDao.getRegistrations(TEST_USER_ID)).thenReturn(deviceRegistrations);

		DeviceRegistrationDataService deviceRegistrationDataService = new DeviceRegistrationDataService(mockDao, setupNotificationCommunicationDataLayerWithMockData());
		return deviceRegistrationDataService;
	}

	private NotificationCommunicationDataLayer setupNotificationCommunicationDataLayerWithMockData() {
		NotificationCommunicationDataLayer mockDao = mock(NotificationCommunicationDataLayer.class);
		String[] deviceTokens = { TEST_DEVICE_TOKEN };
		when(mockDao.send(deviceTokens, TEST_NOTIFICATION_BODY)).thenReturn(new LinkedList<String>());
		
		List<String> invalidDeviceTokens = new LinkedList<String>();
		invalidDeviceTokens.add(TEST_DEVICE_TOKEN);
		when(mockDao.fetchInvalidDeviceTokens()).thenReturn(invalidDeviceTokens);
		
		return mockDao;
	}
	
	private DeviceRegistration createNewDeviceRegistration(String userId, String deviceToken) {
		DeviceRegistration newRegistration = new DeviceRegistration(userId, deviceToken);
		newRegistration.setOptedIn(true);
		return newRegistration;
	}
}
