package com.agilex.healthcare.mobilehealthplatform.datalayer.medication;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.MockRouterHelper;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.Router;
import com.agilex.healthcare.mobilehealthplatform.domain.Medication;
import com.agilex.healthcare.mobilehealthplatform.domain.Medications;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.utility.DateHelper;

public class MedicationDataServiceTest {
	private final String patientId = "1";
	private String drugName = "myddrug";
	private String medDetail = "Sample Medication Detail";
	private MedicationFilter emptyFilter = new MedicationFilter();

	@Test
	public void retrieveOfMedsWithResults() {
		MedicationDataService medicationDataService = setupDataServiceWithMockData(null, drugName, emptyFilter);
		Medications retrievedMeds = medicationDataService.fetchPatientMedications(getPatientIdentifier(), emptyFilter, ScopeFilter.getInstanceForLongitudinalScope());
		checkMedResult(retrievedMeds, 1, drugName);
	}

	@Test
	public void retrieveOfMedsWithNoResultsFromDao() {
		MedicationDataService medicationDataService = setupDataServiceWithNoMockData();
		Medications retrievedMeds = medicationDataService.fetchPatientMedications(getPatientIdentifier(), new MedicationFilter(), ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(0, retrievedMeds.size());
	}

	@Test
	public void retrieveOfMedsWithNullFromDao() {
		MedicationDataService medicationDataService = setupDataServiceWithNullMockData();
		Medications retrievedMeds = medicationDataService.fetchPatientMedications(getPatientIdentifier(), new MedicationFilter(), ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(0, retrievedMeds.size());
	}

	private MedicationDataService setupDataServiceWithMockData(final String medicationSource, String medName, MedicationFilter filter) {
		Medications medications = new Medications();
		Medication medication = new Medication();
		medication.setDrugName(medName);
		medication.setMedicationSource(medicationSource);
		medication.setStatus("active");
		medication.setExpiredDate(DateHelper.get30DaysFromNow());
		medications.add(medication);

		Router mockRouter = MockRouterHelper.createMockRouterToReturnPatientDataList(medications);
		MedicationDataService medicationDataService = new MedicationDataService(mockRouter);
		return medicationDataService;
	}

	private MedicationDataService setupDataServiceWithNoMockData() {
		Medications medications = new Medications();
		Router mockRouter = MockRouterHelper.createMockRouterToReturnPatientDataList(medications);
		MedicationDataService medicationDataService = new MedicationDataService(mockRouter);
		return medicationDataService;
	}

	private MedicationDataService setupDataServiceWithNullMockData() {
		Medications medications = null;
		Router mockRouter = MockRouterHelper.createMockRouterToReturnPatientDataList(medications);
		MedicationDataService medicationDataService = new MedicationDataService(mockRouter);
		return medicationDataService;
	}

	private void checkMedResult(Medications retrievedMeds, int expectedSize, String expectedDrugName) {
		assertEquals(1, retrievedMeds.size());
	}

	@Test
	public void validateMedicationDetailExists() {
		MedicationDataService medicationDataService = setupDataServiceWithMockDataForMedicationDetail(null, drugName, emptyFilter);
		Medications retrievedMeds = medicationDataService.fetchPatientMedications(getPatientIdentifier(), emptyFilter, ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(drugName, retrievedMeds.get(0).getDrugName());
		assertEquals(medDetail, retrievedMeds.get(0).getMedicationDetail());
	}

	@Test
	public void validateMedicationDetailExistsForActiveMeds() {
		MedicationDataService medicationDataService = setupDataServiceWithMockDataForMedicationDetail("OUTPATIENT", drugName, emptyFilter);
		Medications retrievedMeds = medicationDataService.getActivePatientMedications(getPatientIdentifier(), emptyFilter, ScopeFilter.getInstanceForLongitudinalScope());
		assertTrue(retrievedMeds.size() > 0);
		assertEquals(drugName, retrievedMeds.get(0).getDrugName());
		assertEquals(medDetail, retrievedMeds.get(0).getMedicationDetail());
	}

	private MedicationDataService setupDataServiceWithMockDataForMedicationDetail(final String medicationSource, String medName, MedicationFilter filter) {
		Medications medications = new Medications();
		Medication medication = new Medication();
		medication.setDrugName(medName);
		medication.setMedicationSource(medicationSource);
		medication.setActive(true);
		medication.setMedicationDetail(medDetail);
		medication.setExpiredDate(DateHelper.get30DaysFromNow());
		medications.add(medication);

		Router mockRouter = MockRouterHelper.createMockRouterToReturnPatientDataList(medications);
		MedicationDataService medicationDataService = new MedicationDataService(mockRouter);
		return medicationDataService;
	}

	private PatientIdentifier getPatientIdentifier() {
		return new PatientIdentifier("default", patientId);
	}

}
