package com.agilex.healthcare.mobilehealthplatform.datalayer.lab.micro;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.MockRouterHelper;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.Router;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResult;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResults;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.utility.DateHelper;

public class MicrobiologyLabDataServiceRetrieveResultsByTestAndSpecimenTest {
	PatientIdentifier patientIdentifier = new PatientIdentifier("a", "1");
	PatientIdentifier patientIdentifierZeroResults = new PatientIdentifier("a", "2");
	PatientIdentifier patientIdentifierNullResults = new PatientIdentifier("a", "3");

	@Test
	public void retrieveTwoLabs() {
		MicrobiologyLabDataService labdataservice = initializeMockWithData();
		LabResults retrievedResults = labdataservice.getLabResultsByTestAndSpecimen(patientIdentifier, "TestA", "SpecimenA", null);
		assertContainsResultByOrder(retrievedResults, "O1");
		assertContainsResultByOrder(retrievedResults, "O2");
		assertContainsResultByOrder(retrievedResults, "O3");
		assertDoesNotContainResultByOrder(retrievedResults, "O4");
		assertDoesNotContainResultByOrder(retrievedResults, "O4");
	}

	@Test
	public void retrieveOneLab() {
		MicrobiologyLabDataService labdataservice = initializeMockWithData();
		LabResults retrievedResults = labdataservice.getLabResultsByTestAndSpecimen(patientIdentifier, "TestB", "SpecimenA", null);
		assertDoesNotContainResultByOrder(retrievedResults, "O1");
		assertDoesNotContainResultByOrder(retrievedResults, "O2");
		assertDoesNotContainResultByOrder(retrievedResults, "O3");
		assertContainsResultByOrder(retrievedResults, "O4");
		assertDoesNotContainResultByOrder(retrievedResults, "O5");
	}

	@Test
	public void filterToZero() {
		MicrobiologyLabDataService labdataservice = initializeMockWithData();
		LabResults retrievedResults = labdataservice.getLabResultsByTestAndSpecimen(patientIdentifier, "X", "X", null);
		assertEquals(0, retrievedResults.size());
	}

	@Test
	public void retrieveZero() {
		MicrobiologyLabDataService labdataservice = initializeMockWithNoData();
		LabResults retrievedResults = labdataservice.getLabResultsByTestAndSpecimen(patientIdentifierZeroResults, "X", "X", null);
		assertEquals(0, retrievedResults.size());
	}

	@Test
	public void retrieveNull() {
		MicrobiologyLabDataService labdataservice = initializeMockWithNullData();
		LabResults retrievedResults = labdataservice.getLabResultsByTestAndSpecimen(patientIdentifierNullResults, "X", "X", null);
		assertEquals(0, retrievedResults.size());
	}

	private void assertContainsResultByOrder(LabResults labResults, String orderId) {
		LabResult result = findLabResultByOrderId(labResults, orderId);
		assertNotNull(String.format("Unable to find lab result with order id %s in this result set", orderId), result);
	}

	private void assertDoesNotContainResultByOrder(LabResults labResults, String orderId) {
		LabResult result = findLabResultByOrderId(labResults, orderId);
		assertNull(String.format("Unexpectedly found lab result with order id %s in this result set", orderId), result);
	}

	private LabResult findLabResultByOrderId(LabResults labResults, String orderId) {
		LabResult match = null;
		for (LabResult result : labResults)
			if (result.getOrderId().contentEquals(orderId))
				match = result;
		return match;
	}

	@Test
	public void verifyDisplayName() {
		MicrobiologyLabDataService labdataservice = initializeMockWithData();
		LabResults retrievedResults = labdataservice.getLabResultsByTestAndSpecimen(patientIdentifier, "TestA", "SpecimenA", null);
		LabResult result = findLabResultByOrderId(retrievedResults, "O1");
		assertEquals("Test A (Specimen A)", result.getDisplayName());
	}

	@Test
	public void verifyDisplayNameWithoutSpecimen() {
		MicrobiologyLabDataService labdataservice = initializeMockWithData();
		LabResults retrievedResults = labdataservice.getLabResultsByTestAndSpecimen(patientIdentifier, "TestA", "SpecimenA", null);
		LabResult result = findLabResultByOrderId(retrievedResults, "O2");
		assertEquals("Test A", result.getDisplayName());
	}

	@Test
	public void verifyDisplayNameWithoutTest() {
		MicrobiologyLabDataService labdataservice = initializeMockWithData();
		LabResults retrievedResults = labdataservice.getLabResultsByTestAndSpecimen(patientIdentifier, "TestA", "SpecimenA", null);
		LabResult result = findLabResultByOrderId(retrievedResults, "O3");
		assertEquals("Specimen A", result.getDisplayName());
	}

	private MicrobiologyLabDataService initializeMockWithData() {
		LabResults labresults = new LabResults();
		LabResult labresult;

		labresult = new LabResult();
		labresult.setPatientIdentifier(patientIdentifier);
		labresult.setTestId("TestA");
		labresult.setTestname("Test A");
		labresult.setSpecimenId("SpecimenA");
		labresult.setSpecimenName("Specimen A");
		labresult.setOrderId("O1");
		labresult.setResultedDate(DateHelper.parseDate("1/1/2010"));
		labresults.add(labresult);

		labresult = new LabResult();
		labresult.setPatientIdentifier(patientIdentifier);
		labresult.setTestId("TestA");
		labresult.setTestname("Test A");
		labresult.setSpecimenId("SpecimenA");
		labresult.setOrderId("O2");
		labresult.setResultedDate(DateHelper.parseDate("1/1/2010"));
		labresults.add(labresult);

		labresult = new LabResult();
		labresult.setPatientIdentifier(patientIdentifier);
		labresult.setTestId("TestA");
		labresult.setSpecimenId("SpecimenA");
		labresult.setSpecimenName("Specimen A");
		labresult.setOrderId("O3");
		labresult.setResultedDate(DateHelper.parseDate("1/1/2010"));
		labresults.add(labresult);

		labresult = new LabResult();
		labresult.setPatientIdentifier(patientIdentifier);
		labresult.setTestId("TestB");
		labresult.setSpecimenId("SpecimenA");
		labresult.setOrderId("O4");
		labresult.setResultedDate(DateHelper.parseDate("1/1/2010"));
		labresults.add(labresult);

		labresult = new LabResult();
		labresult.setPatientIdentifier(patientIdentifier);
		labresult.setTestId("TestA");
		labresult.setSpecimenId("SpecimenB");
		labresult.setOrderId("O5");
		labresult.setResultedDate(DateHelper.parseDate("1/1/2010"));
		labresults.add(labresult);

		Router router = MockRouterHelper.createMockRouterToReturnPatientDataList(labresults);
		MicrobiologyLabDataService labdataservice = new MicrobiologyLabDataService(router);
		return labdataservice;
	}

	private MicrobiologyLabDataService initializeMockWithNoData() {
		LabResults labresults = new LabResults();
		Router router = MockRouterHelper.createMockRouterToReturnPatientDataList(labresults);
		MicrobiologyLabDataService labdataservice = new MicrobiologyLabDataService(router);
		return labdataservice;
	}

	private MicrobiologyLabDataService initializeMockWithNullData() {
		LabResults labresults = null;
		Router router = MockRouterHelper.createMockRouterToReturnPatientDataList(labresults);
		MicrobiologyLabDataService labdataservice = new MicrobiologyLabDataService(router);
		return labdataservice;

		// MicrobiologyLabDataService labdataservice = new
		// MicrobiologyLabDataService(mockDao,
		// LabDataServiceTestHelper.getMockedLabTestGroupDataLayer());
		// return labdataservice;
	}

}
