package com.agilex.healthcare.mobilehealthplatform.datalayer.lab;

import static org.junit.Assert.assertEquals;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.LabResult;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResults;
import com.agilex.healthcare.utility.TriState;

public class AbnormalityFlagCalculatorTest {
	@Test
	public void high() {
		final String indicator = "H";
		final String value = null;
		final String refRangeHigh = null;
		final String refRangeLow = null;
		final boolean expectedAbnormalityFlag = true;
		final TriState expectedAbnormalityTriFlag = TriState.TRUE;
		runBinaryAbnormalityTest(indicator, value, refRangeHigh, refRangeLow, expectedAbnormalityFlag, expectedAbnormalityTriFlag);
	}

	@Test
	public void low() {
		final String indicator = "L";
		final String value = null;
		final String refRangeHigh = null;
		final String refRangeLow = null;
		final boolean expectedAbnormalityFlag = true;
		final TriState expectedAbnormalityTriFlag = TriState.TRUE;
		runBinaryAbnormalityTest(indicator, value, refRangeHigh, refRangeLow, expectedAbnormalityFlag, expectedAbnormalityTriFlag);
	}

	@Test
	public void anyCharInIndicatorWithValueInRange() {
		final String indicator = "blah!";
		final String value = "5";
		final String refRangeHigh = "6";
		final String refRangeLow = "4";
		final boolean expectedAbnormalityFlag = true;
		final TriState expectedAbnormalityTriFlag = TriState.TRUE;
		runBinaryAbnormalityTest(indicator, value, refRangeHigh, refRangeLow, expectedAbnormalityFlag, expectedAbnormalityTriFlag);
	}

	@Test
	public void normal() {
		final String indicator = "";
		final String value = "2";
		final String refRangeHigh = "3";
		final String refRangeLow = "1";
		final boolean expectedAbnormalityFlag = false;
		final TriState expectedAbnormalityTriFlag = TriState.FALSE;
		runBinaryAbnormalityTest(indicator, value, refRangeHigh, refRangeLow, expectedAbnormalityFlag, expectedAbnormalityTriFlag);
	}

	@Test
	public void nullIndicator() {
		final String indicator = null;
		final String value = null;
		final String refRangeHigh = null;
		final String refRangeLow = null;
		final boolean expectedAbnormalityFlag = false;
		final TriState expectedAbnormalityTriFlag = TriState.UNKNOWN;
		runBinaryAbnormalityTest(indicator, value, refRangeHigh, refRangeLow, expectedAbnormalityFlag, expectedAbnormalityTriFlag);
	}

	@Test
	public void inRangeNumeric() {
		final String indicator = "";
		final String value = "5";
		final String refRangeHigh = "1";
		final String refRangeLow = "10";
		final boolean expectedAbnormalityFlag = false;
		final TriState expectedAbnormalityTriFlag = TriState.FALSE;
		runBinaryAbnormalityTest(indicator, value, refRangeHigh, refRangeLow, expectedAbnormalityFlag, expectedAbnormalityTriFlag);
	}

	@Test
	public void inRangeNumericNonsenseRange() {
		// like above test, but i am making it clear that we don't test the
		// ranges
		// it is considered normal because if not marked but EMR as abnormal
		// and value/range are numeric, then considered 'normal'
		final String indicator = "";
		final String value = "5";
		final String refRangeHigh = "1";
		final String refRangeLow = "2";
		final boolean expectedAbnormalityFlag = false;
		final TriState expectedAbnormalityTriFlag = TriState.FALSE;
		runBinaryAbnormalityTest(indicator, value, refRangeHigh, refRangeLow, expectedAbnormalityFlag, expectedAbnormalityTriFlag);
	}

	@Test
	public void unknownAbnormalityWithNonNumericValue() {
		final String indicator = "";
		final String value = "?";
		final String refRangeHigh = "1";
		final String refRangeLow = "2";
		final boolean expectedAbnormalityFlag = false;
		final TriState expectedAbnormalityTriFlag = TriState.UNKNOWN;
		runBinaryAbnormalityTest(indicator, value, refRangeHigh, refRangeLow, expectedAbnormalityFlag, expectedAbnormalityTriFlag);
	}

	@Test
	public void unknownAbnormalityWithNonNumericHigh() {
		final String indicator = "";
		final String value = "5";
		final String refRangeHigh = "?";
		final String refRangeLow = "2";
		final boolean expectedAbnormalityFlag = false;
		final TriState expectedAbnormalityTriFlag = TriState.UNKNOWN;
		runBinaryAbnormalityTest(indicator, value, refRangeHigh, refRangeLow, expectedAbnormalityFlag, expectedAbnormalityTriFlag);
	}

	@Test
	public void unknownAbnormalityWithNonNumericLow() {
		final String indicator = "";
		final String value = "5";
		final String refRangeHigh = "1";
		final String refRangeLow = "?";
		final boolean expectedAbnormalityFlag = false;
		final TriState expectedAbnormalityTriFlag = TriState.UNKNOWN;
		runBinaryAbnormalityTest(indicator, value, refRangeHigh, refRangeLow, expectedAbnormalityFlag, expectedAbnormalityTriFlag);
	}

	@Test
	public void unknownAbnormalityWithOperatorInRange() {
		final String indicator = "";
		final String value = "5";
		final String refRangeHigh = "<10";
		final String refRangeLow = ">1";
		final boolean expectedAbnormalityFlag = false;
		final TriState expectedAbnormalityTriFlag = TriState.UNKNOWN;
		runBinaryAbnormalityTest(indicator, value, refRangeHigh, refRangeLow, expectedAbnormalityFlag, expectedAbnormalityTriFlag);
	}

	@Test
	public void unknownAbnormalityWithGreaterThanRef() {
		final String indicator = "";
		final String value = "10";
		final String refRangeHigh = "";
		final String refRangeLow = ">5";
		final boolean expectedAbnormalityFlag = false;
		final TriState expectedAbnormalityTriFlag = TriState.UNKNOWN;
		runBinaryAbnormalityTest(indicator, value, refRangeHigh, refRangeLow, expectedAbnormalityFlag, expectedAbnormalityTriFlag);
	}

	@Test
	public void unknownAbnormalityWithLessThanRef() {
		final String indicator = "";
		final String value = "10";
		final String refRangeHigh = "<15";
		final String refRangeLow = "";
		final boolean expectedAbnormalityFlag = false;
		final TriState expectedAbnormalityTriFlag = TriState.UNKNOWN;
		runBinaryAbnormalityTest(indicator, value, refRangeHigh, refRangeLow, expectedAbnormalityFlag, expectedAbnormalityTriFlag);
	}

	private void runBinaryAbnormalityTest(final String indicator, String value, String refRangeHigh, String refRangeLow, final boolean expectedAbnormalityFlag, TriState expectedAbnormalityTriFlag) {
		LabResult lab = new LabResult();
		lab.setIndicator(indicator);
		lab.setValue(value);
		lab.setReferenceLow(refRangeLow);
		lab.setReferenceHigh(refRangeHigh);
		LabResults labs = new LabResults();
		labs.add(lab);

		AbnormalityFlagCalculator calc = new AbnormalityFlagCalculator();
		calc.calculateAndUpdateResultAbnormalityFlag(labs);

		boolean resultFlag = labs.get(0).isAbnormalIndicatorFlag();
		assertEquals(expectedAbnormalityFlag, resultFlag);

		TriState resultTriFlag = labs.get(0).getAbnormalIndicatorTriFlag();
		assertEquals(expectedAbnormalityTriFlag, resultTriFlag);
	}

	@Test
	public void nullInput() {
		LabResults labResults = null;

		AbnormalityFlagCalculator calc = new AbnormalityFlagCalculator();
		calc.calculateAndUpdateResultAbnormalityFlag(labResults);
	}
}
