package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.util.Date;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentProperties;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentProperty;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestion;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestionChoice;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestionChoices;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResponse;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResponses;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResult;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResultReport;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssessmentCode;
import com.agilex.healthcare.utility.DateHelper;

public class ZaritReportGeneratorTest {
	private ZaritReportGenerator reportGenerator = new ZaritReportGenerator();

	@Test
	public void testSimpleReport() {
		AssessmentResult result = generateMockAssessmentResult(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, null);

		checkForNull(resultReport);
		assertNotNull(resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY));
		assertEquals(0,resultReport.getToolCategories().size());
	}

	@Test
	public void testReportWithHighScoringAnswers() {
		AssessmentResult result = generateMockAssessmentResult(1, 5, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0);
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, null);

		checkForNull(resultReport);
		assertNotNull(resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY));
		assertNotNull(resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_HIGH_SCORING_QUESTIONS_EXPLANATION));
		assertTrue(resultReport.getToolCategories().containsTool(ZaritToolCalculator.RELAX));
	}

	@Test
	public void testReportWithOnlyOneHighScoringAnswer() {
		AssessmentResult result = generateMockAssessmentResult(0, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, null);

		checkForNull(resultReport);
		assertNotNull(resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY));
		assertNotNull(resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_HIGH_SCORING_QUESTIONS_EXPLANATION));
		assertEquals(resultReport.getToolCategories().get(0), ZaritToolCalculator.RELAX);
		assertEquals(resultReport.getToolCategories().get(1), ZaritToolCalculator.SUPPORT);
	}

	@Test
	public void testCutoffsWithOnlyOneHighScoringAnswerHasCorrectCounts() {
		AssessmentResult result = generateMockAssessmentResult(0, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, null);

		assertEquals(1, resultReport.getToolCategories().get(0).getRecommendationCount());
		assertEquals(1, resultReport.getToolCategories().get(1).getRecommendationCount());
	}

	@Test
	public void testReportWithTwoHighScoringAnswers() {
		AssessmentResult result = generateMockAssessmentResult(0, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5);
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, null);

		checkForNull(resultReport);
		assertNotNull(resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY));
		assertNotNull(resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_HIGH_SCORING_QUESTIONS_EXPLANATION));
		assertEquals(resultReport.getToolCategories().get(0), ZaritToolCalculator.RELAX);
		assertEquals(resultReport.getToolCategories().get(1), ZaritToolCalculator.SUPPORT);
		assertEquals(resultReport.getToolCategories().get(2), ZaritToolCalculator.CHANGE);
	}

	@Test
	public void testCutoffsWithTwoHighScoringAnswerHasCorrectCounts() {
		AssessmentResult result = generateMockAssessmentResult(0, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5);
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, null);

		assertEquals(2, resultReport.getToolCategories().get(0).getRecommendationCount());
		assertEquals(2, resultReport.getToolCategories().get(1).getRecommendationCount());
		assertEquals(1, resultReport.getToolCategories().get(2).getRecommendationCount());
	}

	@Test
	public void testReportWithZeroScore() {
		AssessmentResult currentResult = generateMockAssessmentResult(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
		AssessmentResultReport resultReport = reportGenerator.generateReport(currentResult, null);

		checkForNull(resultReport);
		String resultSummary = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		assertTrue(resultSummary.contains("not causing you significant distress"));
		assertNull(resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_HIGH_SCORING_QUESTIONS_EXPLANATION));
	}

	@Test
	public void testReportWithLowScore() {
		AssessmentResult currentResult = generateMockAssessmentResult(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1);
		AssessmentResultReport resultReport = reportGenerator.generateReport(currentResult, null);

		checkForNull(resultReport);
		String resultSummary = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		assertTrue(resultSummary.contains("low"));
		assertNull(resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_HIGH_SCORING_QUESTIONS_EXPLANATION));
		assertNotNull(resultSummary);
	}

	@Test
	public void testReportWithMediumScore() {
		AssessmentResult currentResult = generateMockAssessmentResult(1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1);
		AssessmentResultReport resultReport = reportGenerator.generateReport(currentResult, null);

		checkForNull(resultReport);
		String resultSummary = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		assertTrue(resultSummary.contains("medium"));
		assertNull(resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_HIGH_SCORING_QUESTIONS_EXPLANATION));
	}

	@Test
	public void testReportWithHighScore() {
		AssessmentResult currentResult = generateMockAssessmentResult(5, 5, 1, 5, 5, 5, 5, 5, 5, 5, 5, 5);
		AssessmentResultReport resultReport = reportGenerator.generateReport(currentResult, null);

		checkForNull(resultReport);
		String resultSummary = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		assertTrue(resultSummary.contains("considerable amount"));
		assertNotNull(resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_HIGH_SCORING_QUESTIONS_EXPLANATION));
		assertTrue(resultReport.getToolCategories().contains(ZaritToolCalculator.RELAX));
		assertTrue(resultReport.getToolCategories().contains(ZaritToolCalculator.PLAN));
		assertTrue(resultReport.getToolCategories().contains(ZaritToolCalculator.PLEASANT));
	}

	@Test
	public void testReportWithIncreaseInScore() {
		AssessmentResult previousResult = generatePreviousResult(1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1);
		AssessmentResult currentResult = generateMockAssessmentResult(3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3);
		AssessmentResultReport resultReport = reportGenerator.generateReport(currentResult, previousResult);

		checkForNull(resultReport);
		String resultSummary = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		assertTrue(resultSummary.contains("This is an increase of 24 since your last assessment"));
		assertNotNull(resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_HIGH_SCORING_QUESTIONS_EXPLANATION));
	}

	@Test
	public void testReportWithSignificantIncreaseInScore() {
		AssessmentResult previousResult = generatePreviousResult(1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1);
		AssessmentResult currentResult = generateMockAssessmentResult(5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5);
		AssessmentResultReport resultReport = reportGenerator.generateReport(currentResult, previousResult);

		checkForNull(resultReport);

		String resultSummary = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		assertTrue(resultSummary.contains("There has been a significant increase"));
		assertNotNull(resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_HIGH_SCORING_QUESTIONS_EXPLANATION));
	}

	private void checkForNull(AssessmentResultReport resultReport) {
		assertNotNull(resultReport);
		assertNotNull(resultReport.getProperties());
	}

	private AssessmentResult generatePreviousResult(int... responsePointValues) {
		AssessmentResult previousResult = generateMockAssessmentResult(responsePointValues);
		Date yesterday = DateHelper.getYesterday();
		previousResult.setDateTaken(yesterday);
		return previousResult;
	}

	private AssessmentResult generateMockAssessmentResult(int... responsePointValues) {
		AssessmentResult mockResult = new AssessmentResult();
		mockResult.setAssessmentId("mockId");
		mockResult.setScoringAlgorithm(AssessmentCode.SCORING_ALGORITHM_ZARIT);
		mockResult.setResponses(generateMockResponses(responsePointValues));
		int totalScore = 0;
		for (int responsePointValue : responsePointValues) {
			totalScore += responsePointValue;
		}
		mockResult.setScore(totalScore);
		return mockResult;
	}

	private AssessmentResponses generateMockResponses(int... responsePointValues) {
		AssessmentResponses assessmentResponses = new AssessmentResponses();
		int questionNumber = 1;
		for (int responsePointValue : responsePointValues) {
			assessmentResponses = addResponse(assessmentResponses, responsePointValue, questionNumber);
			questionNumber++;
		}
		return assessmentResponses;
	}

	private AssessmentResponses addResponse(AssessmentResponses assessmentResponses, int pointValue, int questionNumber) {
		AssessmentResponse assessmentResponse = new AssessmentResponse();
		AssessmentQuestionChoices selectedChoices = new AssessmentQuestionChoices();
		AssessmentQuestionChoice selectedChoice = new AssessmentQuestionChoice();
		AssessmentProperties properties = new AssessmentProperties();
		AssessmentQuestion assessmentQuestion = new AssessmentQuestion();

		properties.add(new AssessmentProperty(AssessmentCode.RESULT_PROPERTY_POINT_VALUE, "" + pointValue));
		selectedChoice.setProperties(properties);
		selectedChoices.add(selectedChoice);
		assessmentResponse.setSelectedChoices(selectedChoices);

		assessmentQuestion.setQuestionNumber(questionNumber);
		assessmentResponse.setQuestion(assessmentQuestion);

		assessmentResponses.add(assessmentResponse);
		return assessmentResponses;
	}
}
