package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults;

import com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.reports.PainConsolidatedReportGenerator;
import com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.reports.PainConsolidatedReportGeneratorParameter;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssessmentCode;
import com.agilex.healthcare.mobilehealthplatform.test.AssertEx;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.PainGoalEntryStatus;
import com.agilex.healthcare.utility.PainGoalStatus;
import org.junit.*;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import static org.junit.Assert.*;

public class PainConsolidatedReportGeneratorTest {

    @Test
    public void testReportWithMultipleQuestionsPerMonth() {


        AssessmentResults results = new AssessmentResults();

        AssessmentResult result1 = MonthlyPainReportTestHelper.generateResult("01/01/2012", 4, 4, 4, 4, 4, 4, 1, 1, 4, 1, 1, 1, 6);
        AssessmentResult result2 = MonthlyPainReportTestHelper.generateResult("01/03/2012", 4, 4, 4, 4, 4, 4, 1, 1, 4, 1, 1, 1, 6);
        AssessmentResult result3 = MonthlyPainReportTestHelper.generateResult("01/05/2012", 4, 4, 4, 4, 4, 4, 1, 1, 4, 1, 1, 1, 6);
        AssessmentResult result4 = MonthlyPainReportTestHelper.generateResult("01/07/2012", 4, 4, 4, 4, 4, 4, 1, 1, 4, 1, 1, 1, 6);
        AssessmentResult result5 = MonthlyPainReportTestHelper.generateResult("01/09/2012", 4, 4, 4, 4, 4, 4, 1, 1, 4, 1, 1, 1, 6);
        AssessmentResult result6 = MonthlyPainReportTestHelper.generateResult("02/09/2012", 4, 4, 4, 4, 4, 4, 1, 1, 4, 1, 1, 1, 6);

        results.add(result1);
        results.add(result2);
        results.add(result3);
        results.add(result4);
        results.add(result5);
        results.add(result6);

        PainDiaryEntries painDiaryEntries = new PainDiaryEntries();

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);
        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_MONTHLYPAINASSESSMENTS);

        //Should only get not get entries for Duplicate months
        assertFalse(reportText.contains("<td style=\"color: #ffffff;\">Jan</td><td style=\"color: #ffffff;\">Jan</td>"));
        assertTrue(reportText.contains("<td style=\"color: #ffffff;\">Jan</td><td style=\"color: #ffffff;\">Feb</td>"));
    }

    //Prove fix for defect where a month would show up multiple times in some scenarios
    @Test
    public void testReportWithMultipleQuestionsPerMonthOnlyOnePerMonth() {


        AssessmentResults results = new AssessmentResults();

        AssessmentResult result1 = MonthlyPainReportTestHelper.generateResult("01/15/2012", 4, 4, 4, 4, 4, 4, 1, 1, 4, 1, 1, 1, 6);
        AssessmentResult result5 = MonthlyPainReportTestHelper.generateResult("02/02/2012", 4, 4, 4, 4, 4, 4, 1, 1, 4, 1, 1, 1, 6);
        AssessmentResult result6 = MonthlyPainReportTestHelper.generateResult("12/02/2011", 4, 4, 4, 4, 4, 4, 1, 1, 4, 1, 1, 1, 6);

        results.add(result1);
        results.add(result5);
        results.add(result6);

        PainDiaryEntries painDiaryEntries = new PainDiaryEntries();

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);
        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_MONTHLYPAINASSESSMENTS);

        //Should only get not get entries for Duplicate months
        assertFalse(reportText.contains("<td style=\"color: #ffffff;\">Jan</td><td style=\"color: #ffffff;\">Jan</td>"));
    }

    @Test
    public void testReportWithMultipleQuestionsInEachChangeSection() {
        AssessmentResult result = MonthlyPainReportTestHelper.generateMockAssessmentResult(4, 4, 4, 4, 4, 4, 1, 1, 4, 1, 1, 1, 6);
        AssessmentResult previousResult = MonthlyPainReportTestHelper.generatePreviousResult(4, 3, 5, 4, 3, 5, 2, 2, 6, 2, 2, 2, 4);
        AssessmentResults results = new AssessmentResults();
        results.add(result);
        results.add(previousResult);

        PainDiaryEntries painDiaryEntries = new PainDiaryEntries();

        Date takenDate = DateHelper.parseDate("01/01/2012");
        previousResult.setDateTaken(takenDate);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);
        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_MONTHLYPAINASSESSMENTS);

        assertNotNull(reportText);
        assertTrue(reportText.contains("Monthly Pain Assessment Report"));
        assertTrue(reportText.contains("Showing data from 01/01/2012 to 06/01/2012"));
    }

    @Test
    public void testReportCreatesHeader() {

        AssessmentResult result = MonthlyPainReportTestHelper.generateMockAssessmentResult(2, null);
        AssessmentResult previousResult = MonthlyPainReportTestHelper.generatePreviousResult(3, 3);
        AssessmentResults results = new AssessmentResults();
        results.add(result);
        results.add(previousResult);

        PainDiaryEntries painDiaryEntries = new PainDiaryEntries();


        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_HEADER);

        assertTrue(reportText.contains("Patient:      , John"));
    }

    @Test
    public void testReportWithNullScore() {

        AssessmentResult result = MonthlyPainReportTestHelper.generateMockAssessmentResult(2, null);
        AssessmentResult previousResult = MonthlyPainReportTestHelper.generatePreviousResult(3, 3);
        AssessmentResults results = new AssessmentResults();
        results.add(result);
        results.add(previousResult);

        PainDiaryEntries painDiaryEntries = new PainDiaryEntries();


        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_MONTHLYPAINASSESSMENTS);

        assertNotNull(resultReport);
        assertNotNull(reportText);
        assertTrue(reportText.contains("Monthly Pain Assessment Report"));
    }

    private PainConsolidatedReportGenerator createPainConsolidatedReportGenerator(AssessmentResults results, PainDiaryEntries painDiaryEntries) {
        PainGoals painGoals = null;
        ToolTrackingResults toolTrackingResults = null;
        return createPainConsolidatedReportGenerator(results, painDiaryEntries, painGoals, toolTrackingResults);
    }

    private PainConsolidatedReportGenerator createPainConsolidatedReportGenerator(AssessmentResults results, PainDiaryEntries painDiaryEntries, PainGoals painGoals, ToolTrackingResults toolTrackingResults) {
        Patient patient = new Patient();
        patient.setDisplayName("     , John");
        Date startDate = DateHelper.parseDate("01/01/2012");
        Date endDate = DateHelper.parseDate("06/01/2012");

        return createPainConsolidatedReportGenerator(patient, startDate, endDate, results, painDiaryEntries, painGoals, toolTrackingResults);
    }

    private PainConsolidatedReportGenerator createPainConsolidatedReportGenerator(Patient patient, Date startDate, Date endDate, AssessmentResults results, PainDiaryEntries painDiaryEntries, PainGoals painGoals, ToolTrackingResults toolTrackingResults) {
        return new PainConsolidatedReportGenerator(new PainConsolidatedReportGeneratorParameter(patient, startDate, endDate, results, painDiaryEntries, painGoals, toolTrackingResults));
    }


    @Test
    public void calculatesPainDiaryAveragePainWithOneScore() {
        AssessmentResults results = new AssessmentResults();

        List<String> qualityList = new ArrayList<String>();
        PainDiaryEntry oneEntry = createPainDiaryEntry("3", qualityList, "01/01/2012");
        PainDiaryEntries painDiaryEntries = createPainDiaryEntries(oneEntry);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY);

        Assert.assertTrue(reportText.contains("Daily Pain Intensity: 3"));
    }

    @Test
    public void calculatesPainDiaryAveragePainWithTwoScores() {
        AssessmentResults results = new AssessmentResults();

        List<String> qualityList = new ArrayList<String>();
        PainDiaryEntry oneEntry = createPainDiaryEntry("3", qualityList, "01/01/2012");
        PainDiaryEntry twoEntry = createPainDiaryEntry("7", qualityList, "02/01/2012");
        PainDiaryEntries painDiaryEntries = createPainDiaryEntries(oneEntry, twoEntry);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY);

        Assert.assertTrue(reportText.contains("Daily Pain Intensity: 5"));
    }

    @Test
    public void calculatesPainDiaryCommonQualityPain() {
        AssessmentResults results = new AssessmentResults();

        List<String> qualityList1 = new ArrayList<String>();
        qualityList1.add("showme1");
        qualityList1.add("noshow1");
        qualityList1.add("showme2");
        qualityList1.add("noshow2");
        qualityList1.add("showme3");
        qualityList1.add("showme4");
        qualityList1.add("showme5");
        PainDiaryEntry entry1 = createPainDiaryEntry("3", qualityList1, "01/01/2012");

        List<String> qualityList2 = new ArrayList<String>();
        qualityList2.add("showme1");
        qualityList2.add("showme2");
        qualityList2.add("showme3");
        qualityList2.add("noshow3");
        qualityList2.add("noshow4");
        qualityList2.add("noshow5");
        PainDiaryEntry entry2 = createPainDiaryEntry("3", qualityList2, "01/01/2012");

        List<String> qualityList3 = new ArrayList<String>();
        qualityList3.add("showme3");
        qualityList2.add("showme4");
        qualityList2.add("showme5");
        PainDiaryEntry entry3 = createPainDiaryEntry("3", qualityList3, "01/01/2012");


        PainDiaryEntries painDiaryEntries = createPainDiaryEntries(entry1, entry2, entry3);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY);

        Assert.assertTrue(reportText.contains("Pain Quality/Descriptions (Top 5)"));
        Assert.assertTrue(reportText.contains("<li>showme1</li>"));
        Assert.assertTrue(reportText.contains("<li>showme2</li>"));
        Assert.assertTrue(reportText.contains("<li>showme3</li>"));
        Assert.assertTrue(reportText.contains("<li>showme4</li>"));
        Assert.assertTrue(reportText.contains("<li>showme5</li>"));
        Assert.assertFalse(reportText.contains("<li>noshow1</li>"));
        Assert.assertFalse(reportText.contains("<li>noshow2</li>"));
        Assert.assertFalse(reportText.contains("<li>noshow3</li>"));
        Assert.assertFalse(reportText.contains("<li>noshow4</li>"));
        Assert.assertFalse(reportText.contains("<li>noshow5</li>"));
    }

    @Test
    public void calculatesPainDiaryCommonActivators() {
        AssessmentResults results = new AssessmentResults();

        PainDiaryEntry entry1 = createPainDiaryEntryWithActivators("activator1", "01/01/2012");
        PainDiaryEntry entry2 = createPainDiaryEntryWithActivators("activator2", "12/01/2011");
        PainDiaryEntry entry3 = createPainDiaryEntryWithActivators("activator3", "11/01/2011");
        PainDiaryEntry entry4 = createPainDiaryEntryWithActivators("activator4", "10/01/2011");
        PainDiaryEntry entry5 = createPainDiaryEntryWithActivators("activator5", "09/01/2011");
        PainDiaryEntry entry6 = createPainDiaryEntryWithActivators("activator6", "08/01/2011");
        PainDiaryEntry entry7 = createPainDiaryEntryWithActivators("activator7", "07/01/2011");
        PainDiaryEntry entry8 = createPainDiaryEntryWithActivators("activator8", "06/01/2011");

        PainDiaryEntries painDiaryEntries = createPainDiaryEntries(entry1, entry2, entry3, entry4, entry5, entry6, entry7, entry8);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY);

        Assert.assertTrue(reportText.contains("Pain Activators (Last entry)"));
        Assert.assertTrue(reportText.contains("<li>activator1</li>"));
        Assert.assertTrue(reportText.contains("<li>activator2</li>"));
        Assert.assertTrue(reportText.contains("<li>activator3</li>"));
        Assert.assertTrue(reportText.contains("<li>activator4</li>"));
        Assert.assertTrue(reportText.contains("<li>activator5</li>"));
        Assert.assertTrue(reportText.contains("<li>activator6</li>"));
        Assert.assertTrue(reportText.contains("<li>activator7</li>"));

        Assert.assertFalse(reportText.contains("<li>activator8</li>"));
    }

    @Test
    public void calculatesPainDiaryCommonActivatorsWithFewerThan7Entries() {
        AssessmentResults results = new AssessmentResults();

        PainDiaryEntry entry1 = createPainDiaryEntryWithActivators("activator1", "01/01/2012");
        PainDiaryEntry entry2 = createPainDiaryEntryWithActivators("activator2", "12/01/2011");
        PainDiaryEntry entry3 = createPainDiaryEntryWithActivators("activator3", "11/01/2011");

        PainDiaryEntries painDiaryEntries = createPainDiaryEntries(entry1, entry2, entry3);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY);

        Assert.assertTrue(reportText.contains("Pain Activators (Last entry)"));
        Assert.assertTrue(reportText.contains("<li>activator1</li>"));
        Assert.assertTrue(reportText.contains("<li>activator2</li>"));
        Assert.assertTrue(reportText.contains("<li>activator3</li>"));
    }

    @Test
    public void calculatesPainDiaryCommonActivatorsExcludesNull() {
        AssessmentResults results = new AssessmentResults();

        PainDiaryEntry entry1 = createPainDiaryEntryWithActivators("activator1", "01/01/2012");
        PainDiaryEntry entry2 = createPainDiaryEntryWithActivators(null, "12/01/2011");
        PainDiaryEntry entry3 = createPainDiaryEntryWithActivators("activator3", "11/01/2011");

        PainDiaryEntries painDiaryEntries = createPainDiaryEntries(entry1, entry2, entry3);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY);

        Assert.assertTrue(reportText.contains("Pain Activators (Last entry)"));
        Assert.assertTrue(reportText.contains("<li>activator1</li>"));
        Assert.assertTrue(reportText.contains("<li>activator3</li>"));
        Assert.assertFalse(reportText.contains("<li>null</li>"));
    }

    @Test
    public void calculatesPainDiaryMostUsedTreatment() {
        AssessmentResults results = new AssessmentResults();

        List<String> treatmentList = new ArrayList<String>();
        treatmentList.add("coffee");
        treatmentList.add("gaming");
        treatmentList.add("sleep");
        PainDiaryEntry entry = createPainDiaryEntryWithTreatments(treatmentList, "01/01/2012");
        PainDiaryEntries painDiaryEntries = createPainDiaryEntries(entry);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY);

        Assert.assertTrue(reportText.contains("Treatments Used (Top 5)"));
        Assert.assertTrue(reportText.contains("<li>coffee</li>"));
        Assert.assertTrue(reportText.contains("<li>gaming</li>"));
        Assert.assertTrue(reportText.contains("<li>sleep</li>"));
    }

    @Test
    public void calculatesPainDiaryItemsImprovingPain() {
        AssessmentResults results = new AssessmentResults();

        PainDiaryEntry entry1 = createPainDiaryEntryWithItemsImprovingPain("imp1", "01/01/2012");
        PainDiaryEntry entry2 = createPainDiaryEntryWithItemsImprovingPain("imp2", "01/01/2011");

        PainDiaryEntries painDiaryEntries = createPainDiaryEntries(entry1, entry2);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY);

        Assert.assertTrue(reportText.contains("Items that Improved Pain (Last Entry)"));
        Assert.assertTrue(reportText.contains("<li>imp1</li>"));
        Assert.assertTrue(reportText.contains("<li>imp2</li>"));
    }

    @Test
    public void calculatesPainDiaryItemsWorseningPain() {
        AssessmentResults results = new AssessmentResults();

        PainDiaryEntry entry1 = createPainDiaryEntryWithItemsWorseningPain("worse1", "01/01/2012");
        PainDiaryEntry entry2 = createPainDiaryEntryWithItemsWorseningPain("worse2", "01/01/2011");

        PainDiaryEntries painDiaryEntries = createPainDiaryEntries(entry1, entry2);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY);

        Assert.assertTrue(reportText.contains("Items that Worsened Pain (Last Entry)"));
        Assert.assertTrue(reportText.contains("<li>worse1</li>"));
        Assert.assertTrue(reportText.contains("<li>worse2</li>"));
    }

    @Test
    public void painDiaryComments() {
        AssessmentResults results = new AssessmentResults();

        String comments = "comment1";
        PainDiaryEntry entry = createPainDiaryEntryWithComments(comments, "01/01/2012");
        PainDiaryEntries painDiaryEntries = createPainDiaryEntries(entry);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY);

        Assert.assertTrue(reportText.contains("Comments"));
        Assert.assertTrue(reportText.contains("<li>01/01/2012: comment1</li>"));
    }

    @Test
    public void painDiaryWithNullComments() {
        AssessmentResults results = new AssessmentResults();

        String comments = null;
        PainDiaryEntry entry = createPainDiaryEntryWithComments(comments, "01/01/2012");
        PainDiaryEntries painDiaryEntries = createPainDiaryEntries(entry);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY);

        Assert.assertTrue(reportText.contains("Comments"));
        Assert.assertFalse(reportText.contains("<li>01/01/2012: null</li>"));
    }

    @Test
    public void calculatesPainDiaryMoodPainImpact() {
        AssessmentResults results = new AssessmentResults();

        PainDiaryEntry oneEntry = createPainDiaryEntryWithMoodLevel("3", "01/01/2012");
        PainDiaryEntry twoEntry = createPainDiaryEntryWithMoodLevel("7", "02/01/2012");
        PainDiaryEntries painDiaryEntries = createPainDiaryEntries(oneEntry, twoEntry);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY);

        Assert.assertTrue(reportText.contains("Pain negative impact on Mood: 5"));
    }

    @Test
    public void calculatesPainDiaryMovementPainImpact() {
        AssessmentResults results = new AssessmentResults();

        PainDiaryEntry oneEntry = createPainDiaryEntryWithImpactOnWalking("3", "01/01/2012");
        PainDiaryEntry twoEntry = createPainDiaryEntryWithImpactOnWalking("7", "02/01/2012");
        PainDiaryEntries painDiaryEntries = createPainDiaryEntries(oneEntry, twoEntry);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY);

        Assert.assertTrue(reportText.contains("Pain interference on Movement: 5"));
    }

    @Test
    public void calculatesPainDiarySleepPainImpact() {
        AssessmentResults results = new AssessmentResults();

        PainDiaryEntry oneEntry = createPainDiaryEntryWithAmountRefreshedAfterSleep("3", "01/01/2012");
        PainDiaryEntry twoEntry = createPainDiaryEntryWithAmountRefreshedAfterSleep("7", "02/01/2012");
        PainDiaryEntries painDiaryEntries = createPainDiaryEntries(oneEntry, twoEntry);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY);

        Assert.assertTrue(reportText.contains("How refreshed from Sleep: 5"));
    }

    @Test
    public void calculatesPainDiaryActivitiesPainImpact() {
        AssessmentResults results = new AssessmentResults();

        PainDiaryEntry oneEntry = createPainDiaryEntryWithPainInterference("3", "01/01/2012");
        PainDiaryEntry twoEntry = createPainDiaryEntryWithPainInterference("7", "02/01/2012");
        PainDiaryEntries painDiaryEntries = createPainDiaryEntries(oneEntry, twoEntry);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY);

        Assert.assertTrue(reportText.contains("Pain interference with Activities: 5"));
    }

    @Test
    public void calculatesPainDiaryAverageDepression() {
        AssessmentResults results = new AssessmentResults();

        PainDiaryEntry oneEntry = createPainDiaryEntryWithDepression("1", "01/01/2012");
        PainDiaryEntry twoEntry = createPainDiaryEntryWithDepression("3", "02/01/2012");
        PainDiaryEntries painDiaryEntries = createPainDiaryEntries(oneEntry, twoEntry);

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY);

        Assert.assertTrue(reportText.contains("Depression Level: 2"));
    }

    @Test
    public void generatesPainGoalReport() {
        AssessmentResult result = MonthlyPainReportTestHelper.generateMockAssessmentResult(2, null);
        AssessmentResult previousResult = MonthlyPainReportTestHelper.generatePreviousResult(3, 3);
        AssessmentResults results = new AssessmentResults();
        results.add(result);
        results.add(previousResult);

        PainDiaryEntries painDiaryEntries = new PainDiaryEntries();

        PainGoals painGoals = createPainGoals();
        ToolTrackingResults toolTrackingResults = new ToolTrackingResults();

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries, painGoals, toolTrackingResults);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINGOALS);

        assertNotNull(resultReport);
        assertNotNull(reportText);
        assertTrue(reportText.contains("Pain Goal Report"));
        assertTrue(reportText.contains("Complete Pain Goal Entry"));
        assertFalse(reportText.contains("Active Pain Goal Entry"));
    }

    private PainGoals createPainGoals() {
        PainGoals painGoals = new PainGoals();
        PainGoal painGoal = new PainGoal();
        painGoal.setStatus(PainGoalStatus.PainGoalStatusActive);
        painGoal.setTargetDate(DateHelper.parseDate("12/31/2012"));
        painGoal.setDateStarted(DateHelper.parseDate("01/01/2012"));

        PainGoalEntries painGoalEntries = new PainGoalEntries();
        PainGoalEntry entry = new PainGoalEntry();
        entry.setStatus(PainGoalEntryStatus.PainGoalEntryStatusComplete);
        entry.setComments("Complete Pain Goal Entry");
        entry.setEntryDate(DateHelper.parseDate("01/01/2012"));
        painGoalEntries.add(entry);

        entry = new PainGoalEntry();
        entry.setStatus(PainGoalEntryStatus.PainGoalEntryStatusActive);
        entry.setComments("Active Pain Goal Entry");
        entry.setEntryDate(DateHelper.parseDate("01/01/2012"));
        painGoalEntries.add(entry);
        painGoal.setPainGoalEntries(painGoalEntries);
        painGoals.add(painGoal);

        painGoal = new PainGoal();
        painGoal.setStatus(PainGoalStatus.PainGoalStatusActive);
        painGoal.setTargetDate(DateHelper.parseDate("11/31/2012"));
        painGoal.setDateStarted(DateHelper.parseDate("02/01/2012"));
        painGoals.add(painGoal);

        painGoal = new PainGoal();
        painGoal.setStatus(PainGoalStatus.PainGoalStatusComplete);
        painGoal.setPercentComplete("100");
        painGoal.setDateCompleted(DateHelper.parseDate("12/31/2012"));
        painGoal.setDateStarted(DateHelper.parseDate("01/01/2012"));
        painGoals.add(painGoal);

        painGoal = new PainGoal();
        painGoal.setStatus(PainGoalStatus.PainGoalStatusComplete);
        painGoal.setDateCompleted(DateHelper.parseDate("11/31/2012"));
        painGoal.setDateStarted(DateHelper.parseDate("02/01/2012"));
        painGoals.add(painGoal);

        painGoal = new PainGoal();
        painGoal.setStatus(PainGoalStatus.PainGoalStatusDeleted);
        painGoal.setDateDeleted(DateHelper.parseDate("12/31/2012"));
        painGoal.setDateStarted(DateHelper.parseDate("01/01/2012"));
        painGoals.add(painGoal);

        painGoal = new PainGoal();
        painGoal.setStatus(PainGoalStatus.PainGoalStatusDeleted);
        painGoal.setDateDeleted(DateHelper.parseDate("11/31/2012"));
        painGoal.setDateStarted(DateHelper.parseDate("02/01/2012"));
        painGoals.add(painGoal);


        painGoal.setPainGoalEntries(new PainGoalEntries());
        return painGoals;
    }

    @Test
    public void generatesToolTrackingReport() {
        AssessmentResult result = com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.MonthlyPainReportTestHelper.generateMockAssessmentResult(2, null);
        AssessmentResult previousResult = com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.MonthlyPainReportTestHelper.generatePreviousResult(3, 3);
        AssessmentResults results = new AssessmentResults();
        results.add(result);
        results.add(previousResult);

        PainDiaryEntries painDiaryEntries = new PainDiaryEntries();
        PainGoals painGoals = createPainGoals();

        ToolTrackingResults toolTrackingResults = new ToolTrackingResults();

        createToolTrackingResult(toolTrackingResults, "01/01/2012", "Muscle Relaxation");
        createToolTrackingResult(toolTrackingResults, "02/01/2012", "Muscle Relaxation");
        createToolTrackingResult(toolTrackingResults, "02/01/2012", "Activity Pacing");
        createToolTrackingResult(toolTrackingResults, "03/01/2012", "Z Activity");

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries, painGoals, toolTrackingResults);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_TOOLTRACKING);

        assertNotNull(resultReport);
        assertNotNull(reportText);
        AssertEx.assertFragmentOccurs(reportText, "Tool Tracking Summary", 1);
        //despite 2 tool tracking entries for tool1, only one is shown on report
        AssertEx.assertFragmentOccurs(reportText, "Muscle Relaxation", 1);
        AssertEx.assertFragmentOccurs(reportText, "Activity Pacing", 1);
    }

    @Test
    public void generatesToolTrackingReportWhenDateIsNull() {
        AssessmentResult result = com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.MonthlyPainReportTestHelper.generateMockAssessmentResult(2, null);
        AssessmentResult previousResult = com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.MonthlyPainReportTestHelper.generatePreviousResult(3, 3);
        AssessmentResults results = new AssessmentResults();
        results.add(result);
        results.add(previousResult);

        PainDiaryEntries painDiaryEntries = new PainDiaryEntries();
        PainGoals painGoals = createPainGoals();

        ToolTrackingResults toolTrackingResults = new ToolTrackingResults();

        createToolTrackingResult(toolTrackingResults);
        createToolTrackingResult(toolTrackingResults, "02/01/2012", "tool1");

        PainConsolidatedReportGenerator reportGenerator = createPainConsolidatedReportGenerator(results, painDiaryEntries, painGoals, toolTrackingResults);

        AssessmentResultReport resultReport = reportGenerator.generateReport();
        String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.PAIN_CONSOLIDATED_REPORT_TOOLTRACKING);

        assertNotNull(resultReport);
        assertNotNull(reportText);
        assertTrue(reportText.contains("Tool Tracking Summary"));
    }

    private void createToolTrackingResult(ToolTrackingResults toolTrackingResults) {
        ToolTrackingResult toolTrackingResult = new ToolTrackingResult();
        toolTrackingResults.add(toolTrackingResult);
    }

    private void createToolTrackingResult(ToolTrackingResults toolTrackingResults, String dateTracked, String toolName) {
        ToolTrackingResult toolTrackingResult = new ToolTrackingResult();
        toolTrackingResult.setTool(toolName);
        toolTrackingResult.setDateTracked(DateHelper.parseDate(dateTracked));
        toolTrackingResult.setFrequency("30");
        toolTrackingResult.setUsefulness(20);
        toolTrackingResult.setSuccessRate(10);
        toolTrackingResults.add(toolTrackingResult);
    }

    private PainDiaryEntries createPainDiaryEntries(PainDiaryEntry... entries){
        PainDiaryEntries painEntries = new PainDiaryEntries();
        Collections.addAll(painEntries, entries);
        return painEntries;
    }

    private PainDiaryEntry createPainDiaryEntry(String averageIntensity, List<String> qualityList, String createdDate) {
        PainDiaryEntry entry = new PainDiaryEntry();
        entry.setDateCreated(DateHelper.parseDate(createdDate));
        entry.setPainQuestionValues(new PainDiaryReport());
        entry.getPainQuestionValues().setAveragePainIntensity(averageIntensity);
        entry.getPainQuestionValues().setPainQuality(qualityList);

        return entry;
    }

    private PainDiaryEntry createPainDiaryEntryWithMoodLevel(String moodLevel, String createdDate) {
        PainDiaryEntry entry = new PainDiaryEntry();
        entry.setDateCreated(DateHelper.parseDate(createdDate));
        entry.setPainQuestionValues(new PainDiaryReport());
        entry.getPainQuestionValues().setMoodLevel(moodLevel);
        return entry;
    }

    private PainDiaryEntry createPainDiaryEntryWithImpactOnWalking(String movementLevel, String createdDate) {
        PainDiaryEntry entry = new PainDiaryEntry();
        entry.setDateCreated(DateHelper.parseDate(createdDate));
        entry.setPainQuestionValues(new PainDiaryReport());
        entry.getPainQuestionValues().setImpactOnWalking(movementLevel);
        return entry;

    }

    private PainDiaryEntry createPainDiaryEntryWithAmountRefreshedAfterSleep(String sleep, String createdDate) {
        PainDiaryEntry entry = new PainDiaryEntry();
        entry.setDateCreated(DateHelper.parseDate(createdDate));
        entry.setPainQuestionValues(new PainDiaryReport());
        entry.getPainQuestionValues().setAmountRefreshedAfterSleep(sleep);
        return entry;
    }

    private PainDiaryEntry createPainDiaryEntryWithPainInterference(String activitiesLevel, String createdDate) {
        PainDiaryEntry entry = new PainDiaryEntry();
        entry.setDateCreated(DateHelper.parseDate(createdDate));
        entry.setPainQuestionValues(new PainDiaryReport());
        entry.getPainQuestionValues().setPainInterference(activitiesLevel);
        return entry;

    }

    private PainDiaryEntry createPainDiaryEntryWithActivators(String activators, String createdDate){
        PainDiaryEntry entry = new PainDiaryEntry();
        entry.setDateCreated(DateHelper.parseDate(createdDate));
        entry.setPainQuestionValues(new PainDiaryReport());
        entry.getPainQuestionValues().setItemsThatActivatedPain(activators);
        return entry;
    }

    private PainDiaryEntry createPainDiaryEntryWithTreatments(List<String> treatments, String createdDate) {
        PainDiaryEntry entry = new PainDiaryEntry();
        entry.setDateCreated(DateHelper.parseDate(createdDate));
        entry.setPainQuestionValues(new PainDiaryReport());
        entry.getPainQuestionValues().setTreatment(treatments);
        return entry;
    }

    private PainDiaryEntry createPainDiaryEntryWithItemsImprovingPain(String improvingList, String createdDate) {
        PainDiaryEntry entry = new PainDiaryEntry();
        entry.setDateCreated(DateHelper.parseDate(createdDate));
        entry.setPainQuestionValues(new PainDiaryReport());
        entry.getPainQuestionValues().setItemsThatImprovedPain(improvingList);
        return entry;
    }

    private PainDiaryEntry createPainDiaryEntryWithItemsWorseningPain(String worseningList, String createdDate) {
        PainDiaryEntry entry = new PainDiaryEntry();
        entry.setDateCreated(DateHelper.parseDate(createdDate));
        entry.setPainQuestionValues(new PainDiaryReport());
        entry.getPainQuestionValues().setItemsThatWorsenPain(worseningList);
        return entry;
    }
    private PainDiaryEntry createPainDiaryEntryWithDepression(String depressionLevel, String createdDate) {
        PainDiaryEntry entry = new PainDiaryEntry();
        entry.setDateCreated(DateHelper.parseDate(createdDate));
        entry.setPainQuestionValues(new PainDiaryReport());
        entry.getPainQuestionValues().setDepressionLevel(depressionLevel);
        return entry;

    }

    private PainDiaryEntry createPainDiaryEntryWithComments(String comments, String createdDate) {
        PainDiaryEntry entry = new PainDiaryEntry();
        entry.setDateCreated(DateHelper.parseDate(createdDate));
        entry.setPainQuestionValues(new PainDiaryReport());
        entry.getPainQuestionValues().setComments(comments);
        return entry;
    }
}
