package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.Date;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResult;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResultReport;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssessmentCode;
import com.agilex.healthcare.utility.DateHelper;

public class MonthlyPainReportGeneratorTest {
	private MonthlyPainReportGenerator reportGenerator = new MonthlyPainReportGenerator();

	@Test
	public void testReportWithNullScore() {
		AssessmentResult result = MonthlyPainReportTestHelper.generateMockAssessmentResult(2, null);
		AssessmentResult previousResult = MonthlyPainReportTestHelper.generatePreviousResult(3, 3);

		Date takenDate = DateHelper.parseDate("01/01/2012");
		result.setDateTaken(takenDate);
		
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, previousResult);
		String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		
		assertNotNull(reportText);
		assertTrue(reportText.contains("Monthly Pain Assessment Report"));
		assertTrue(reportText.contains("Date: 01/01/2012"));
	}
	
	@Test
	public void testReportWithNoPriorScore() {
		AssessmentResult result = MonthlyPainReportTestHelper.generateMockAssessmentResult(2, 1, 1, 1);
		Date takenDate = DateHelper.parseDate("01/01/2012");
		result.setDateTaken(takenDate);
		
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, null);
		String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);

		assertNotNull(reportText);
		assertTrue(reportText.contains("Monthly Pain Assessment Report"));
		assertTrue(reportText.contains("Date: 01/01/2012"));
		assertFalse(reportText.contains("The following items have changed since the last assessment you tool on "));
		assertTrue(reportText.contains("You have no prior assessments for comparison"));
		assertTrue(reportText.contains("Q1: null Score: 2"));
	}


	@Test
	public void testReportWithSameScore() {
		Integer[] commonResult = {2, 1, 1, 1};
		AssessmentResult result = MonthlyPainReportTestHelper.generateMockAssessmentResult(commonResult);
		AssessmentResult previousResult = MonthlyPainReportTestHelper.generatePreviousResult(commonResult);

		Date takenDate = DateHelper.parseDate("01/01/2012");
		previousResult.setDateTaken(takenDate);
		
		
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, previousResult);
		String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		
		assertNotNull(reportText);
		
		assertTrue(reportText.contains("Since your last assessment on 01/01/2012"));
		
		assertTrue(reportText.contains("These items have not changed"));
		assertFalse(reportText.contains("These items have improved"));
		assertFalse(reportText.contains("These items have worsened"));
	}

	@Test
	public void responsesSorted() {
		Integer[] commonResult = {2, 1, 1, 1};
		AssessmentResult result = MonthlyPainReportTestHelper.generateMockAssessmentResult(commonResult);
		AssessmentResult previousResult = MonthlyPainReportTestHelper.generatePreviousResult(commonResult);

		Date takenDate = DateHelper.parseDate("01/01/2012");
		previousResult.setDateTaken(takenDate);
		
		
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, previousResult);
		String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		
		assertNotNull(reportText);
		String question4 = "Q4: Daily Activity Interference - 1";
		String question3 = "Q3: Lowest pain level - 1";
		assertTrue(reportText.contains(question3));
		assertTrue(reportText.contains(question4));
		
		
		//get positions and make sure they are sorted correctly
		int question3Index = reportText.indexOf(question3);
		int question4Index = reportText.indexOf(question4);
		
		assertTrue(question4Index > question3Index);
		
	
	}

	//TODO: Improve this test, ensure the same/improve/worsen section titles occur once and only once
	@Test
	public void testReportWithMultipleQuestionsInEachChangeSection() {
		AssessmentResult result = MonthlyPainReportTestHelper.generateMockAssessmentResult(4, 4, 4, 4, 4, 4, 1, 1, 4, 1, 1, 1, 6);
		AssessmentResult previousResult = MonthlyPainReportTestHelper.generatePreviousResult(4, 3, 5, 4, 3, 5, 2, 2, 6, 2, 2, 2, 4);

		Date takenDate = DateHelper.parseDate("01/01/2012");
		previousResult.setDateTaken(takenDate);
		
		
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, previousResult);
		String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		
		assertNotNull(reportText);
		
	}

	@Test
	public void testReportWithWorsenedScore() {
		AssessmentResult result = MonthlyPainReportTestHelper.generateMockAssessmentResult(5);
		AssessmentResult previousResult = MonthlyPainReportTestHelper.generatePreviousResult(3);


		AssessmentResultReport resultReport = reportGenerator.generateReport(result, previousResult);
		String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		
		assertNotNull(reportText);
		assertFalse(reportText.contains("These items have not changed"));
		assertFalse(reportText.contains("These items have improved"));
		assertTrue(reportText.contains("These items have worsened"));
	}
	@Test
	public void testReportWithImprovedScore() {
		AssessmentResult result = MonthlyPainReportTestHelper.generateMockAssessmentResult(3);
		AssessmentResult previousResult = MonthlyPainReportTestHelper.generatePreviousResult(5);


		AssessmentResultReport resultReport = reportGenerator.generateReport(result, previousResult);
		String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		
		assertNotNull(reportText);
		assertFalse(reportText.contains("These items have not changed"));
		assertTrue(reportText.contains("These items have improved"));
		assertFalse(reportText.contains("These items have worsened"));
	}

}
