package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.Date;

import org.joda.time.DateTime;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentEligibility;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResult;
import com.agilex.healthcare.utility.DateHelper;

public class EligibilityCalculatorTest {
	private EligibilityCalculator calculator = new EligibilityCalculator();

	@Test
	public void eligibilityWithZeroWaitingPeriod() {
		AssessmentResult mostRecentlyCompletedResult = new AssessmentResult();
		mostRecentlyCompletedResult.setDateTaken(DateHelper.getToday());
		AssessmentEligibility eligibility = calculator.determineEligibility(0, mostRecentlyCompletedResult);

		assertTrue(eligibility.isEligibile());
	}

	@Test
	public void eligibilityWithResultTakenToday() {
		AssessmentResult mostRecentlyCompletedResult = new AssessmentResult();
		mostRecentlyCompletedResult.setDateTaken(DateHelper.getToday());
		AssessmentEligibility eligibility = calculator.determineEligibility(7, mostRecentlyCompletedResult);

		assertFalse(eligibility.isEligibile());
	}

	@Test
	public void eligibilityWithResultTakenYesterday() {
		AssessmentResult mostRecentlyCompletedResult = new AssessmentResult();
		mostRecentlyCompletedResult.setDateTaken(DateHelper.getYesterday());
		AssessmentEligibility eligibility = calculator.determineEligibility(7, mostRecentlyCompletedResult);

		assertFalse(eligibility.isEligibile());
		assertEquals(6, eligibility.getDaysLeft());
		assertNotNull(eligibility.getMessage());
	}

	@Test
	public void resultTakenWithinWaitingPeriod() {
		AssessmentResult mostRecentlyCompletedResult = new AssessmentResult();
		DateTime today = new DateTime();
		Date sixDaysAgo = today.minusDays(6).toDate();

		mostRecentlyCompletedResult.setDateTaken(sixDaysAgo);
		AssessmentEligibility eligibility = calculator.determineEligibility(7, mostRecentlyCompletedResult);

		assertFalse(eligibility.isEligibile());
		assertEquals(1, eligibility.getDaysLeft());
		assertNotNull(eligibility.getMessage());
		System.out.println(eligibility.getMessage());
	}

	@Test
	public void resultTakenSlightlyMoreThanSevenDaysAgo() {
		AssessmentResult mostRecentlyCompletedResult = new AssessmentResult();
		DateTime today = new DateTime();
		Date sevenDaysAgo = today.minusDays(7).toDate();
		Date moreThanSevenDaysAgo = DateHelper.minusMinutes(sevenDaysAgo,30);

		mostRecentlyCompletedResult.setDateTaken(moreThanSevenDaysAgo);
		AssessmentEligibility eligibility = calculator.determineEligibility(7, mostRecentlyCompletedResult);

		assertTrue(eligibility.isEligibile());
		assertEquals(0, eligibility.getDaysLeft());
		assertEquals(null,eligibility.getMessage());
	}

	@Test
	public void resultTakenSlightlyLessThanSevenDaysAgo() {
		DateTime today = new DateTime();
		if (today.getHourOfDay() < 23 && today.getMinuteOfHour() < 50) {
			// JUSTIFICATION: this test is not applicable when it's being run between 11:55 PM and 12:00 AM.
			// to be safe, the test will be ignored when run between 11:50 PM (five minutes earlier) and 12:00 AM
			AssessmentResult mostRecentlyCompletedResult = new AssessmentResult();
			Date sevenDaysAgo = today.minusDays(7).toDate();
			Date lessThanSevenDaysAgo = DateHelper.plusMinutes(sevenDaysAgo,5);
	
			mostRecentlyCompletedResult.setDateTaken(lessThanSevenDaysAgo);
			AssessmentEligibility eligibility = calculator.determineEligibility(7, mostRecentlyCompletedResult);
	
			assertTrue(eligibility.isEligibile());
			assertEquals(0, eligibility.getDaysLeft());
			assertEquals(null,eligibility.getMessage());
		}
	}
	
	@Test
	public void resultTakenOutsideWaitingPeriod() {
		AssessmentResult mostRecentlyCompletedResult = new AssessmentResult();
		DateTime today = new DateTime();
		Date sixDaysAgo = today.minusDays(8).toDate();

		mostRecentlyCompletedResult.setDateTaken(sixDaysAgo);
		AssessmentEligibility eligibility = calculator.determineEligibility(7, mostRecentlyCompletedResult);

		assertTrue(eligibility.isEligibile());
		assertEquals(0, eligibility.getDaysLeft());
		assertEquals(null,eligibility.getMessage());
	}

}
