package com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder;

import java.net.URI;

import javax.ws.rs.core.UriBuilder;

import com.agilex.healthcare.mobilehealthplatform.domain.AtomLink;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.Patients;
import com.agilex.healthcare.mobilehealthplatform.domain.code.VitalSectionCode;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.UriDtoValidator;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.UriHelper;
import com.agilex.healthcare.utility.ArrayAppender;

public class PatientLinkBuilder {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientLinkBuilder.class);

	private static String LONGITUDINAL = "longitudinal";
	private static String OPERATIONAL = "operational";
	
	private URI baseUri;

	public PatientLinkBuilder(URI baseUri) {
		this.baseUri = baseUri;
	}

	public static URI createPatientSelfUri(PatientIdentifier patientIdentifier, URI baseUri) {
		PatientLinkBuilder linkBuilder = new PatientLinkBuilder(baseUri);
		return linkBuilder.createPatientSelfUri(patientIdentifier);
	}

	public URI createPatientSelfUri(PatientIdentifier patientIdentifier) {
		return UriBuilder.fromUri(this.baseUri).path("patient").path(patientIdentifier.getAssigningAuthority()).path(patientIdentifier.getUniqueId()).build();
	}

	public static URI createPatientSelfUri(Patient patient, URI baseUri) {
		URI uri = null;
		if (patient == null) {
			uri = null;
		} else {
			uri = createPatientSelfUri(patient.getPatientIdentifier(), baseUri);
		}
		return uri;
	}

	public void fillLinks(Patients patients, URI requestUri) {
		if (patients != null) {
			UriDtoValidator.validate(patients);

			patients.setSelfUri(requestUri);
			fillCustomRelatedLinks(patients);

			for (Patient dataItem : patients) {
				this.fillLinks(dataItem, null);
			}
		}
	}

	private void fillCustomRelatedLinks(Patients patients) {
		// why does this blank method exist?
		// answer: it is here to remain consistent with the pattern in the
		// other link builder implementations
	}

	public void fillLinks(Patient dataItem, URI requestUri) {
		if (dataItem != null) {
			UriDtoValidator.validate(dataItem);
			UriHelper.setSelfUri(dataItem, requestUri, this.get      icalSelfUri(dataItem));
			fillCustomRelatedLinks(dataItem);
		}
	}

	private URI get      icalSelfUri(Patient dataItem) {
		return createPatientSelfUri(dataItem.getPatientIdentifier(), this.getBaseUri());
	}

	private void fillCustomRelatedLinks(Patient patient) {
		logger.debug("Getting Patient Links from PatientLinkBuilder");

		appendRelatedLink(patient, LinkTitles.PatientIdentifiers, "identifiers");
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientDemographics, "demographics");
		appendRelatedLink(patient, LinkTitles.PatientPreference, "preference");
		appendRelatedLink(patient, LinkTitles.PatientImage, "patient-image");

		fillCalendarLinks(patient);

		fillProblemLinks(patient);

		fillAllergyLinks(patient);

		fillRadiologyLinks(patient);

		fillVitalsLinks(patient);

		fillJournalLinks(patient);

		fillAdmissionLinks(patient);

		fillSurgeryLinks(patient);

		fillPatientSecureMessageRelatedLinks(patient);

		fillPatientMedicationLinks(patient);

		appendRelatedLink(patient, LinkTitles.PatientMedicationRefills, "medication-refills");

		fillPatientLabLinks(patient);

		appendRelatedLink(patient, LinkTitles.PatientDocuments, "documents");

		fillPatientNotificationRelatedLinks(patient);

		appendRelatedLink(patient, LinkTitles.HealthAdvocateRequest, "health-advocate-requests");
		appendRelatedLink(patient, LinkTitles.HealthAdvocateRequestEligibility, "health-advocate-requests", "eligibility");
		appendRelatedLink(patient, LinkTitles.PainDiary, "pain-diary");
		appendRelatedLink(patient, LinkTitles.PainGoals, "pain-goals");
		appendRelatedLink(patient, LinkTitles.PainGoalsReport, "pain-goals", "report");

		fillToolTrackingResultLinks(patient);
		
		appendRelatedLink(patient, LinkTitles.Assessments,  "assessments" );
//		appendRelatedLink(patient, LinkTitles.PatientAssessmentResults,  "assessments", "results");
//		appendRelatedLink(patient, LinkTitles.PatientAssessmentDrafts, "assessments", "results", "drafts");
//		appendRelatedLink(patient, LinkTitles.PatientAssessmentGraphData, "assessments", "results", "graphdata");
//		appendRelatedLink(patient, LinkTitles.PatientAssessmentTools, "assessments", "results", "recommended-tools");
//		appendRelatedLink(patient, LinkTitles.PatientAssessmentEligibility, "assessments","eligibility");
		appendRelatedLink(patient, LinkTitles.MyGoalsInventory,  "mygoals");
		appendRelatedLink(patient, LinkTitles.CustomDataView,  "customview");
	}

	private void fillJournalLinks(Patient patient) {
		appendRelatedLink(patient, LinkTitles.PatientMood, new MoodLinkBuilder(getBaseUri()));
		appendRelatedLink(patient, LinkTitles.PatientExercise, new ExerciseLinkBuilder(getBaseUri()));
		appendRelatedLink(patient, LinkTitles.PatientDiet, new DietLinkBuilder(getBaseUri()));
		appendRelatedLink(patient, LinkTitles.PatientContactLogs, new ContactLogLinkBuilder(getBaseUri()));
		appendRelatedLink(patient, LinkTitles.DailyEvents, "dailyevents");
	}

	private void fillPatientSecureMessageRelatedLinks(Patient patient) {
		URI secureMessageUri = UriBuilder.fromUri(this.getBaseUri()).path("securemessage-service").path(patient.getPatientIdentifier().getAssigningAuthority()).path(patient.getPatientIdentifier().getUniqueId()).path("secure-message-user").build();
		patient.getLink().addRelatedLink(LinkTitles.SecureMessageUser, secureMessageUri);
	}

	private void fillPatientNotificationRelatedLinks(Patient patient) {
		URI notificationsUri = UriBuilder.fromUri(this.getBaseUri()).path("notification-service").path("user").path("id").path(patient.getPatientIdentifier().getAssigningAuthority()).path(patient.getPatientIdentifier().getUniqueId()).path("notifications").build();
		patient.getLink().addRelatedLink(LinkTitles.PatientNotifications, notificationsUri);

		URI deviceRegistrationUri = UriBuilder.fromUri(this.getBaseUri()).path("notification-service").path("user").path("id").path(patient.getPatientIdentifier().getAssigningAuthority()).path(patient.getPatientIdentifier().getUniqueId()).path("device-registrations").build();
		patient.getLink().addRelatedLink(LinkTitles.PatientDevices, deviceRegistrationUri);
	}

	private void fillPatientLabLinks(Patient patient) {
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientChemistryLabResults, "lab", "results");
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientChemistryLabTestHistory, "lab", "tests");
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientChemistryLabTestHistoryByGroup, "lab", "groups");

		appendRelatedLinksWithScopes(patient, LinkTitles.PatientMicrobiologyLabResults, "lab-microbiology", "results");
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientMicrobiologyLabTestHistory, "lab-microbiology", "tests");
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientMicrobiologyLabTestHistoryByGroup, "lab-microbiology", "groups");
	}

	private void fillPatientMedicationLinks(Patient patient) {
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientMedications, "medications");
		appendRelatedLink(patient, LinkTitles.PatientActiveMedications, "medications", "active");
		appendRelatedLink(patient, LinkTitles.PatientOutpatientMedications, "medications", "outpatient");
		appendRelatedLink(patient, LinkTitles.PatientInpatientMedications, "medications", "inpatient");
		appendRelatedLink(patient, LinkTitles.PatientActiveOutpatientMedications, "medications", "outpatient", "active");
		appendRelatedLink(patient, LinkTitles.PatientActiveInpatientMedications, "medications", "inpatient", "active");
		appendRelatedLink(patient, LinkTitles.PatientMedicationsActiveLongitudinalScope, "medications", "scope", LONGITUDINAL, "active");
		appendRelatedLink(patient, LinkTitles.PatientMedicationsActiveOperationalScope, "medications", "scope", OPERATIONAL, "active");
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientInpatientMedications, "medications", "inpatient");
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientOutpatientMedications, "medications", "outpatient");
		appendRelatedLink(patient, LinkTitles.PatientMedicationsActiveInpatientLongitudinalScope, "medications", "inpatient", "scope", LONGITUDINAL, "active");
		appendRelatedLink(patient, LinkTitles.PatientMedicationsActiveInpatientOperationalScope, "medications", "inpatient", "scope", OPERATIONAL, "active");
		appendRelatedLink(patient, LinkTitles.PatientMedicationsActiveOutpatientLongitudinalScope, "medications", "inpatient", "scope", LONGITUDINAL, "active");
		appendRelatedLink(patient, LinkTitles.PatientMedicationsActiveOutpatientOperationalScope, "medications", "inpatient", "scope", OPERATIONAL, "active");
	}

	private void fillVitalsLinks(Patient patient) {
		appendRelatedLink(patient, LinkTitles.PatientEnteredVitals, "patient-entered-vitals");

		appendRelatedLinksWithScopes(patient, LinkTitles.PatientVitals, "vitals");
		appendRelatedLink(patient, LinkTitles.PatientVitalsGraphLongitudinal, "vitals", "scope", LONGITUDINAL, "graphdata");
		appendRelatedLink(patient, LinkTitles.PatientVitalsGraphOperational, "vitals", "scope", OPERATIONAL, "graphdata");
		appendRelatedLink(patient, LinkTitles.PatientVitalsTableLongitudinal, "vitals", "scope", LONGITUDINAL, "tabledata");
		appendRelatedLink(patient, LinkTitles.PatientVitalsTableOperational, "vitals", "scope", OPERATIONAL, "tabledata");
		
		URI patientEnteredVitals = UriBuilder.fromUri(patient.getSelfUri()).path("patient-entered-vitals").build();
		fillVitalSectionLink(patient, LinkTitles.PatientEnteredVitalsBloodPressure, VitalSectionCode.BloodPressure, patientEnteredVitals);
		fillVitalSectionLink(patient, LinkTitles.PatientEnteredVitalsPain, VitalSectionCode.Pain, patientEnteredVitals);
		fillVitalSectionLink(patient, LinkTitles.PatientEnteredVitalsRespiration, VitalSectionCode.Respiration, patientEnteredVitals);
		fillVitalSectionLink(patient, LinkTitles.PatientEnteredVitalsTemperature, VitalSectionCode.Temperature, patientEnteredVitals);
		fillVitalSectionLink(patient, LinkTitles.PatientEnteredVitalsHeight, VitalSectionCode.Height, patientEnteredVitals);
		fillVitalSectionLink(patient, LinkTitles.PatientEnteredVitalsWeight, VitalSectionCode.Weight, patientEnteredVitals);
		appendRelatedLink(patient, LinkTitles.PatientEnteredVitalsGraphData, "patient-entered-vitals", "graphdata");
		appendRelatedLink(patient, LinkTitles.PatientEnteredVitalsTableData, "patient-entered-vitals", "tabledata");
	}
	
	private void fillVitalSectionLink(Patient patient, String linkTitle, String sectionCode, URI uri) {
		appendRelatedLink(patient, linkTitle, UriBuilder.fromUri(uri).queryParam("section", sectionCode).build());
	}
	
	private void fillAllergyLinks(Patient patient) {
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientAllergies, "allergies");
	}
	
	private void fillAdmissionLinks(Patient patient) {
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientAdmissions, "admissions");
	}
	
	private void fillSurgeryLinks(Patient patient) {
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientSurgeries, "surgeries");
	}
	
	private void fillRadiologyLinks(Patient patient) {
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientRadiologyTests, "radiology", "tests");
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientRadiologyResults, "radiology", "results");
	}

	private void fillCalendarLinks(Patient patient) {
		appendRelatedLink(patient, LinkTitles.PatientCalendarEvents, "calendar", "calendarEvent");
		appendRelatedLink(patient, LinkTitles.PatientCalendarPublicKey, "calendar", "key");
		appendRelatedLink(patient, LinkTitles.PatientPastAppointments, "appointments", "past");
		appendRelatedLink(patient, LinkTitles.PatientFutureAppointments, "appointments", "future");
		appendRelatedLink(patient, LinkTitles.PatientAppointments, "appointments");
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientAppointments, "appointments");
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientMentalHealthAppointments, "appointments", "mental-health");
	}

	private void fillProblemLinks(Patient patient) {
		appendRelatedLinksWithScopes(patient, LinkTitles.PatientProblems, "problems");
		appendRelatedLink(patient, LinkTitles.PatientOutpatientProblems, "problems");
		appendRelatedLink(patient, LinkTitles.PatientInpatientProblems, "problems");
	}
	
	private void fillToolTrackingResultLinks(Patient patient) {
		appendRelatedLink(patient, LinkTitles.ToolTrackingResults, "toolTracking");
	}

	private void appendRelatedLink(Patient patient, String uriTitle, PatientDataLinkBuilder<?, ?> patientDataLinkBuilder) {
		URI uri = patientDataLinkBuilder.get      icalUriForCollection(patient.getSelfUri());
		appendRelatedLink(patient, uriTitle, uri);
	}

	private AtomLink appendRelatedLink(Patient patient, String uriTitle, URI uri) {
		return patient.getLink().addRelatedLink(uriTitle, uri);
	}

	private void appendRelatedLinksWithScopes(Patient patient, String uriTitle, String... uriPath) {
		appendRelatedLink(patient, uriTitle, ArrayAppender.append(uriPath, "scope", LONGITUDINAL));
		appendRelatedLink(patient, uriTitle + "-longitudinal", ArrayAppender.append(uriPath, "scope", LONGITUDINAL));
		appendRelatedLink(patient, uriTitle + "-operational", ArrayAppender.append(uriPath, "scope", OPERATIONAL));
	}

	private AtomLink appendRelatedLink(Patient patient, String uriTitle, String... uriPath) {
		UriBuilder uriBuilder = UriBuilder.fromUri(patient.getSelfUri());
		for (String singleUriPath : uriPath) {
			uriBuilder.path(singleUriPath);
		}
		URI uri = uriBuilder.build();
		return appendRelatedLink(patient, uriTitle, uri);
	}

	private URI getBaseUri() {
		return this.baseUri;
	}

}