package com.agilex.healthcare.mobilehealthplatform.utils.uriformaters;

import java.net.URI;
import java.net.URISyntaxException;

import javax.annotation.Resource;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Service;

import com.agilex.healthcare.mobilehealthplatform.domain.AtomLink;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.ResourceDirectory;
import com.agilex.healthcare.mobilehealthplatform.utils.PropertyHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.atom.AtomUtility;
import com.sun.syndication.feed.synd.SyndFeed;

@Service
@Scope(value=ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class ResourceDirectoryBuilder {

	@Resource
	private PropertyHelper propertyHelper;

	private static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(ResourceDirectoryBuilder.class);

    private static final String MHP_LOGIN_URL = "oauth.login_url";
	private static final String MHP_AUTHORIZE_URL = "oauth.authorize_url";
	private static final String MHP_TOKEN_URL = "oauth.token_url";
	private static final String EAS_URL = "easUrl";
	
	public ResourceDirectoryBuilder() {
	}

	public ResourceDirectory getResourceDirectory(URI systemBaseUri, URI resourceDirectoryUri) {

		ResourceDirectory directory = new ResourceDirectory();
		AtomLink link;

		link = new AtomLink();
		link.setHref(resourceDirectoryUri);
		// link.setType(MediaType.APPLICATION_ATOM_XML);
		link.setTitle("resource-directory");
		directory.getLink().setSelfLink(link);

		link = new AtomLink();
		link.setHref(systemBaseUri);
		link.setType(MediaType.APPLICATION_ATOM_XML);
		link.setTitle(LinkTitles.SystemRoot);
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("user-session").build());
		link.setType(MediaType.APPLICATION_XML);
		link.setTitle(LinkTitles.UserSession);
		directory.getLink().add(link);
		
		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("public").path("antibiogram").path("facility").build());
		link.setType(MediaType.APPLICATION_XML);
		link.setTitle(LinkTitles.AntibiogramFacility);
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("public").path("user-session").build());
		link.setType(MediaType.APPLICATION_XML);
		link.setTitle(LinkTitles.PublicUserSession);
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("mhpuser").build());
		link.setType(MediaType.APPLICATION_JSON);
		link.setTitle("mhpuser");
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("public").path("health").queryParam("e", "").build());
		link.setType(MediaType.TEXT_PLAIN);
		link.setTitle("health-check");
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("patients").build());
		link.setTitle("patients");
		link.setType(MediaType.APPLICATION_XML);
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("metrics").build());
		link.setTitle("system-metrics");
		link.setType(MediaType.APPLICATION_XML);
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("provider").path("appointments").build());
		link.setTitle("current-provider-appointments");
		link.setType(MediaType.APPLICATION_XML);
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("notification-service").path("opt-out").path("enabled").build());
		link.setTitle("notification-opt-out-enabled");
		link.setType(MediaType.APPLICATION_XML);
		directory.getLink().add(link);
		
		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("facility-service").path("facilities").build());
		link.setTitle("facility-service");
		link.setType(MediaType.APPLICATION_XML);
		directory.getLink().add(link);

		link = new AtomLink();
		boolean useDefault = true;
		String mhpAuthorizeUrl = propertyHelper.getProperty(MHP_AUTHORIZE_URL);
		try {
			if (NullChecker.isNotNullish(mhpAuthorizeUrl)) {
				URI mhpAuthorizeUri = new URI(mhpAuthorizeUrl);
				link.setHref(mhpAuthorizeUri);
				useDefault = false;
			}
		} catch (URISyntaxException e) {
			LOGGER.error("Mhp authorize URL is invalid", e);
		}
		if (useDefault) {
			link.setHref(UriBuilder.fromUri(stripRestFromBaseURI(systemBaseUri)).path("oauth").path("authorize").build());
		}
		link.setTitle(LinkTitles.OAuthAuthorize);
		link.setType(MediaType.TEXT_HTML);
		directory.getLink().add(link);

        link = new AtomLink();
        useDefault = true;
        String mhpLoginUrl = propertyHelper.getProperty(MHP_LOGIN_URL);
        try {
            if (NullChecker.isNotNullish(mhpLoginUrl)) {
                URI mhpLoginUri = new URI(mhpLoginUrl);
                link.setHref(mhpLoginUri);
                useDefault = false;
            }
        } catch (URISyntaxException e) {
            LOGGER.error("Mhp login URL is invalid", e);
        }
        if (useDefault) {
            link.setHref(UriBuilder.fromUri(stripRestFromBaseURI(systemBaseUri)).path("oauthlogin").build());
        }
        link.setTitle(LinkTitles.OAuthLogin);
        link.setType(MediaType.TEXT_HTML);
        directory.getLink().add(link);

		link = new AtomLink();
		useDefault = true;
		String mhpTokenUrl = propertyHelper.getProperty(MHP_TOKEN_URL);
		try {
			if (NullChecker.isNotNullish(mhpTokenUrl)) {
				URI mhpTokenUri = new URI(mhpTokenUrl);
				link.setHref(mhpTokenUri);
				useDefault = false;
			}
		} catch (URISyntaxException e) {
			LOGGER.error("Mhp token URL is invalid", e);
		}
		if (useDefault) {
			link.setHref(UriBuilder.fromUri(stripRestFromBaseURI(systemBaseUri)).path("oauth").path("token").build());
		}	
		link.setTitle(LinkTitles.OAuthToken);
		link.setType(MediaType.TEXT_HTML);
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(stripRestFromBaseURI(systemBaseUri)).path("j_spring_security_check").build());
		link.setTitle("login-credentials");
		link.setType(MediaType.TEXT_HTML);
		directory.getLink().add(link);



		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(stripRestFromBaseURI(systemBaseUri)).path("logout").build());
		link.setTitle("logout");
		link.setType(MediaType.TEXT_HTML);
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("public").path("feedback").build());
		link.setTitle("feedback");
		link.setType(MediaType.TEXT_HTML);
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("public").path("feedback").path("appointment-request").build());
		link.setTitle(LinkTitles.AppointmentRequestFeedback);
		link.setType(MediaType.APPLICATION_JSON);
		directory.getLink().add(link);
		
		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("public").path("learn").path("conditions").build());
		link.setTitle("learn-conditions");
		link.setType(MediaType.APPLICATION_JSON);
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("public").path("assessments").build());
		link.setTitle("assessments");
		link.setType(MediaType.APPLICATION_XML);
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("public").path("assessments").path("health-assessment").build());
		link.setType(MediaType.APPLICATION_JSON);
		link.setTitle(LinkTitles.HealthAssessmentAssessments);
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("appointment-service").path("metadata").build());
		link.setType(MediaType.APPLICATION_JSON);
		link.setTitle(LinkTitles.AppointmentsMetadata);
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("appointment-service").path("appointment-requests").build());
		link.setType(MediaType.APPLICATION_JSON);
		link.setTitle(LinkTitles.AppointmentRequests);
		directory.getLink().add(link);
		
		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("appointment-service").path("metrics").path("no-icn-or-dfn").build());
		link.setType(MediaType.APPLICATION_JSON);
		link.setTitle(LinkTitles.AppointmentMetricsNoIcnOrDfn);
		directory.getLink().add(link);
		
		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("appointment-service").path("metrics").path("no-dfn-cancel-request").build());
		link.setType(MediaType.APPLICATION_JSON);
		link.setTitle(LinkTitles.AppointmentMetricsNoDfnCancelRequest);
		directory.getLink().add(link);
		
		link = new AtomLink();
		String easUrl = propertyHelper.getProperty(EAS_URL);
		link.setHref(UriBuilder.fromUri(easUrl).build());
		link.setType(MediaType.TEXT_HTML);
		link.setTitle("eas");
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("public").path("assessments").path("results").build());
		link.setType(MediaType.APPLICATION_JSON);
		link.setTitle(LinkTitles.AssessmentResults);
		directory.getLink().add(link);
		
		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("authorizationCheck").build());
		link.setType(MediaType.APPLICATION_JSON);
		link.setTitle("authorizationCheck");
		directory.getLink().add(link);
		
		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("public").path("mygoals").build());
		link.setType(MediaType.APPLICATION_JSON);
		link.setTitle(LinkTitles.MyGoalsInventory);
		directory.getLink().add(link);

		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("public").path("customdataview").build());
		link.setType(MediaType.APPLICATION_JSON);
		link.setTitle(LinkTitles.CustomDataView);
		directory.getLink().add(link);
		
		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("public").path("launchpad").path("metadata").build());
		link.setType(MediaType.APPLICATION_JSON);
		link.setTitle(LinkTitles.LaunchpadMetadata);
		directory.getLink().add(link);
		
		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("public").path("launchpad").path("launchpadItem").build());
		link.setType(MediaType.APPLICATION_JSON);
		link.setTitle(LinkTitles.LaunchpadItem);
		directory.getLink().add(link);
		
		link = new AtomLink();
		link.setHref(UriBuilder.fromUri("http://labtestsonline.org").build());
		link.setTitle(LinkTitles.LabTestReferenceLink);
		directory.getLink().add(link);
		
		link = new AtomLink();
		link.setHref(UriBuilder.fromUri(systemBaseUri).path("public").path("domains").build());
		link.setTitle(LinkTitles.MbbDomains);
		directory.getLink().add(link);

        link = new AtomLink();
        link.setHref(UriBuilder.fromUri(stripWebContextFromBaseURI(systemBaseUri)).path("launchpad").build());
        link.setTitle(LinkTitles.LaunchPadLink);
        link.setType(MediaType.TEXT_HTML);
        directory.getLink().add(link);
		
		addTokenLink(systemBaseUri, directory);
        addLastAccessTimeLink(systemBaseUri, directory);

		return directory;
	}

	private void addTokenLink(URI systemBaseUri, ResourceDirectory directory) {
		AtomLink link = new AtomLink();
		link.setTitle("token");
		link.setHref(UriBuilder.fromUri(stripWebContextFromBaseURI(systemBaseUri)).path("AuthorizationServices/rest/token").build());

		directory.getLink().add(link);
	}

    private void addLastAccessTimeLink(URI systemBaseUri, ResourceDirectory directory) {
        AtomLink link = new AtomLink();
        link.setTitle("last-accesstime");
        link.setHref(UriBuilder.fromUri(stripWebContextFromBaseURI(systemBaseUri)).path("AuthorizationServices/rest/resourceLastAccessedTime").build());

        directory.getLink().add(link);
    }
	
	public SyndFeed getResourceDirectoryAsAtomFeed(URI systemBaseUri, URI resourceDirectoryUri) {
		ResourceDirectory resourceDirectory = getResourceDirectory(systemBaseUri, resourceDirectoryUri);
		SyndFeed feed = AtomUtility.createAtomFeed(resourceDirectory.getLink(), resourceDirectoryUri, "mhp-resources");
		return feed;
	}

	private static URI stripRestFromBaseURI(URI baseURI) {

		String stringURI = baseURI.toString();
		stringURI = stringURI.replaceAll("/rest", "");

		URI uri = URI.create(stringURI);

		return uri;
	}
	
	private static URI stripWebContextFromBaseURI(URI baseURI) {

		String stringURI = baseURI.toString();
		stringURI = stringURI.replaceAll("/MobileHealthPlatformWeb/rest", "");

		URI uri = URI.create(stringURI);

		return uri;
	}
	
}