package com.agilex.healthcare.mobilehealthplatform.utils;

import java.net.URI;

import com.agilex.healthcare.mobilehealthplatform.domain.PatientData;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientDataCollection;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import com.agilex.healthcare.mobilehealthplatform.domain.Problem;
import com.agilex.healthcare.mobilehealthplatform.domain.Problems;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.MhpObjectFactory;
import com.agilex.healthcare.utility.NullChecker;

public class PatientIdentifierHelper {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientIdentifierHelper.class);

	public static String getPatientIdFromUri(URI uri) {
		String id = null;
		if (uri != null)
			id = getPatientIdFromUri(uri.toString());
		return id;
	}

	public static String getPatientIdFromUri(String uri) {
		String id = null;
		if (NullChecker.isNotNullish(uri)) {
			final String patientPattern = "/patient/";
			int locationOfPattern = uri.indexOf(patientPattern);
			if (locationOfPattern > 0) {
				int idStart = locationOfPattern + patientPattern.length();
				id = uri.substring(idStart);
			}
		}
		return id;
	}

	public static void updatePatientIdentifier(PatientDataCollection<?> dataCollection, PatientIdentifier patientIdentifier) {
		logger.debug("updating patient identifier for collection");
		dataCollection.setPatientIdentifier(patientIdentifier);

		for (PatientData data : dataCollection) {
			updatePatientIdentifier(data, patientIdentifier);
		}
	}

	public static void updatePatientIdentifier(PatientData data, PatientIdentifier patientIdentifier) {
		if (data != null)
			data.setPatientIdentifier(patientIdentifier);
	}

	public static void setAssigningAuthority(PatientDataCollection<?> dataCollection, String assigningAuthority) {
		for (PatientData data : dataCollection) {
			setAssigningAuthority(data, assigningAuthority);
		}
	}

	public static void setAssigningAuthority(PatientData data, String assigningAuthority) {
		data.getPatientIdentifier().setAssigningAuthority(assigningAuthority);
	}

	public static void updateCommunity(Problems problems, String communityId) {
		for (Problem problem : problems) {
			problem.getDataIdentifier().setSystemId(communityId);
		}
	}
	
	public static PatientIdentifiers translatePatientIdentifier(PatientIdentifier patientIdentifier, String targetAssigningAuthority) {

		PatientIdentifiers matcingPatientIdentifiers = new PatientIdentifiers();

        try {
            if(targetAssigningAuthority.equals(patientIdentifier.getAssigningAuthority())){
                logger.debug("No translations are neccessary as patient's assigning authority and target assigning authority are same for the patient identifier");
                matcingPatientIdentifiers.add(patientIdentifier);
                return matcingPatientIdentifiers;
            }

            // Get all the identifiers for given patient identifier
            PatientIdentifiers correspondingIdentifiers = MhpObjectFactory.getInstance().getPatientCorrelationService()
                    .getCorrespondIdentifiers(patientIdentifier);


            // Get all the identifiers matching the target assigning authority
            for (PatientIdentifier possibleMatch : correspondingIdentifiers) {
                if (possibleMatch.getAssigningAuthority().toLowerCase().contains(targetAssigningAuthority.toLowerCase())) {
                    matcingPatientIdentifiers.add(possibleMatch);
                }
            }

            logger.debug("translated patient identifier for assigning authority into corresponding identifiers");
        }
        catch(Exception e) {
            logger.error("Caught a " + e.getClass().getName() + ", returning empty object: " + e.toString());
        }

        return matcingPatientIdentifiers;
	}
	
	public static PatientIdentifier getDfnPatientidentifierForQuery(PatientIdentifiers identifiers){
		PatientIdentifier dfnPatientIdentifier = new PatientIdentifier();
		if(NullChecker.isNullish(identifiers))
			return dfnPatientIdentifier;

		StringBuffer stations = new StringBuffer();
		StringBuffer uniqueIds = new StringBuffer();
		int i=0;
		for(PatientIdentifier identifier : identifiers){
			if( i > 0){
				uniqueIds.append(",");
				stations.append(",");
			}
			stations.append(identifier.getAssigningAuthority().substring(4));
			uniqueIds.append(identifier.getUniqueId());
			i++;
		}
		//logger.debug("uniqueIds:"+uniqueIds.toString());
		//logger.debug("stations:"+stations.toString());
		dfnPatientIdentifier.setUniqueId(uniqueIds.toString());
		dfnPatientIdentifier.setAssigningAuthority(stations.toString());
		return dfnPatientIdentifier;
	}
}
