package com.agilex.healthcare.mobilehealthplatform.serviceregistry;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;

import com.agilex.healthcare.utility.NullChecker;

public class ServiceRegistry {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(ServiceRegistry.class);

	private static ServiceRegistry _instance = new ServiceRegistry();
	private Map<String, Community> communityMapIndexedByCommunityId = new ConcurrentHashMap<String, Community>();
	private Map<String, Community> communityMapIndexedByAssigningAuthorityId = new ConcurrentHashMap<String, Community>();
	private String primaryCommunityId = null;

	static {
		getInstance().reset();
	}

	public static ServiceRegistry getInstance() {
		return _instance;
	}

	public static ServiceRegistry getLocalInstance() {
		return new ServiceRegistry();
	}

	public List<Community> getCommunities() {
		if (communityMapIndexedByCommunityId == null)
			reset();

		List<Community> communityList = new ArrayList<Community>();
		for (Entry<String, Community> entry : communityMapIndexedByCommunityId.entrySet())
			communityList.add(entry.getValue());
		return communityList;

	}

	public Community getPrimaryCommunity() {
		return getCommunityById(primaryCommunityId);
	}

	public Community getCommunityById(String homeCommunityId) {
		return getCommunityById(homeCommunityId, true);
	}

	public Community getCommunityById(String homeCommunityId, boolean warnIfNotFound) {
		Community matchedCommunity = communityMapIndexedByCommunityId.get(homeCommunityId);

		if (matchedCommunity == null)
			logger.warn("no match found when looking up community");

		return matchedCommunity;
	}

	public Community getCommunityByAssigningAuthority(String assigningAuthority) {
		return communityMapIndexedByAssigningAuthorityId.get(assigningAuthority);
	}

	public ServiceRegistry registerCommunity(Community community) {
		if (NullChecker.isNullish(community.getId()))
			throw new IllegalArgumentException("Home community is required");
		if (NullChecker.isNullish(community.getAssigningAuthority()))
			throw new IllegalArgumentException("Assigning authority is required");

		Community existingRegistration = getCommunityById(community.getId(), false);
		if (existingRegistration != null) {
			unregisterCommunity(existingRegistration);
		}

		communityMapIndexedByCommunityId.put(community.getId(), community);
		communityMapIndexedByAssigningAuthorityId.put(community.getAssigningAuthority(), community);
		return this;
	}

	private void unregisterCommunity(Community community) {
		if (NullChecker.isNullish(community.getId()))
			throw new IllegalArgumentException("Home community is required");
		if (NullChecker.isNullish(community.getAssigningAuthority()))
			throw new IllegalArgumentException("Assigning authority is required");

		communityMapIndexedByCommunityId.remove(community.getId());
		communityMapIndexedByAssigningAuthorityId.remove(community.getAssigningAuthority());
	}

	public void registerPrimaryCommunity(Community community) {
		registerCommunity(community);
		setPrimaryCommunityId(community.getId());
	}

	public void setPrimaryCommunityId(String communityId) {
		if (this.getCommunityById(communityId) == null)
			throw new IllegalArgumentException(
					"In order to set the primary community id, the community must be registered.  Either first utilize register(Community) or 'registerPrimaryCommunity(Community)'");
		primaryCommunityId = communityId;
	}

	public ServiceRegistry reset() {
		communityMapIndexedByCommunityId = new ConcurrentHashMap<String, Community>();
		communityMapIndexedByAssigningAuthorityId = new ConcurrentHashMap<String, Community>();

		resetForNormalMode();
		return this;
	}

	private void resetForNormalMode() {
		Community community = new Community("default", "default", new MhpObjectFactory());
		registerPrimaryCommunity(community);
	}

	public ServiceRegistry clear() {
		communityMapIndexedByCommunityId.clear();
		return this;
	}

}
