package com.agilex.healthcare.mobilehealthplatform.serviceregistry;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.w3c.dom.Document;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestHandler;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataFetchRequestReader;
import com.agilex.healthcare.utility.ModeHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.XmlHelper;
import com.agilex.healthcare.utility.XpathHelper;

public class DomainServiceRegistry {
	private static final String PREFERRED_CONFIGURATION_TEMPLATE = "ServiceRegistry.%s.conf.xml";
	private static final String DEFAULT_CONFIGURATION_FILE = "ServiceRegistry.conf.xml";
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(DomainServiceRegistry.class);
	private MhpObjectFactory mhpObjectFactory;
	private static Document cachedConfiguration;

	public DomainServiceRegistry() {
		this.mhpObjectFactory = MhpObjectFactory.getInstance();
	}

	public DomainServiceRegistry(MhpObjectFactory mhpObjectFactory) {
		if (mhpObjectFactory == null)
			mhpObjectFactory = MhpObjectFactory.getInstance();
		this.mhpObjectFactory = mhpObjectFactory;
	}

	public Collection<DataSystem> getDataSystems(ScopeFilter scopeFilter, String domain) {
		Collection<DataSystem> dataSystems = new ArrayList<DataSystem>();
		if (NullChecker.isNotNullish(scopeFilter.getSystemIdentifier())) {
			DataSystem dataSystem = getDataSystemByName(scopeFilter.getSystemIdentifier());
			dataSystems.add(dataSystem);
		} else {
			dataSystems = getDataSystemByScopeDomain(scopeFilter.getScopePreset(), domain);
		}
		return dataSystems;
	}

	private String getDefaultSystemNameByScope(String scope) {
		final String xpath = String.format("/serviceRegistry/scopedomain-to-system-mapping[@scope='%s'][@domain='%s']", scope, "*");
		return XpathHelper.getString(getConfiguration(), xpath);
	}

	private Document getConfiguration() {
		if (cachedConfiguration == null) {
			loadConfigurationIntoCache();
		}
		return cachedConfiguration;
	}

	private synchronized void loadConfigurationIntoCache() {
		if (cachedConfiguration == null) {
			// TODO: consider caching this, or any of the methods in this class
			String mode = ModeHelper.getCurrentMode();
			InputStream configurationStream = getConfigurationStream(String.format(PREFERRED_CONFIGURATION_TEMPLATE, mode), DEFAULT_CONFIGURATION_FILE);
			cachedConfiguration = XmlHelper.loadXml(configurationStream);
			if (configurationStream != null) {
				try {
					configurationStream.close();
				} catch (IOException e) {
					logger.warn("Could not release stream.");
				}
			}
		}
	}

	private InputStream getConfigurationStream(String preferredResourceName, String backupResourceName) {
		InputStream stream = getConfigurationStream(preferredResourceName);
		if (stream == null) {
			stream = getConfigurationStream(backupResourceName);
		}
		return stream;
	}

	private InputStream getConfigurationStream(String resourceName) {
		InputStream stream = this.getClass().getResourceAsStream(resourceName);
		logger.debug(String.format("loading configuration from resource file [%s]", resourceName));
		return stream;
	}

	public <T> List<SystemsDataLayerFactory<T>> getDataLayerFactoriesByScopeFilter(ScopeFilter scopeFilter, String domain) {
		List<SystemsDataLayerFactory<T>> datalayerfactories;
		if (NullChecker.isNotNullish(scopeFilter.getSystemIdentifier())) {
			datalayerfactories = getDataLayerFactoriesBySystemDomain(scopeFilter.getSystemIdentifier(), domain);
		} else {
			datalayerfactories = getDataLayerFactoriesByScopeDomain(scopeFilter.getScopePreset(), domain);
		}
		return datalayerfactories;
	}

	public <T> SystemsDataLayerFactory<T> getSingleDataLayerFactoriesByScopeFilter(ScopeFilter scopeFilter, String domain) {
		List<SystemsDataLayerFactory<T>> systemList = getDataLayerFactoriesByScopeFilter(scopeFilter, domain);

		if (systemList == null || systemList.size() == 0)
			throw new RuntimeException(String.format("Failed to determine system for [scope=%s][domain=%s]", scopeFilter, domain));

		return systemList.get(0);
	}

	public <T> T getDataLayerByScopeFilter(ScopeFilter scopeFilter, String domain) {
		SystemsDataLayerFactory<T> singleDataLayerFactoriesByScopeFilter = this.getSingleDataLayerFactoriesByScopeFilter(scopeFilter, domain);
		T datalayer = singleDataLayerFactoriesByScopeFilter.getDataLayerInstance();
		return datalayer;
	}

	public <T> T getDataLayerBySystem(String systemIdentifier, String domain) {
		return getDataLayerByScopeFilter(ScopeFilter.getInstanceForSystem(systemIdentifier), domain);
	}

	public <T> T getDataLayerByScope(String scope, String domain) {
		return getDataLayerByScopeFilter(ScopeFilter.getInstanceForScope(scope), domain);
	}

	public <T> T getDataLayerByMessage(RequestMessage requestMessage) {
		PatientDataFetchRequestReader messageReader = PatientDataFetchRequestReader.fromRequest(requestMessage);
		DomainServiceRegistry serviceRegistry = new DomainServiceRegistry(mhpObjectFactory);

		String systemId = extractSystemId(messageReader);
		String scope = extractScope(messageReader);

		T datalayer;
		if (NullChecker.isNotNullish(systemId)) {
			datalayer = serviceRegistry.getDataLayerBySystem(systemId, messageReader.getDomain());
		} else {
			datalayer = serviceRegistry.getDataLayerByScope(scope, messageReader.getDomain());
		}
		return datalayer;
	}

	public DataSystem getSystemByMessage(RequestMessage requestMessage) {
		PatientDataFetchRequestReader messageReader = PatientDataFetchRequestReader.fromRequest(requestMessage);

		String systemId = extractSystemId(messageReader);

		if (NullChecker.isNullish(systemId)) {
			String scope = extractScope(messageReader);
			systemId = getDataSystemNameByScopeDomain(scope, messageReader.getDomain());
		}

		DataSystem system = getDataSystemByName(systemId);

		return system;
	}

	private String extractSystemId(PatientDataFetchRequestReader messageReader) {
		String systemId;
		if ((messageReader.getScopeFilter() != null) && (NullChecker.isNotNullish(messageReader.getScopeFilter().getSystemIdentifier()))) {
			systemId = messageReader.getScopeFilter().getSystemIdentifier();
		} else if (messageReader.getDataIdentifier() != null && NullChecker.isNotNullish(messageReader.getDataIdentifier().getSystemId())) {
			systemId = messageReader.getDataIdentifier().getSystemId();
		} else {
			systemId = null;
		}
		return systemId;
	}

	private String extractScope(PatientDataFetchRequestReader messageReader) {
		String scope;
		if ((messageReader.getScopeFilter() != null) && (NullChecker.isNotNullish(messageReader.getScopeFilter().getScopePreset()))) {
			scope = messageReader.getScopeFilter().getScopePreset();
		} else {
			scope = null;
		}
		return scope;
	}

	public <T> List<SystemsDataLayerFactory<T>> getDataLayerFactoriesByScopeDomain(String scope, String domain) {
		List<SystemsDataLayerFactory<T>> datalayerfactories = new ArrayList<SystemsDataLayerFactory<T>>();

		logger.debug(String.format("begin getDataLayerFactoriesByScopeDomain [scope=%s][domain=%s]", scope, domain));
		String datasystems = getDataSystemNameByScopeDomain(scope, domain);
		logger.debug(String.format("[scope=%s][domain=%s]=>[datasystems=%s]", scope, domain, datasystems));
		if (NullChecker.isNullish(datasystems))
			return datalayerfactories;

		String[] datasystemArray = datasystems.split(",");
		for (String datasystem : datasystemArray) {
			List<SystemsDataLayerFactory<T>> dataLayerFactoriesForSystem = getDataLayerFactoriesBySystemDomain(datasystem, domain);
			datalayerfactories.addAll(dataLayerFactoriesForSystem);
		}

		logger.debug(String.format("end getDataLayerFactoriesByScopeDomain [scope=%s][domain=%s]=>[%s factory(s)]", scope, domain, datalayerfactories.size()));
		return datalayerfactories;
	}

	public <T> List<SystemsDataLayerFactory<T>> getDataLayerFactoriesBySystemDomain(String datasystemName, String domain) {
		List<SystemsDataLayerFactory<T>> datalayerfactories = new ArrayList<SystemsDataLayerFactory<T>>();

		String beanname = getBeanNameBySystemDomain(datasystemName, domain);
		logger.debug(String.format("[datasystem=%s][domain=%s]=>[beanname=%s]", datasystemName, domain, beanname));

		T datalayer = this.mhpObjectFactory.getBean(beanname);
		DataSystem system = getDataSystemByName(datasystemName);
		SystemsDataLayerFactory<T> datalayerfactory = SystemsDataLayerFactory.getInstance(datalayer, system);
		datalayerfactories.add(datalayerfactory);

		return datalayerfactories;
	}

	private String getBeanNameBySystemDomain(String system, String domain) {
		return system + "." + domain + "." + "datalayer";
	}

	public DataSystem getDataSystemByName(String datasystemName) {
		final String assigningAuthorityXpath = String.format("/serviceRegistry/system[@name='%s']/assigningAuthority", datasystemName);
		String assigningAuthority = XpathHelper.getString(getConfiguration(), assigningAuthorityXpath);

		DataSystem system = new DataSystem(datasystemName, assigningAuthority);
		return system;
	}

	private Collection<DataSystem> getDataSystemByScopeDomain(String scope, String domain) {
		Collection<DataSystem> dataSystems = new ArrayList<DataSystem>();
		String dataSystemNames = getDataSystemNameByScopeDomain(scope, domain);
		logger.debug(String.format("[scope=%s][domain=%s]=>[datasystems=%s]", scope, domain, dataSystemNames));
		if (NullChecker.isNullish(dataSystemNames))
			return dataSystems;

		String[] datasystemArray = dataSystemNames.split(",");
		for (String datasystemName : datasystemArray) {
			DataSystem dataSystem = getDataSystemByName(datasystemName);
			dataSystems.add(dataSystem);
		}

		logger.debug(String.format("end get datasystems [scope=%s][domain=%s]=>[%s systems]", scope, domain, dataSystems.size()));
		return dataSystems;
	}

	public String getDataSystemNameByScopeDomain(String scope, String domain) {
		logger.debug(String.format("looking up scope-domain to system mapping.  There are a total of %s entries", XpathHelper.getNodeList(getConfiguration(), "/serviceRegistry/scopedomain-to-system-mapping").getLength()));
		final String xpath = String.format("/serviceRegistry/scopedomain-to-system-mapping[@scope='%s'][@domain='%s']", scope, domain);
		String datasystem = XpathHelper.getString(getConfiguration(), xpath);
		if (NullChecker.isNullish(datasystem))
			datasystem = getDefaultSystemNameByScope(scope);
		if (NullChecker.isNullish(datasystem))
			datasystem = null;
		return datasystem;
	}

	public RequestHandler getRequestHandlerByRequestType(RequestMessage requestMessage) {
		// i would like to see three strategies:
		// 1) determine beanname by convention: {domain}.{requesttype}.handler
		// (allergy.retrievelist.handler)
		// 2) specify beanname in ServiceRegistry configuration
		// 3) discover by asking all handlers "can you handle?"

		String beanName = String.format("%s.handler", requestMessage.getType());
		RequestHandler handler = mhpObjectFactory.getBean(beanName);
		return handler;
	}

}
