package com.agilex.healthcare.mobilehealthplatform.security;

import com.agilex.healthcare.mobilehealthplatform.datalayer.securemessage.SecureMessageUserVerificationCache;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import org.springframework.security.access.AccessDecisionVoter;
import org.springframework.security.access.ConfigAttribute;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.web.FilterInvocation;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import java.util.Collection;

/*
    This should only be configured to vote on '/securemessage-service/' URIs
 */
public class SecureMessageUserVoter<S> implements AccessDecisionVoter<S> {

    private final String SM_USER_SESSION_KEY = "SecureMessage.Id";

    @Override
    public boolean supports(ConfigAttribute attribute) {
        return "IS_SECURE_MESSAGE_USER".equals(attribute.getAttribute());
    }

    @Override
    public boolean supports(Class<?> clazz) {
        return true;
    }

    @Override
    public int vote(Authentication authentication, S s, Collection<ConfigAttribute> attributes) {


        for(ConfigAttribute attribute : attributes) {
            if(this.supports(attribute)) {
                return isViewingOwnData(authentication, s);
            }
        }
        return ACCESS_ABSTAIN;
    }

    int isViewingOwnData(Authentication authentication, S s) {
        if(!isConsumer(authentication.getAuthorities())) return ACCESS_DENIED;

        String path = VoterHelper.getPath((FilterInvocation) s);
        PatientIdentifier patientIdentifier = VoterHelper.getPatientIdentifierFromUri(path);

        String userId = fetchUserId(authentication, s);
        if(!"SM".equals(patientIdentifier.getAssigningAuthority()))
            return ACCESS_ABSTAIN; // Voter not equipped to handle non-SM assigning authorities
        else if(userId != null && userId.equals(patientIdentifier.getUniqueId()))
            return ACCESS_GRANTED;
        else
            return ACCESS_DENIED;

    }

    private String fetchUserId(Authentication authentication, S s) {
        String userId = fetchUserIdFromSession(s);

        if(userId == null) {
            userId = fetchUserIdFromStorage(authentication);
            setSecureMessageIdInSession(s, userId);
        }

        return userId;
    }

    private String fetchUserIdFromSession(S s) {
        HttpServletRequest request = ((FilterInvocation) s).getHttpRequest();
        HttpSession session = request.getSession();
        String userId = (String) session.getAttribute(SM_USER_SESSION_KEY);

        return userId;
    }

    private String fetchUserIdFromStorage(Authentication authentication) {
        PatientIdentifier icnIdentifier = fetchICNForUser(authentication);

        SecureMessageUserVerificationCache dataService = SecureMessageUserVerificationCache.getInstance();
        return dataService.fetchUserId(icnIdentifier);
    }

    private PatientIdentifier fetchICNForUser(Authentication authentication) {
        PatientIdentifiers patientIdentifiers = VoterHelper.getPermittedPatientIdentifiers(authentication);

        PatientIdentifier found = null;
        for(PatientIdentifier patientIdentifier : patientIdentifiers) {
            if("EDIPI".equals(patientIdentifier.getAssigningAuthority())) {
                found = patientIdentifier;
                break;
            }
        }

        return found;
    }

    private void setSecureMessageIdInSession(S s, String userId) {
        HttpServletRequest request = ((FilterInvocation) s).getHttpRequest();
        HttpSession session = request.getSession();
        session.setAttribute(SM_USER_SESSION_KEY, userId);
    }

    private boolean isConsumer(
            Collection<? extends GrantedAuthority> authorities) {
        return VoterHelper.hasRole(authorities, Roles.ROLE_CONSUMER);
    }
}
