package com.agilex.healthcare.mobilehealthplatform.security;

import java.util.Collection;

import javax.servlet.http.HttpSession;

import org.springframework.security.access.AccessDecisionVoter;
import org.springframework.security.access.ConfigAttribute;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.web.FilterInvocation;

import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import com.agilex.healthcare.utility.NullChecker;

//additional change made on PatientVoter branch
public class PatientUserVoter<S> implements AccessDecisionVoter<S> {

    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientUserVoter.class);

    @Override
    public boolean supports(ConfigAttribute attribute) {
        return "IS_PATIENT_OR_ADVOCATE".equals(attribute.getAttribute());
    }

    @Override
    public boolean supports(Class<?> clazz) {
        return true;
    }

    // Abstain-able vote
    @Override
    public int vote(Authentication authentication, S s,
                    Collection<ConfigAttribute> attributes) {

        for (ConfigAttribute attribute : attributes) {
            if (this.supports(attribute)) {
                // authentication.authorities
                if (isConsumer(authentication.getAuthorities())) {
                    int roaVote = voteOnRoa(s, authentication);
                    if (roaVote == AccessDecisionVoter.ACCESS_DENIED) {
                        return roaVote;
                    }

                    if (checkIsViewingPermittedPatient(authentication, s)) {
                        return AccessDecisionVoter.ACCESS_GRANTED;
                    } else {
                        return AccessDecisionVoter.ACCESS_DENIED;
                    }
                } else {
                    //Not a patient role, so abstain
                    return AccessDecisionVoter.ACCESS_ABSTAIN;
                }
            }
        }
        return AccessDecisionVoter.ACCESS_ABSTAIN;
    }

    private int voteOnRoa(S s, Authentication authentication) {

        logger.debug("Checking Right-of-Access...");

        HttpSession session = VoterHelper.getHttpSession(s);
        final String roaSessionKey = "PatientUserVoter.hasAcceptedRightOfAccess";
        Boolean hasAcceptedRightOfAccess = (Boolean) session.getAttribute(roaSessionKey);

        //if value is null, make dataservice call
        if (hasAcceptedRightOfAccess == null) {
            logger.debug("Right-of-Access not currently stored in session, checking elsewhere...");
            MhpUser user = MhpUserFactory.createFromAuthentication(authentication);
            hasAcceptedRightOfAccess = VoterHelper.getRightOfAccessAccepted(user);

            //Update session if value no longer null
            if (hasAcceptedRightOfAccess != null && hasAcceptedRightOfAccess) {
                logger.debug("Right of access found... evaluating");
                session.setAttribute(roaSessionKey, hasAcceptedRightOfAccess);
            }
        }

        //use roa value for decision
        if (hasAcceptedRightOfAccess == null) {
            logger.debug("User does not have right of access: Denied");
            return AccessDecisionVoter.ACCESS_DENIED;
        }
        if (!hasAcceptedRightOfAccess) {
            logger.debug("User refused Right-of-access: Denied");
            //user has not accepted, deny access
            return AccessDecisionVoter.ACCESS_DENIED;
        } else {
            logger.debug("User accepted Right-of-access: Abstain");
            //user has accepted, allow, but do not grant access
            return AccessDecisionVoter.ACCESS_ABSTAIN;
        }
    }

    private boolean isConsumer(
            Collection<? extends GrantedAuthority> authorities) {
        return VoterHelper.hasRole(authorities, Roles.ROLE_CONSUMER);
    }

    private boolean checkIsViewingPermittedPatient(Authentication authentication, S s) {

    	PatientIdentifier targetPatientIdentifier = getTargetPatientIdentifier(s);

        if (NullChecker.isNullish(targetPatientIdentifier.getAssigningAuthority()) || NullChecker.isNullish(targetPatientIdentifier.getUniqueId())) {
        	return false;
        }
    	
    	if(isTargetIdentifierSameAsPatientIdentifier(authentication, targetPatientIdentifier)){
    		return true;
    	}else if (isTargetIdentifierBelongToCurrentPatient(authentication, targetPatientIdentifier)){
    		return true;
    	}

        return false;
    }

    private boolean isTargetIdentifierSameAsPatientIdentifier(Authentication authentication, PatientIdentifier targetPatientIdentifier) {
    	PatientIdentifier currentPatientIdentifier = VoterHelper.getUser(authentication).getPatient().getPatientIdentifier();
    	
    	if(currentPatientIdentifier.equals(targetPatientIdentifier)){
    		return true;
    	}else{
    		return false;
    	}
	}

	private boolean isTargetIdentifierBelongToCurrentPatient(Authentication authentication, PatientIdentifier targetPatientIdentifier) {
		PatientIdentifiers permittedPatientIdentifiers = null;
		
		try {
			permittedPatientIdentifiers = VoterHelper.getPermittedPatientIdentifiers(authentication);
		} catch (Exception e) {
			return false;
		}
		
		for (PatientIdentifier permittedPatientIdentifier : permittedPatientIdentifiers) {
			if (targetPatientIdentifier.equals(permittedPatientIdentifier)) {
				return true;
			}
		}
		
		return false;
	}

	private PatientIdentifier getTargetPatientIdentifier(S s) {
        // Get the path that we can extract target patient information
        String path = VoterHelper.getPath((FilterInvocation) s);
        return VoterHelper.getPatientIdentifierFromUri(path);
    }

}
