package com.agilex.healthcare.mobilehealthplatform.ical;

import java.net.SocketException;
import java.util.Date;

import net.fortuna.ical4j.model.Calendar;
import net.fortuna.ical4j.model.ComponentList;
import net.fortuna.ical4j.model.DateTime;
import net.fortuna.ical4j.model.Dur;
import net.fortuna.ical4j.model.PropertyList;
import net.fortuna.ical4j.model.component.VAlarm;
import net.fortuna.ical4j.model.component.VEvent;
import net.fortuna.ical4j.model.property.CalScale;
import net.fortuna.ical4j.model.property.Clazz;
import net.fortuna.ical4j.model.property.Description;
import net.fortuna.ical4j.model.property.Location;
import net.fortuna.ical4j.model.property.ProdId;
import net.fortuna.ical4j.model.property.Uid;
import net.fortuna.ical4j.model.property.Version;
import net.fortuna.ical4j.util.UidGenerator;

import com.agilex.healthcare.mobilehealthplatform.domain.Appointment;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointments;

public class ICalHelper {

	public static Calendar transformAppointmentsToIcal(Appointments appointments) {
		Calendar icsCalendar = new Calendar();
		setUpCalendar(icsCalendar);

		for (Appointment appointment : appointments) {
			VEvent appointmentEvent = convertAppointmentToVEvent(appointment);
			icsCalendar.getComponents().add(appointmentEvent);
		}

		return icsCalendar;
	}

	private static VEvent convertAppointmentToVEvent(Appointment appointment) {

		DateTime start = new DateTime(appointment.getAppointmentStartDate());
		DateTime end = new DateTime(calculateEndDate(appointment));
		String summary = appointment.getReasonForVisit();

		VEvent appointmentEvent = new VEvent(start, end, summary);
		addAlarmToVEvent(appointment, appointmentEvent);
		PropertyList properties = appointmentEvent.getProperties();
		properties.add(generateUid());
		properties.add(Clazz.PRIVATE);
		properties.add(new Location(appointment.getClinicName()));
		properties.add(new Description(appointment.getClinicalNotes()));

		return appointmentEvent;
	}

	private static void setUpCalendar(Calendar calendar) {
		calendar.getProperties().add(new ProdId("-//Events Calendar//iCal4j 1.0//EN"));
		calendar.getProperties().add(CalScale.GREGORIAN);
		calendar.getProperties().add(Version.VERSION_2_0);
	}

	private static Date calculateEndDate(Appointment appointment) {
		Date endDate = appointment.getAppointmentEndDate();
		if (endDate == null) {
			endDate = calculateEndDateByDuration(appointment);
		}
		return endDate;
	}

	private static void addAlarmToVEvent(Appointment appointment, VEvent event) {
		if (appointment.getAlarmOffset() != null) {
			int minuteOffset = appointment.getAlarmOffset();
			VAlarm alarm = new VAlarm(new Dur(0, 0, minuteOffset, 0));
			ComponentList componentList = event.getAlarms();
			componentList.add(alarm);
		}
	}

	private static Date calculateEndDateByDuration(Appointment appt) {
		java.util.Calendar cal = java.util.Calendar.getInstance();
		cal.setTime(appt.getAppointmentStartDate());

		Integer appointmentDuration = appt.getAppointmentDuration();
		if (appointmentDuration != null) {
			cal.add(java.util.Calendar.MINUTE, appt.getAppointmentDuration());
		}

		return cal.getTime();
	}

	private static Uid generateUid() {
		UidGenerator ug;
		try {
			ug = new UidGenerator("uidGen");
		} catch (SocketException e) {
			throw new RuntimeException("Could not generate UID via UidGenerator. ", e);
		}
		return ug.generateUid();
	}

}
