package com.agilex.healthcare.mobilehealthplatform.datalayer.vital;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.*;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.*;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.restservice.exception.InvalidVitalsException;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.*;
import com.agilex.healthcare.mobilehealthplatform.validator.VitalValidatorFactory;
import com.agilex.healthcare.mobilehealthplatform.validator.vitals.VitalValidator;

public class PatientEnteredVitalDataService {

	private Router router;
	
	public PatientEnteredVitalDataService() {
		this.router = new Router();
	}
	
	public PatientEnteredVitalDataService(Router router)  {
		this.router = router;
	}
	
	public VitalEntries fetchVitalEntries(PatientIdentifier patientIdentifier, VitalEntryFilter filter, ScopeFilter scopeFilter) {
		RequestMessage request = VitalEntryRequestBuilder.forPatientEntreredVitalEntries().forPatientIdentifier(patientIdentifier).forFilter(filter).forDomain(Domain.patientEnteredVitals).forScope(scopeFilter).build();
		ResponseMessage response = router.execute(request);
		VitalEntries vitalEntries = PatientDataResponseReader.<VitalEntries, VitalEntry>fromResponse(response).getDataList();
		
		return vitalEntries;
	}
	
	public VitalEntry fetchVitalEntry(PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier, ScopeFilter scopeFilter) {
		RequestMessage request = VitalEntryRequestBuilder.forPatientEnteredVitalEntry().forPatientIdentifier(patientIdentifier).forDataIdentifier(dataIdentifier).forDomain(Domain.patientEnteredVitals).forScope(scopeFilter).build();
		ResponseMessage response = router.execute(request);
		VitalEntry vitalEntry = PatientDataResponseReader.<VitalEntries, VitalEntry>fromResponse(response).getDataItem();
		
		return vitalEntry;
	}
	
	public VitalEntry save(VitalEntry vitalEntry) {
		validateVitals(vitalEntry);

        addSurrogate(vitalEntry);

		RequestMessage request = PatientDataEditRequestBuilder.forCreate().forDomain(Domain.patientEnteredVitals).forData(vitalEntry).forScope(ScopeFilter.getInstanceForOperationalScope()).build();
		ResponseMessage response = router.execute(request);
		VitalEntry saved = PatientDataResponseReader.<VitalEntries, VitalEntry>fromResponse(response).getDataItem();
		
		return saved;
	}
	
	public void delete(VitalEntry vitalEntry) {
        addSurrogate(vitalEntry);

		RequestMessage request = PatientDataEditRequestBuilder.forDelete().forDomain(Domain.patientEnteredVitals).forData(vitalEntry).forScope(ScopeFilter.getInstanceForOperationalScope()).build();
		router.execute(request);
	}

	private void validateVitals(VitalEntry vitalEntry) {
		VitalValidator validator = VitalValidatorFactory.getVitalsValdiator(vitalEntry);
		if (validator != null) {
			ValidationResult<VitalEntry> validationResult = validator.validate(vitalEntry);
			if (!validationResult.isValid()) {
				throw new InvalidVitalsException(validationResult.getErrors());
			}
		}
	}

    private void addSurrogate(VitalEntry data) {
        MhpUserFactory.addSurrogateToPatientData(data);
    }

}
